--[[
    LoggingUtil

    This file contains functions to print debug info, warnings and errors.

    DISCLAIMER: For this script to work properly with the printDev functions, the game needs to get started with the
                parameter "-devWarnings"! Otherwise the printDev functions will not print anything!

        @author:    [LSFM] BayernGamers
        @date:      12.11.2024
        @version:   2.0

        History:    v1.0 @30.11.2023 - initial implementation in FS 22
                    -------------------------------------------------------------------------------------------
                    v2.0 @12.11.2024 - convert to FS25

        License: 	This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

                    Terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage: 
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.

                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

                        Full License Text:
                            The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

LoggingUtil = {}

LoggingUtil.enableDebug = true
LoggingUtil.debugLevel = 1


-- Print debug info
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printDevInfo(message, debugLevel, addPrefix, luaName)
    if LoggingUtil.enableDebug and debugLevel >= LoggingUtil.debugLevel then
        local prefix = ""
        if addPrefix then
            prefix = "::" .. luaName .. ":: "
        end
        Logging.devInfo(prefix .. message)
    end
end

-- Print debug warning
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printDevWarning(message, debugLevel, addPrefix, luaName)
    if LoggingUtil.enableDebug and debugLevel >= LoggingUtil.debugLevel then
        local prefix = ""
        if addPrefix then
            prefix = "::" .. luaName .. ":: "
        end
        Logging.devWarning(prefix .. message)
    end
end

-- Print debug error
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printDevError(message, debugLevel, addPrefix, luaName)
    if LoggingUtil.enableDebug and debugLevel >= LoggingUtil.debugLevel then
        local prefix = ""
        if addPrefix then
            prefix = "::" .. luaName .. ":: "
        end
        Logging.devError(prefix .. message)
    end
end

-- Print info
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printInfo(message, addPrefix, luaName)
    local prefix = ""
    if addPrefix then
        prefix = "::" .. luaName .. ":: "
    end
    Logging.info(prefix .. message)
end

-- Print warning
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printWarning(message, addPrefix, luaName)
    local prefix = ""
    if addPrefix then
        prefix = "::" .. luaName .. ":: "
    end
    Logging.warning(prefix .. message)
end

-- Print error
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printError(message, addPrefix, luaName)
    local prefix = ""
    if addPrefix then
        prefix = "::" .. luaName .. ":: "
    end
    Logging.error(prefix .. message)
end

-- Print XML info
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printXMLInfo(xmlFile, message, addPrefix, luaName)
    local prefix = ""
    if addPrefix then
        prefix = "::" .. luaName .. ":: "
    end
    Logging.xmlInfo(xmlFile, prefix .. message)
end

-- Print XML warning
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printXMLWarning(xmlFile, message, addPrefix, luaName)
    local prefix = ""
    if addPrefix then
        prefix = "::" .. luaName .. ":: "
    end
    Logging.xmlWarning(xmlFile, prefix .. message)
end

-- Print XML error
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil.printXMLError(xmlFile, message, addPrefix, luaName)
    local prefix = ""
    if addPrefix then
        prefix = "::" .. luaName .. ":: "
    end
    Logging.xmlError(xmlFile, prefix .. message)
end
