--[[
    LockSteeringAxlesEvent

    Multiplayer event for locking/unlocking the steering axles of a vehicle.

	@author: 		[LSFM] BayernGamers
	@contributors: 	ifko[nator]
	@date: 			11.11.2024
	@version:		1.0
	
	History:	v1.0 @07.05.2017 - initial implementation in FS25
				------------------------------------------------------------------------------------------------

	License: 	This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

                    Terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage: 
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.

                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

                        Full License Text:
                            The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil

LockSteeringAxlesEvent = {}
LockSteeringAxlesEvent_mt = Class(LockSteeringAxlesEvent, Event)

InitEventClass(LockSteeringAxlesEvent, "LockSteeringAxlesEvent")

function LockSteeringAxlesEvent.emptyNew()
	local self = Event.new(LockSteeringAxlesEvent_mt)
    log.printDevInfo("Created new empty instance of 'LockSteeringAxlesEvent'", 1, true, "LockSteeringAxles.lua")
    
	return self
end

function LockSteeringAxlesEvent.new(trailer, lockSteeringAxle)
	local self = LockSteeringAxlesEvent.emptyNew()
	
	self.trailer = trailer
	self.steerAxleIsLocked = lockSteeringAxle
    log.printDevInfo("Created new instance of 'LockSteeringAxlesEvent' with trailer: " .. tostring(trailer) .. " and lockSteeringAxle: " .. tostring(lockSteeringAxle), 1, true, "LockSteeringAxles.lua")
	
	return self
end

function LockSteeringAxlesEvent:readStream(streamId, connection)
	self.trailer = NetworkUtil.readNodeObject(streamId)
	self.steerAxleIsLocked = streamReadBool(streamId)
    log.printDevInfo("Read stream of 'LockSteeringAxlesEvent' with trailer: " .. tostring(self.trailer) .. " and lockSteeringAxle: " .. tostring(self.steerAxleIsLocked), 1, true, "LockSteeringAxles.lua")
	
    self:run(connection)
end

function LockSteeringAxlesEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.trailer)
	
	streamWriteBool(streamId, self.steerAxleIsLocked)
    log.printDevInfo("Wrote stream of 'LockSteeringAxlesEvent' with trailer: " .. tostring(self.trailer) .. " and lockSteeringAxle: " .. tostring(self.steerAxleIsLocked), 1, true, "LockSteeringAxles.lua")
end

function LockSteeringAxlesEvent:run(connection)
	if not connection:getIsServer() then
        log.printDevInfo("Received 'LockSteeringAxlesEvent' on client side. Broadcasting event...", 1, true, "LockSteeringAxles.lua")
		g_server:broadcastEvent(LockSteeringAxlesEvent.new(self.trailer, self.steerAxleIsLocked), nil, connection, self.trailer)
	end
	
    if self.trailer ~= nil then
        log.printDevInfo("Locking/Unlocking steering axles of trailer: " .. tostring(self.trailer) .. " to: " .. tostring(self.steerAxleIsLocked), 1, true, "LockSteeringAxles.lua")
        self.trailer:setSteeringAxleActive(self.steerAxleIsLocked, true)
	end
end