--[[
AutoTurnOffTurnLights

Adds the ability to automatically turn off the turn lights when the steering wheel is turned back to the center position.

	@author: 		[LSFM] BayernGamers
	@contributors: 	ifko[nator]
	@date: 			06.11.2024
	@version:		3.0

	History:		v1.0 @02.01.2019 - initial implementation in FS 19
					---------------------------------------------------
					v2.0 @18.11.2021 - convert to FS 22
					---------------------------------------------------
					v2.5 @01.04.2022 - changed loading logic
					---------------------------------------------------
					v2.6 @21.04.2022 - fix for patch 1.4 and higher
					---------------------------------------------------
					v3.0 @06.11.2024 - convert to FS 25
									 - reworked loading logic

	
	License: 		This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

					Terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.

						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

						Full License Text:
							The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

AutoTurnOffTurnLights = {}

function AutoTurnOffTurnLights.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Drivable, specializations)
end

function AutoTurnOffTurnLights.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", AutoTurnOffTurnLights)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", AutoTurnOffTurnLights)
end

function AutoTurnOffTurnLights:onLoad(savegame)
	local specDrivable = self.spec_drivable
	
	specDrivable.hasSteerToTurnLeftDirection = false
	specDrivable.hasSteerToTurnRightDirection = false
end

function AutoTurnOffTurnLights:onUpdate(dt, isActiveForInput, isActiveForInputIngnoreSelection, isSelected)
	local specDrivable = self.spec_drivable

	if self:getIsActive() and specDrivable ~= nil and specDrivable.steeringWheel ~= nil then
		local _, steeringRotY, _ = getRotation(specDrivable.steeringWheel.node)
		
		if specDrivable.hasSteerToTurnLeftDirection or specDrivable.hasSteerToTurnRightDirection then
			local allowAutoTurnOffTurnLights = false
			
			if specDrivable.hasSteerToTurnLeftDirection then
				allowAutoTurnOffTurnLights = steeringRotY <= 0.001
			elseif specDrivable.hasSteerToTurnRightDirection then
				allowAutoTurnOffTurnLights = steeringRotY >= -0.001
			end
			
			if allowAutoTurnOffTurnLights then
				self:setTurnLightState(Lights.TURNLIGHT_OFF)
				
				specDrivable.hasSteerToTurnLeftDirection = false
				specDrivable.hasSteerToTurnRightDirection = false
			end
		elseif self.spec_lights.turnLightState == Lights.TURNLIGHT_LEFT then
			specDrivable.hasSteerToTurnLeftDirection = steeringRotY > 0.1
		elseif self.spec_lights.turnLightState == Lights.TURNLIGHT_RIGHT then
			specDrivable.hasSteerToTurnRightDirection = steeringRotY < -0.1
		end
	end
end