-- QuestBoardHUD.lua
-- Окно мини-квестов для плейсаблов с спецой questBoard
-- Архитектура как у IngredientCookerHUD: register / reloadHUD / INSTANCE.

QuestBoardHUD = {}
local QuestBoardHUD_mt = Class(QuestBoardHUD, MessageDialog)

QuestBoardHUD.modDirectory      = g_currentModDirectory or ""
QuestBoardHUD.consoleRegistered = false
QuestBoardHUD.INSTANCE          = nil

local LOG = "[QuestBoardHUD] "

-- ===== helpers =====
local function _l10n(s)
    if not s or s == "" then return "" end
    if s:sub(1,1) ~= "$" then return s end
    local k = s:sub(2)
    if g_i18n and g_i18n:hasText(k) then return g_i18n:getText(k) end
    if k:sub(1,5) == "l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then
        return g_i18n:getText(k:sub(6))
    end
    return s
end

local function _questItemName(id, name)
    -- если явно задано имя, просто локализуем его
    if name and name ~= "" then
        return _l10n(name)
    end

    if not id or id == "" then
        return ""
    end

    -- если это уже "$ключ"
    if id:sub(1, 1) == "$" then
        return _l10n(id)
    end

    -- пробуем шаблон l10n_ + id с заменой точек на подчёркивания
    --   food.beef -> l10n_food_beef
    local key = "l10n_" .. id:gsub("%.", "_")
    local raw = "$" .. key
    local txt = _l10n(raw)
    if txt ~= raw then
        return txt
    end

    -- иначе показываем сырой id
    return id
end

-- клиентский подсчёт количества предметов
local function _getItemCountClient(id)
    if id == nil or id == "" then
        return 0
    end

    -- 1) Наш InventoryHUD
    if Inventory and Inventory.API and Inventory.API.getTotalCount then
        local ok, res = pcall(Inventory.API.getTotalCount, id)
        if ok and res then
            return res or 0
        end
    end

    -- 2) Старый g_inventory
    if g_inventory and g_inventory.getTotalCount then
        local farmId = 1
        if g_currentMission and g_currentMission.player and g_currentMission.player.farmId then
            farmId = g_currentMission.player.farmId
        end

        local ok, res = pcall(g_inventory.getTotalCount, g_inventory, id, farmId)
        if ok and res then
            return res or 0
        end
    end

    return 0
end

local function _getLiveFarmerBaseDir_QB()
    local candidates = {}

    if PlaceableQuestBoard and PlaceableQuestBoard.modDirectory and PlaceableQuestBoard.modDirectory ~= "" then
        table.insert(candidates, PlaceableQuestBoard.modDirectory)
    end
    if QuestBoardHUD.modDirectory and QuestBoardHUD.modDirectory ~= "" then
        table.insert(candidates, QuestBoardHUD.modDirectory)
    end

    if g_modManager ~= nil then
        if g_modManager.getModByName ~= nil then
            local m = g_modManager:getModByName("FS25_liveFarmer")
            if m ~= nil then
                local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                if dir and dir ~= "" then
                    table.insert(candidates, dir)
                end
            end
        end
        if g_modManager.mods ~= nil then
            for _, m in pairs(g_modManager.mods) do
                local name     = m.modName or m.name or ""
                local fileName = m.modFileName or ""
                if name == "FS25_liveFarmer" or fileName:lower():find("fs25_livefarmer") ~= nil then
                    local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                    if dir and dir ~= "" then
                        table.insert(candidates, dir)
                    end
                end
            end
        end
    end

    if debug and debug.getinfo then
        local info = debug.getinfo(_getLiveFarmerBaseDir_QB, "S")
        if info and info.source then
            local src = info.source
            if src:sub(1,1) == "@" then src = src:sub(2) end
            src = src:gsub("\\", "/")
            local idx = src:lower():find("/fs25_livefarmer/")
            if idx then
                local dir = src:sub(1, idx + string.len("/FS25_liveFarmer/") - 1)
                table.insert(candidates, dir)
            end
        end
    end

    if g_modsDirectory ~= nil then
        table.insert(candidates, g_modsDirectory .. "/FS25_liveFarmer/")
    end

    local tried = {}
    for _, baseDir in ipairs(candidates) do
        if type(baseDir) == "string" and baseDir ~= "" then
            baseDir = baseDir:gsub("\\", "/")
            if baseDir:sub(-1) ~= "/" then
                baseDir = baseDir .. "/"
            end
            if not tried[baseDir] then
                tried[baseDir] = true
                local testPath = Utils.getFilename("gui/QuestBoardHUD.xml", baseDir)
                if testPath and fileExists(testPath) then
                    return baseDir
                end
            end
        end
    end

    return QuestBoardHUD.modDirectory or (g_currentModDirectory or "")
end


-- ===== console commands (как у IngredientCookerHUD) =====
function QuestBoardHUD.registerConsole()
    if QuestBoardHUD.consoleRegistered then return end
    if addConsoleCommand ~= nil then
        addConsoleCommand(
            "gsReloadQuestBoardHUD",
            "Reload QuestBoard HUD xml (close/reopen if open)",
            "reloadHUD",
            QuestBoardHUD
        )
        addConsoleCommand(
            "gsReloadQBH",
            "Reload QuestBoard HUD xml (alias)",
            "reloadHUD",
            QuestBoardHUD
        )
        QuestBoardHUD.consoleRegistered = true
        print(LOG .. "Console command registered: gsReloadQuestBoardHUD")
    end
end

function QuestBoardHUD.reloadHUD()
    if not g_gui then
        print(LOG .. "g_gui is not ready")
        return
    end

    local baseDir = _getLiveFarmerBaseDir_QB and _getLiveFarmerBaseDir_QB() or (QuestBoardHUD.modDirectory or g_currentModDirectory or "")
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/QuestBoardHUD.xml", baseDir)
    if not (xmlPath and fileExists(xmlPath)) then
        print(LOG .. "HUD XML not found to reload (baseDir=" .. tostring(baseDir) .. ")")
        return
    end

    ------------------------------------------------------------
    -- 1) Запоминаем состояние и закрываем, если открыт
    ------------------------------------------------------------
    local keepPlaceable, wasOpen = nil, false
    if QuestBoardHUD.INSTANCE then
        keepPlaceable = QuestBoardHUD.INSTANCE.placeable
    end

    if g_gui.currentGuiName == "QuestBoardHUD" then
        wasOpen = true
        pcall(function() g_gui:showGui("") end)
    end

    ------------------------------------------------------------
    -- 2) Полностью выгружаем старый GUI
    ------------------------------------------------------------
    if g_gui.guis and g_gui.guis["QuestBoardHUD"] then
        pcall(function() g_gui:unloadGui("QuestBoardHUD") end)
        g_gui.guis["QuestBoardHUD"] = nil
    end

    if QuestBoardHUD.INSTANCE then
        pcall(function() QuestBoardHUD.INSTANCE:delete() end)
        QuestBoardHUD.INSTANCE = nil
    end

    ------------------------------------------------------------
    -- 3) Регистрируем заново
    ------------------------------------------------------------
    local ok = QuestBoardHUD.register()
    print(LOG .. "Reload result: " .. tostring(ok))

    ------------------------------------------------------------
    -- 4) Если HUD был открыт — открываем снова
    ------------------------------------------------------------
    if ok and wasOpen and keepPlaceable and QuestBoardHUD.INSTANCE then
        QuestBoardHUD.INSTANCE.placeable    = keepPlaceable
        QuestBoardHUD.INSTANCE.currentQuest = nil
        QuestBoardHUD.INSTANCE:updateView()
        g_gui:showGui("QuestBoardHUD")
    end
end

-- ===== register GUI =====
function QuestBoardHUD.register()
    if not g_gui then return false end

    QuestBoardHUD.registerConsole()

    if g_gui.guis and g_gui.guis["QuestBoardHUD"] then
        QuestBoardHUD.INSTANCE = g_gui.guis["QuestBoardHUD"].target
        return true
    end

    local inst = QuestBoardHUD.new()

    local baseDir = _getLiveFarmerBaseDir_QB()
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/QuestBoardHUD.xml", baseDir)
    local ok = false

    if xmlPath and xmlPath ~= "" then
        ok = pcall(function()
            g_gui:loadGui(xmlPath, "QuestBoardHUD", inst)
        end)
    end

    if (not ok) or not (g_gui.guis and g_gui.guis["QuestBoardHUD"]) then
        local path1 = baseDir .. "gui/QuestBoardHUD.xml"
        local path2 = baseDir .. "QuestBoardHUD.xml"
        for _, p in ipairs({path1, path2}) do
            if p and p ~= "" then
                local loaded = pcall(function()
                    g_gui:loadGui(p, "QuestBoardHUD", inst)
                end)
                if loaded and g_gui.guis and g_gui.guis["QuestBoardHUD"] then
                    ok = true
                    xmlPath = p
                    break
                end
            end
        end
    end

    if ok and g_gui.guis and g_gui.guis["QuestBoardHUD"] then
        QuestBoardHUD.INSTANCE = g_gui.guis["QuestBoardHUD"].target
        return true
    else
        Logging.error("[QuestBoardHUD] XML not found (baseDir=%s, xmlPath=%s)", tostring(baseDir), tostring(xmlPath))
        pcall(function() inst:delete() end)
        return false
    end
end

-- ===== ctor =====
function QuestBoardHUD.new(target, custom_mt)
    local self = MessageDialog.new(target, custom_mt or QuestBoardHUD_mt)

    self.placeable    = nil
    self.currentQuest = nil

    self.dialogTitleElement = nil
    self.questTitle         = nil
    self.questDesc          = nil
    self.questReqLabel      = nil
    self.questReqText       = nil
    self.questRewardLabel   = nil
    self.questRewardText    = nil
    self.questStatusText    = nil

    self.acceptButton       = nil
    self.completeButton     = nil
    self.closeButton        = nil

    return self
end

function QuestBoardHUD:delete()
    self.placeable    = nil
    self.currentQuest = nil
end

-- ===== lifecycle =====
function QuestBoardHUD:onOpen()
    local sc = self.superClass and self:superClass()
    if sc and sc.onOpen then sc.onOpen(self) end

    if g_inputBinding and g_inputBinding.setShowMouseCursor then
        g_inputBinding:setShowMouseCursor(true)
    end

    -- привязка элементов по id
    self.dialogTitleElement = self:getDescendantById("dialogTitleElement")
    self.questTitle         = self:getDescendantById("questTitle")
    self.questDesc          = self:getDescendantById("questDesc")
    self.questReqLabel      = self:getDescendantById("questReqLabel")
    self.questReqText       = self:getDescendantById("questReqText")
    self.questRewardLabel   = self:getDescendantById("questRewardLabel")
    self.questRewardText    = self:getDescendantById("questRewardText")
    self.questStatusText    = self:getDescendantById("questStatusText")

    self.acceptButton       = self:getDescendantById("acceptButton")
    self.completeButton     = self:getDescendantById("completeButton")
    self.closeButton        = self:getDescendantById("closeButton")

    -- Обновляем заголовок из title в XML плейсабла
    if self.dialogTitleElement and self.placeable and self.placeable.spec_questBoard then
        local titleKey = self.placeable.spec_questBoard.title or "$ui_quest_board_title"
        self.dialogTitleElement:setText(_l10n(titleKey))
    end

    -- Запрашиваем у плейсабла текущее состояние/новый квест
    if self.placeable and QB_PlayerActionEvent and QB_Events then
        print(LOG .. "onOpen: requesting quest (ACTION_REQUEST_NEW)")
        QB_PlayerActionEvent.sendToServer(self.placeable, QB_Events.ACTION_REQUEST_NEW)
    end

    self:updateView()
end

function QuestBoardHUD:onClose()
    if g_inputBinding and g_inputBinding.setShowMouseCursor then
        g_inputBinding:setShowMouseCursor(false)
    end

    local sc = self.superClass and self:superClass()
    if sc and sc.onClose then sc.onClose(self) end

    self.placeable    = nil
    self.currentQuest = nil
end

-- ===== обновление UI =====
function QuestBoardHUD:updateView()
    -- нет плейсабла — очищаем
    if not self.placeable then
        if self.questTitle      then self.questTitle:setText(_l10n("$l10n_ui_questBoard_noQuest")) end
        if self.questDesc       then self.questDesc:setText("") end
        if self.questReqText    then
            self.questReqText:setText("")
            self.questReqText:setTextColor(1, 1, 1, 1)
        end
        if self.questRewardText then self.questRewardText:setText("") end
        if self.questStatusText then
            self.questStatusText:setText("")
            self.questStatusText:setTextColor(1, 1, 1, 1)
        end
        return
    end

    local spec = self.placeable.spec_questBoard
    if not spec then return end

    local q = self.placeable.getCurrentQuest and self.placeable:getCurrentQuest() or nil
    self.currentQuest = q

    if not q then
        if self.questTitle      then self.questTitle:setText(_l10n("$l10n_ui_questBoard_noQuest")) end
        if self.questDesc       then self.questDesc:setText("") end
        if self.questReqText    then
            self.questReqText:setText("")
            self.questReqText:setTextColor(1, 1, 1, 1)
        end
        if self.questRewardText then self.questRewardText:setText("") end
        if self.questStatusText then
            self.questStatusText:setText("")
            self.questStatusText:setTextColor(1, 1, 1, 1)
        end
        return
    end

    local isActive = (spec.activeQuestId ~= nil and spec.activeQuestId == q.id)

    -- Заголовок и описание
    if self.questTitle then
        self.questTitle:setText(_l10n(q.name))
    end
    if self.questDesc then
        self.questDesc:setText(_l10n(q.desc or ""))
    end

    ----------------------------------------------------------------------
    -- Требования
    ----------------------------------------------------------------------
    local reqLines = {}
    local allOk    = true
    local hasReqs  = false

    for _, req in ipairs(q.requirements or {}) do
        hasReqs = true
        local need = req.count or 0
        local name = _questItemName(req.id, req.name)
        local line

        if isActive then
            local have = _getItemCountClient(req.id)
            if have < need then
                allOk = false
            end
            line = string.format("• %s: %d / %d", name, have, need)
        else
            line = string.format("• %s: %d", name, need)
        end

        table.insert(reqLines, line)
    end

    if self.questReqText then
        self.questReqText:setText(table.concat(reqLines, "\n"))

        -- цвет требований:
        --  - активен и не хватает хоть чего-то → оранжевый
        --  - активен и всего хватает → зелёный
        --  - не активен → дефолт (белый)
        if isActive and hasReqs then
            if allOk then
                -- всё собрано
                self.questReqText:setTextColor(0.60, 1.00, 0.40, 1.00) -- зелёный
            else
                -- ещё не всё
                self.questReqText:setTextColor(1.00, 0.80, 0.30, 1.00) -- оранжевый
            end
        else
            self.questReqText:setTextColor(1.00, 1.00, 1.00, 1.00)
        end
    end

    ----------------------------------------------------------------------
    -- Награды (только деньги + предметы)
    ----------------------------------------------------------------------
    local rewardLines = {}

    -- Деньги
    if q.moneyReward and q.moneyReward > 0 then
        table.insert(rewardLines, string.format("• %d $", q.moneyReward))
    end

    -- Предметы-награды
    if q.rewardItems then
        for _, ri in ipairs(q.rewardItems) do
            table.insert(rewardLines, string.format("• %dx %s",
                ri.count or 0,
                _questItemName(ri.id, ri.name)
            ))
        end
    end

    if self.questRewardText then
        self.questRewardText:setText(table.concat(rewardLines, "\n"))
    end

    ----------------------------------------------------------------------
    -- Статус
    ----------------------------------------------------------------------
    local statusKey = self.placeable.getQuestStatusText and self.placeable:getQuestStatusText(q) or ""
    if self.questStatusText then
        self.questStatusText:setText(_l10n(statusKey or ""))

        -- цвета статуса:
        --   активен        → зелёный
        --   не принят      → красный
        --   другой активен → жёлтый
        local r, g, b, a = 1, 1, 1, 1
        if statusKey == "$l10n_ui_questBoard_status_active" then
            r, g, b = 0.60, 1.00, 0.40
        elseif statusKey == "$l10n_ui_questBoard_status_notAccepted" then
            r, g, b = 1.00, 0.30, 0.30
        else
            -- "other" или что-то ещё
            r, g, b = 1.00, 0.80, 0.30
        end
        self.questStatusText:setTextColor(r, g, b, a)
    end
end

-- ===== кнопки =====
function QuestBoardHUD:onClickAccept()
    print(LOG .. "onClickAccept")
    if not (self.placeable and self.currentQuest) then
        print(LOG .. "onClickAccept: no placeable or no currentQuest")
        return
    end
    if self.placeable.acceptCurrentQuest then
        print(LOG .. "onClickAccept: calling placeable:acceptCurrentQuest()")
        self.placeable:acceptCurrentQuest()
    else
        print(LOG .. "onClickAccept: placeable.acceptCurrentQuest is nil")
    end
    -- дальше состояние придёт с сервера через QB_SetPlayerQuestStateEvent
end

function QuestBoardHUD:onClickComplete()
    print(LOG .. "onClickComplete")
    if not (self.placeable and self.currentQuest) then
        print(LOG .. "onClickComplete: no placeable or no currentQuest")
        return
    end
    if self.placeable.tryCompleteCurrentQuest then
        print(LOG .. "onClickComplete: calling placeable:tryCompleteCurrentQuest()")
        self.placeable:tryCompleteCurrentQuest()
    else
        print(LOG .. "onClickComplete: placeable.tryCompleteCurrentQuest is nil")
    end
    -- тоже, обновление придёт с сервера
end

function QuestBoardHUD:onClickClose()
    g_gui:showGui("")
end
