-- QB_Events.lua
-- Сетевые события для доски квестов (PlaceableQuestBoard)
-- Делает так, что у каждого игрока свои квесты (по farmId).

QB_Events = {}

local LOG = "[QuestBoard-Net] "

-- Типы действий игрока
QB_Events.ACTION_REQUEST_SYNC  = 0  -- запросить актуальное состояние
QB_Events.ACTION_ACCEPT_QUEST  = 1  -- принять текущий квест
QB_Events.ACTION_TRY_COMPLETE  = 2  -- попытка сдать квест
QB_Events.ACTION_REQUEST_NEW   = 3  -- запросить новый рандомный квест (если нет)

-----------------------------------------------------------------------
-- Event: действие игрока (client -> server)
-----------------------------------------------------------------------

QB_PlayerActionEvent = {}
local QB_PlayerActionEvent_mt = Class(QB_PlayerActionEvent, Event)
InitEventClass(QB_PlayerActionEvent, "QB_PlayerActionEvent")

function QB_PlayerActionEvent.emptyNew()
    return Event.new(QB_PlayerActionEvent_mt)
end

--- object  : placeable (QuestBoard)
--- action  : см. QB_Events.ACTION_*
function QB_PlayerActionEvent.new(object, action)
    local self = QB_PlayerActionEvent.emptyNew()
    self.object = object
    self.action = action
    return self
end

function QB_PlayerActionEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.object)
    streamWriteUIntN(streamId, self.action, 3)
end

function QB_PlayerActionEvent:readStream(streamId, connection)
    self.object = NetworkUtil.readNodeObject(streamId)
    self.action = streamReadUIntN(streamId, 3)
    self:run(connection)
end

-- На сервере
function QB_PlayerActionEvent:run(connection)
    if not (self.object and self.object.spec_questBoard) then
        return
    end

    if g_server == nil then
        return
    end

    -- connection здесь — это соединение клиента, от которого пришёл запрос
    self.object:qb_handlePlayerAction(self.action, connection)
end

--- Вызов с клиента
function QB_PlayerActionEvent.sendToServer(object, action)
    -- Здесь мы всегда на КЛИЕНТЕ (GUI). Даже в Singleplayer клиент и сервер разделены.
    if g_client ~= nil then
        g_client:getServerConnection():sendEvent(QB_PlayerActionEvent.new(object, action))
    else
        -- На всякий случай fallback: если вдруг вызывается на сервере напрямую.
        if g_server ~= nil then
            object:qb_handlePlayerAction(action, nil)
        end
    end
end

-----------------------------------------------------------------------
-- Event: синхронизация состояния квеста игрока (server -> client)
-----------------------------------------------------------------------

QB_SetPlayerQuestStateEvent = {}
local QB_SetPlayerQuestStateEvent_mt = Class(QB_SetPlayerQuestStateEvent, Event)
InitEventClass(QB_SetPlayerQuestStateEvent, "QB_SetPlayerQuestStateEvent")

function QB_SetPlayerQuestStateEvent.emptyNew()
    return Event.new(QB_SetPlayerQuestStateEvent_mt)
end

--- object       : placeable (QuestBoard)
--- currentIndex : индекс квеста (0 если нет)
--- activeId     : string id активного квеста или ""
function QB_SetPlayerQuestStateEvent.new(object, currentIndex, activeId)
    local self = QB_SetPlayerQuestStateEvent.emptyNew()
    self.object       = object
    self.currentIndex = currentIndex or 0
    self.activeId     = activeId or ""
    return self
end

function QB_SetPlayerQuestStateEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.object)
    streamWriteUIntN(streamId, self.currentIndex or 0, 8)
    streamWriteString(streamId, self.activeId or "")
end

function QB_SetPlayerQuestStateEvent:readStream(streamId, connection)
    self.object       = NetworkUtil.readNodeObject(streamId)
    self.currentIndex = streamReadUIntN(streamId, 8)
    self.activeId     = streamReadString(streamId)
    self:run(connection)
end

-- На КЛИЕНТЕ (сюда прилетает от сервера)
function QB_SetPlayerQuestStateEvent:run(connection)
    if not (self.object and self.object.spec_questBoard) then
        return
    end

    local spec = self.object.spec_questBoard
    spec.currentIndex  = self.currentIndex or 0
    if self.activeId ~= nil and self.activeId ~= "" then
        spec.activeQuestId = self.activeId
    else
        spec.activeQuestId = nil
    end

    -- Если HUD открыт для этого плейсабла — обновляем
    if QuestBoardHUD and QuestBoardHUD.INSTANCE and QuestBoardHUD.INSTANCE.placeable == self.object then
        QuestBoardHUD.INSTANCE:updateView()
    end
end

--- Отправка СЕРВЕРОМ клиенту (или всем клиентам)
--  object       : серверный экземпляр плейсабла
--  connection   : соединение конкретного клиента; если nil — broadcast всем
function QB_SetPlayerQuestStateEvent.sendToClient(object, connection, currentIndex, activeId)
    if g_server == nil then return end

    local event = QB_SetPlayerQuestStateEvent.new(object, currentIndex, activeId)

    if connection ~= nil then
        -- отправляем конкретному клиенту (обычный случай: ответ тому, кто открыл GUI)
        connection:sendEvent(event)
    else
        -- если connection == nil, шлём всем клиентам (в SP сюда тоже прилетит локальный клиент)
        g_server:broadcastEvent(event, false, nil, object)
    end
end
