-- PlaceableQuestBoard.lua
-- Спеца плейсабла для мини-квестов (доска квестов)
-- Один набор квестов и одно состояние на КАЖДЫЙ плейсабл.
-- Состояние сохраняется в savegame.

PlaceableQuestBoard = {}
PlaceableQuestBoard.modDirectory = g_currentModDirectory or ""
PlaceableQuestBoard.modName      = g_currentModName or "hungresystem"
PlaceableQuestBoard.SAVE_KEY     = "questBoard"

local LOG = "[QuestBoard] "

-------------------------------------------------------------------------------
-- Спеца registration
-------------------------------------------------------------------------------

function PlaceableQuestBoard.prerequisitesPresent(specializations)
    return true
end

function PlaceableQuestBoard.registerXMLPaths(schema, basePath)
    -- <questBoard triggerNode="" title="">
    schema:register(XMLValueType.STRING,     basePath .. ".questBoard#title",       "Quest board title")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".questBoard#triggerNode", "Quest board trigger")

    -- <quests><quest . /></quests>
    local questKey = basePath .. ".questBoard.quests.quest(?)"
    schema:register(XMLValueType.STRING, questKey .. "#id",          "Quest id")
    schema:register(XMLValueType.STRING, questKey .. "#name",        "Quest name (l10n or text)")
    schema:register(XMLValueType.STRING, questKey .. "#desc",        "Quest description")
    schema:register(XMLValueType.INT,    questKey .. "#moneyReward", "Money reward")
    schema:register(XMLValueType.INT,    questKey .. "#vigorBonus",  "Vigor bonus in percent")

    -- <require id="" count="" name="">
    local reqKey = questKey .. ".require(?)"
    schema:register(XMLValueType.STRING, reqKey .. "#id",    "Required item id")
    schema:register(XMLValueType.INT,    reqKey .. "#count", "Required item count")
    schema:register(XMLValueType.STRING, reqKey .. "#name",  "Required item name (l10n or text, optional)")

    -- <rewardItem id="" count="" name="">
    local rewKey = questKey .. ".rewardItem(?)"
    schema:register(XMLValueType.STRING, rewKey .. "#id",    "Reward item id")
    schema:register(XMLValueType.INT,    rewKey .. "#count", "Reward item count")
    schema:register(XMLValueType.STRING, rewKey .. "#name",  "Reward item name (l10n or text, optional)")
end

function PlaceableQuestBoard.registerFunctions(placeableType)
    SpecializationUtil.registerFunction(placeableType, "qb_onTriggerCallback", PlaceableQuestBoard.qb_onTriggerCallback)

    -- API для HUD
    SpecializationUtil.registerFunction(placeableType, "getCurrentQuest",    PlaceableQuestBoard.getCurrentQuest)
    SpecializationUtil.registerFunction(placeableType, "getQuestStatusText", PlaceableQuestBoard.getQuestStatusText)

    -- Серверная логика (один квест на плейсабл)
    SpecializationUtil.registerFunction(placeableType, "qb_handlePlayerAction", PlaceableQuestBoard.qb_handlePlayerAction)
    SpecializationUtil.registerFunction(placeableType, "qb_serverTryComplete",  PlaceableQuestBoard.qb_serverTryComplete)

    -- локальный метод открытия GUI (как picOpenGUI)
    SpecializationUtil.registerFunction(placeableType, "qbOpenGUI", PlaceableQuestBoard.qbOpenGUI)

    -- кнопки HUD
    SpecializationUtil.registerFunction(placeableType, "acceptCurrentQuest",      PlaceableQuestBoard.acceptCurrentQuest)
    SpecializationUtil.registerFunction(placeableType, "tryCompleteCurrentQuest", PlaceableQuestBoard.tryCompleteCurrentQuest)
end

function PlaceableQuestBoard.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",              PlaceableQuestBoard)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",            PlaceableQuestBoard)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream",        PlaceableQuestBoard)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream",       PlaceableQuestBoard)
    SpecializationUtil.registerEventListener(placeableType, "onReadUpdateStream",  PlaceableQuestBoard)
    SpecializationUtil.registerEventListener(placeableType, "onWriteUpdateStream", PlaceableQuestBoard)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement", PlaceableQuestBoard)
end

-------------------------------------------------------------------------------
-- Activatable (как у IngredientCooker)
-------------------------------------------------------------------------------

QB_Activatable = {}
local QB_Activatable_mt = Class(QB_Activatable)

function QB_Activatable.new(placeable)
    local self = setmetatable({}, QB_Activatable_mt)
    self.placeable    = placeable
    self.activateText =
        (g_i18n and g_i18n:getText("l10n_ui_questBoard_open"))
        or "Открыть квесты (R)"
    return self
end

function QB_Activatable:getIsActivatable()
    if g_localPlayer == nil then return false end
    if g_localPlayer.getIsInVehicle and g_localPlayer:getIsInVehicle() then
        return false
    end
    return self.placeable ~= nil
end

function QB_Activatable:run()
    if self.placeable ~= nil and self.placeable.qbOpenGUI ~= nil then
        self.placeable:qbOpenGUI()
    end
end

-------------------------------------------------------------------------------
-- onLoad / net
-------------------------------------------------------------------------------

function PlaceableQuestBoard:onLoad(savegame)
    self.spec_questBoard = self.spec_questBoard or {}
    local spec = self.spec_questBoard

    local xmlFile = self.xmlFile

    -- корневой ключ: если xmlNodeName пустой, используем "placeable"
    local rootKey  = self.xmlNodeName or "placeable"
    local questKey = rootKey .. ".questBoard"

    if not xmlFile:hasProperty(questKey) then
        Logging.warning("%sNo <questBoard> block in %s", LOG, tostring(self.configFileName))
        return
    end

    spec.title = xmlFile:getString(questKey .. "#title", "$ui_quest_board_title")

    -- триггер
    local triggerNode = xmlFile:getValue(questKey .. "#triggerNode", nil, self.components, self.i3dMappings)
    if triggerNode == nil then
        Logging.warning("%sNo triggerNode for questBoard in %s", LOG, tostring(self.configFileName))
        return
    end

    spec.triggerNode = triggerNode
    addTrigger(triggerNode, "qb_onTriggerCallback", self)

    spec.playerInTrigger = false

    -- activatable для R / ACTIVATE_OBJECT (как у PIC)
    if not self.qbActivatable then
        self.qbActivatable = QB_Activatable.new(self)
        self.qbActivatable.activateText =
            (g_i18n and g_i18n:getText("l10n_ui_questBoard_open")) or "Открыть квесты (R)"
    end

    -- заранее загрузим HUD, как делает PlaceableIngredientCooker
    if g_gui and (not (g_gui.guis and g_gui.guis["QuestBoardHUD"])) and QuestBoardHUD and QuestBoardHUD.register then
        QuestBoardHUD.register()
    end

    -------------------------------------------------------------------------
    -- Загрузка списка квестов из config XML
    -------------------------------------------------------------------------
    spec.quests = {}

    local i = 0
    while true do
        local baseKey = string.format("%s.quests.quest(%d)", questKey, i)
        local questId = xmlFile:getString(baseKey .. "#id")
        if questId == nil then
            break
        end

        local qName       = xmlFile:getString(baseKey .. "#name", "")
        local qDesc       = xmlFile:getString(baseKey .. "#desc", "")
        local moneyReward = xmlFile:getInt(baseKey .. "#moneyReward", 0)
        local vigorBonus  = xmlFile:getInt(baseKey .. "#vigorBonus", 0)

        local q = {
            id           = questId,
            name         = qName,
            desc         = qDesc,
            moneyReward  = moneyReward,
            vigorBonus   = vigorBonus,
            requirements = {},
            rewardItems  = {}
        }

        -- требования: <require id="" count="" name="">
        local j = 0
        while true do
            local reqKey = string.format("%s.require(%d)", baseKey, j)
            local rId = xmlFile:getString(reqKey .. "#id")
            if rId == nil then break end

            local rCount = xmlFile:getInt(reqKey .. "#count", 1)
            local rName  = xmlFile:getString(reqKey .. "#name", "")

            table.insert(q.requirements, {
                id    = rId,
                count = rCount,
                name  = rName
            })

            j = j + 1
        end

        -- награды: <rewardItem ... >
        local k = 0
        while true do
            local rewKey = string.format("%s.rewardItem(%d)", baseKey, k)
            local rid = xmlFile:getString(rewKey .. "#id")
            if rid == nil then break end

            local rc      = xmlFile:getInt   (rewKey .. "#count",      1)
            local rName   = xmlFile:getString(rewKey .. "#name",       "")
            local rDesc   = xmlFile:getString(rewKey .. "#desc",       "")
            local rIcon   = xmlFile:getString(rewKey .. "#icon",       "")
            local rKind   = xmlFile:getString(rewKey .. "#kind",       "")
            local rHungry = xmlFile:getFloat (rewKey .. "#hungerGain", 0)
            local rVigor  = xmlFile:getFloat (rewKey .. "#vigorGain",  0)
            local rEffId  = xmlFile:getString(rewKey .. "#effectId",   "")
            local rEffFile= xmlFile:getString(rewKey .. "#effectsFile","")

            table.insert(q.rewardItems, {
                id         = rid,
                count      = rc,
                name       = rName,
                desc       = rDesc,
                icon       = rIcon,
                kind       = rKind,
                hungerGain = rHungry,
                vigorGain  = rVigor,
                effectId   = rEffId,
                effectsFile= rEffFile
            })
			
			-- ДОБАВЬ СРАЗУ ПОСЛЕ table.insert:
			local ri = q.rewardItems[#q.rewardItems]
			if ri.icon ~= "" and not fileExists(ri.icon) then
				ri.icon = PlaceableQuestBoard.modDirectory .. ri.icon
			end

            k = k + 1
        end


        table.insert(spec.quests, q)
        i = i + 1
    end

    -------------------------------------------------------------------------
    -- Инициализация состояния по умолчанию (ОДИН квест на плейсабл)
    -------------------------------------------------------------------------
    spec.currentIndex  = 0   -- индекс квеста из spec.quests (0 = нет)
    spec.activeQuestId = nil -- id активного квеста или nil

    -------------------------------------------------------------------------
    -- Загрузка состояния из savegame (ТОЛЬКО НА СЕРВЕРЕ)
    -------------------------------------------------------------------------
    if savegame ~= nil and g_server ~= nil then
        local sXml  = savegame.xmlFile
        local sKey  = string.format("%s.%s", savegame.key, PlaceableQuestBoard.SAVE_KEY)

        if sXml:hasProperty(sKey) then
            local savedIndex = sXml:getInt(sKey .. "#currentIndex", 0)
            local savedId    = sXml:getString(sKey .. "#activeQuestId", "")

            if #spec.quests > 0 then
                -- клэмп индекса
                if savedIndex < 0 or savedIndex > #spec.quests then
                    savedIndex = 0
                end
            else
                savedIndex = 0
            end

            spec.currentIndex  = savedIndex
            spec.activeQuestId = (savedId ~= nil and savedId ~= "") and savedId or nil

            print(string.format("%sLoaded state for placeable %s: index=%d, activeId=%s",
                LOG, tostring(self.configFileName), spec.currentIndex or 0, tostring(spec.activeQuestId or "nil")))
        end
    end
end

function PlaceableQuestBoard:onFinalizePlacement()
    -- Ничего не делаем: первый квест выдаётся по запросу игрока.
end

function PlaceableQuestBoard:onDelete()
    local spec = self.spec_questBoard
    if spec and spec.triggerNode then
        removeTrigger(spec.triggerNode)
        spec.triggerNode = nil
        spec.playerInTrigger = false
    end

    if self.qbActivatable and g_currentMission and g_currentMission.activatableObjectsSystem then
        g_currentMission.activatableObjectsSystem:removeActivatable(self.qbActivatable)
    end
    self.qbActivatable = nil
end

-- Базовая сетка: основное состояние держим в кастомных эвентах, поэтому тут ничего
function PlaceableQuestBoard:onWriteStream(streamId, connection) end
function PlaceableQuestBoard:onReadStream(streamId, connection) end
function PlaceableQuestBoard:onWriteUpdateStream(streamId, connection, dirtyMask) end
function PlaceableQuestBoard:onReadUpdateStream(streamId, timestamp, connection) end

-- Сохранение состояния в savegame (ТОЛЬКО НА СЕРВЕРЕ)
function PlaceableQuestBoard:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self.spec_questBoard
    if not spec or g_server == nil then
        return
    end

    local saveKey = string.format("%s.%s", key, PlaceableQuestBoard.SAVE_KEY)

    xmlFile:setInt(saveKey .. "#currentIndex", spec.currentIndex or 0)
    xmlFile:setString(saveKey .. "#activeQuestId", spec.activeQuestId or "")
end

-------------------------------------------------------------------------------
-- Внутренняя логика квеста (один на плейсабл)
-------------------------------------------------------------------------------

-- Выбрать рандомный квест
function PlaceableQuestBoard.qb_pickRandomQuest(self)
    local spec = self.spec_questBoard
    if not spec then return end
    if #spec.quests == 0 then return end

    local idx = 1 + math.floor(math.random() * #spec.quests)
    spec.currentIndex = math.max(1, math.min(idx, #spec.quests))

    print(string.format("%sPicked random quest index=%d for placeable %s",
        LOG, spec.currentIndex, tostring(self.configFileName)))
end

-- Вернуть текущий квест
function PlaceableQuestBoard.qb_getQuest(self)
    local spec = self.spec_questBoard
    if not spec or #spec.quests == 0 then return nil end

    local idx = spec.currentIndex or 0
    if idx < 1 or idx > #spec.quests then
        return nil
    end

    return spec.quests[idx]
end

-------------------------------------------------------------------------------
-- Trigger / activatableObjectsSystem (как у IngredientCooker)
-------------------------------------------------------------------------------

function PlaceableQuestBoard:qb_onTriggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
    if not (onEnter or onLeave) then return end
    if g_localPlayer == nil then return end
    if otherActorId ~= g_localPlayer.rootNode then return end

    local spec = self.spec_questBoard
    if not spec then return end

    if onEnter then
        spec.playerInTrigger = true

        if Platform.gameplay
            and Platform.gameplay.autoActivateTrigger
            and self.qbActivatable
            and self.qbActivatable:getIsActivatable()
        then
            self.qbActivatable:run()
        else
            if g_currentMission and g_currentMission.activatableObjectsSystem and self.qbActivatable then
                g_currentMission.activatableObjectsSystem:addActivatable(self.qbActivatable)
            end
        end
    elseif onLeave then
        spec.playerInTrigger = false
        if g_currentMission and g_currentMission.activatableObjectsSystem and self.qbActivatable then
            g_currentMission.activatableObjectsSystem:removeActivatable(self.qbActivatable)
        end
    end
end

-------------------------------------------------------------------------------
-- Открытие GUI (как picOpenGUI)
-------------------------------------------------------------------------------

function PlaceableQuestBoard:qbOpenGUI()
    if not g_gui then return end

    -- 1) Регистрируем HUD, если ещё не
    if QuestBoardHUD and QuestBoardHUD.register then
        QuestBoardHUD.register()
    end

    -- 2) Запрашиваем у сервера квест
    if QB_PlayerActionEvent and QB_Events then
        print(LOG .. "Requesting quest state (ACTION_REQUEST_NEW)")
        QB_PlayerActionEvent.sendToServer(self, QB_Events.ACTION_REQUEST_NEW)
    end

    -- 3) Находим guiDef
    local guiDef = g_gui.guis and g_gui.guis["QuestBoardHUD"]
    if not guiDef then
        print(LOG .. "QuestBoardHUD guiDef not found in g_gui.guis")
        return
    end

    -- 4) Настраиваем инстанс HUD
    local inst = guiDef.target
    if inst then
        inst.placeable    = self
        inst.currentQuest = nil

        if inst.updateView then
            inst:updateView()
        end

        -- 5) Показываем диалог
        g_gui:showGui("QuestBoardHUD")
    end
end

-------------------------------------------------------------------------------
-- API для HUD (client side)
-------------------------------------------------------------------------------

-- Текущий квест (для клиента)
function PlaceableQuestBoard:getCurrentQuest()
    local spec = self.spec_questBoard
    if not spec or #spec.quests == 0 then return nil end
    local idx = spec.currentIndex or 0
    if idx < 1 or idx > #spec.quests then
        return nil
    end
    return spec.quests[idx]
end

function PlaceableQuestBoard:getQuestStatusText(q)
    local spec = self.spec_questBoard
    if not spec then return "" end
    if spec.activeQuestId == nil or spec.activeQuestId == "" then
        return "$l10n_ui_questBoard_status_notAccepted"
    end
    if spec.activeQuestId == q.id then
        return "$l10n_ui_questBoard_status_active"
    end
    return "$l10n_ui_questBoard_status_other"
end

-------------------------------------------------------------------------------
-- Обработка действий игрока (сервер)
-------------------------------------------------------------------------------

local function _getFarmIdFromConnection(self, connection)
    if connection ~= nil and connection.getFarmId ~= nil then
        local fid = connection:getFarmId()
        if fid ~= nil and fid ~= 0 then
            return fid
        end
    end

    if g_currentMission ~= nil and g_currentMission.player ~= nil then
        if g_currentMission.player.farmId ~= nil then
            return g_currentMission.player.farmId
        end
    end

    if self.getOwnerFarmId ~= nil then
        return self:getOwnerFarmId()
    end

    return 1
end

function PlaceableQuestBoard:qb_handlePlayerAction(action, connection)
    if g_server == nil then return end
    local spec = self.spec_questBoard
    if not spec then return end

    print(string.format("%sqb_handlePlayerAction action=%s placeable=%s",
        LOG, tostring(action), tostring(self.configFileName)))

    if action == QB_Events.ACTION_REQUEST_SYNC then
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")

    elseif action == QB_Events.ACTION_REQUEST_NEW then
        if (spec.currentIndex or 0) == 0 then
            PlaceableQuestBoard.qb_pickRandomQuest(self)
        end
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")

    elseif action == QB_Events.ACTION_ACCEPT_QUEST then
        if (spec.currentIndex or 0) == 0 then
            PlaceableQuestBoard.qb_pickRandomQuest(self)
        end
        local q = PlaceableQuestBoard.qb_getQuest(self)
        if q then
            spec.activeQuestId = q.id
            print(string.format("%sQuest accepted: id=%s (placeable %s)",
                LOG, tostring(q.id), tostring(self.configFileName)))
        end
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")

    elseif action == QB_Events.ACTION_TRY_COMPLETE then
        self:qb_serverTryComplete(connection)
    end
end

function PlaceableQuestBoard.qb_serverTryComplete(self, connection)
    local spec = self.spec_questBoard
    if not spec then return end

    local q = PlaceableQuestBoard.qb_getQuest(self)
    if not q then
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
        return
    end

    -- Квест должен быть именно активным
    if spec.activeQuestId ~= q.id then
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
        return
    end

    local farmId = _getFarmIdFromConnection(self, connection)
    print(string.format("%sTry complete quest id=%s for farmId=%d", LOG, tostring(q.id), farmId))

    ----------------------------------------------------------------------
    -- Проверяем наличие предметов
    ----------------------------------------------------------------------
    for _, req in ipairs(q.requirements or {}) do
        local have = _getItemCount(req.id, farmId)
        local need = req.count or 1

        if have < need then
            print(string.format("%sNot enough items for %s: have=%d need=%d",
                LOG, tostring(req.id), have, need))
            QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
            return
        end
    end

    ----------------------------------------------------------------------
    -- Списываем предметы
    ----------------------------------------------------------------------
    for _, req in ipairs(q.requirements or {}) do
        _tryRemoveItems(req.id, req.count or 1, farmId)
    end

    ----------------------------------------------------------------------
    -- Награда: деньги + предметы (БЕЗ прямого бонуса бодрости)
    ----------------------------------------------------------------------

    -- Деньги
    if q.moneyReward and q.moneyReward > 0 then
        if g_currentMission ~= nil and g_currentMission.addMoney ~= nil then
            g_currentMission:addMoney(q.moneyReward, farmId, MoneyType.OTHER, true, true)
        end
    end

    -- Предметы-награды: пробрасываем всю структуру ri
    for _, ri in ipairs(q.rewardItems or {}) do
        _giveItems(ri.id, ri.count or 1, farmId, ri)
    end

    print(string.format("%sQuest completed: id=%s, rewards (money+items) given", LOG, tostring(q.id)))

    ----------------------------------------------------------------------
    -- Сбрасываем квест и выдаём новый
    ----------------------------------------------------------------------
    spec.activeQuestId = nil
    spec.currentIndex  = 0
    PlaceableQuestBoard.qb_pickRandomQuest(self)

    QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
end


-------------------------------------------------------------------------------
-- Интеграция с Inventory / награда (СЕРВЕР)
-------------------------------------------------------------------------------

local function _getItemCount(id, farmId)
    -- 1) Наш HUD-инвентарь
    if Inventory and Inventory.API and Inventory.API.getTotalCount then
        return Inventory.API.getTotalCount(id) or 0
    end

    -- 2) Старый g_inventory (если где-то есть)
    if g_inventory and g_inventory.getTotalCount then
        return g_inventory:getTotalCount(id, farmId) or 0
    end

    return 0
end

local function _tryRemoveItems(id, count, farmId)
    -- 1) Наш HUD-инвентарь
    if Inventory and Inventory.API and Inventory.API.removeItems then
        return Inventory.API.removeItems(id, count or 1)
    end

    -- 2) Старый g_inventory
    if g_inventory and g_inventory.removeItems then
        return g_inventory:removeItems(id, count or 1, farmId)
    end

    return false
end

local function _giveItems(id, count, farmId, data)
    count = count or 1

    -- 1) Наш InventoryHUD
    if Inventory and Inventory.API and Inventory.API.addItemsById then
        local name, desc, icon, kind, hungerGain, vigorGain, effectId, effectsFile

        if type(data) == "table" then
            name        = data.name
            desc        = data.desc
            icon        = data.icon
            kind        = data.kind
            hungerGain  = data.hungerGain
            vigorGain   = data.vigorGain
            effectId    = data.effectId
            effectsFile = data.effectsFile
        end

        -- Если каких-то полей нет в XML – они будут nil
        -- Inventory.API.addItemsById сам подставит дефолты ("" или 0)
        return Inventory.API.addItemsById(
            id,
            count,
            name,
            desc,
            icon,
            kind,
            hungerGain,
            vigorGain,
            effectId,
            effectsFile
        )
    end

    -- 2) Старый g_inventory – без доп. данных
    if g_inventory and g_inventory.addItems then
        return g_inventory:addItems(id, count, farmId)
    end

    return false
end



function PlaceableQuestBoard:qb_serverTryComplete(connection)
    local spec = self.spec_questBoard
    if not spec then return end

    local q = PlaceableQuestBoard.qb_getQuest(self)
    if not q then
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
        return
    end

    -- Квест должен быть именно активным, а не просто текущим
    if spec.activeQuestId ~= q.id then
        QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
        return
    end

    local farmId = _getFarmIdFromConnection(self, connection)

    print(string.format("%sTry complete quest id=%s for farmId=%d", LOG, tostring(q.id), farmId))

    ----------------------------------------------------------------------
    -- ПРОВЕРКА НАЛИЧИЯ ПРЕДМЕТОВ
    ----------------------------------------------------------------------
    for _, req in ipairs(q.requirements or {}) do
        local have = _getItemCount(req.id, farmId)
        local need = req.count or 1

        if have < need then
            print(string.format("%sNot enough items for %s: have=%d need=%d",
                LOG, tostring(req.id), have, need))
            QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
            return
        end
    end

    ----------------------------------------------------------------------
    -- СПИСЫВАЕМ ПРЕДМЕТЫ
    ----------------------------------------------------------------------
    for _, req in ipairs(q.requirements or {}) do
        _tryRemoveItems(req.id, req.count or 1, farmId)
    end

    ----------------------------------------------------------------------
    -- НАГРАДА: ДЕНЬГИ + ПРЕДМЕТЫ (без прямого бонуса бодрости)
    ----------------------------------------------------------------------

    -- Деньги
    if q.moneyReward and q.moneyReward > 0 then
        if g_currentMission ~= nil and g_currentMission.addMoney ~= nil then
            g_currentMission:addMoney(q.moneyReward, farmId, MoneyType.OTHER, true, true)
        end
    end

    -- Предметы-награды: передаём всю структуру ri в _giveItems
    for _, ri in ipairs(q.rewardItems or {}) do
        _giveItems(ri.id, ri.count or 1, farmId, ri)
    end

    print(string.format("%sQuest completed: id=%s, rewards (money+items) given", LOG, tostring(q.id)))

    ----------------------------------------------------------------------
    -- СБРАСЫВАЕМ КВЕСТ И ВЫБИРАЕМ СЛЕДУЮЩИЙ
    ----------------------------------------------------------------------
    spec.activeQuestId = nil
    spec.currentIndex  = 0
    PlaceableQuestBoard.qb_pickRandomQuest(self)

    QB_SetPlayerQuestStateEvent.sendToClient(self, connection, spec.currentIndex or 0, spec.activeQuestId or "")
end

-------------------------------------------------------------------------------
-- Публичные методы, которые дергает HUD (client side)
-------------------------------------------------------------------------------

function PlaceableQuestBoard:acceptCurrentQuest()
    print(LOG .. "Client: acceptCurrentQuest clicked, sending event")
    QB_PlayerActionEvent.sendToServer(self, QB_Events.ACTION_ACCEPT_QUEST)
end

function PlaceableQuestBoard:tryCompleteCurrentQuest()
    print(LOG .. "Client: tryCompleteCurrentQuest clicked, sending event")
    QB_PlayerActionEvent.sendToServer(self, QB_Events.ACTION_TRY_COMPLETE)
end
