-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

PlaceableCooker = {}
PlaceableCooker.modDirectory = g_currentModDirectory or ""
PlaceableCooker.modName      = g_currentModName or "PlaceableCooker"

local _evPath = Utils.getFilename("scripts/placeableCooker/PCC_Events.lua", PlaceableCooker.modDirectory)
if fileExists(_evPath) then
    source(_evPath)
else
    print("[PlaceableCooker] PCC_Events.lua not found at "..tostring(_evPath))
end

--------------------------------------------------------------------------------
-- XML schema + savegame schema
--------------------------------------------------------------------------------
function PlaceableCooker.prerequisitesPresent(_)
    return true
end

function PlaceableCooker.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.STRING,     basePath .. ".cooker#title",       "GUI title (l10n or text)")
    schema:register(XMLValueType.STRING,     basePath .. ".cooker#effectsFile", "ItemEffects xml (optional)")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".cooker#triggerNode", "Trigger node for activatable")

    local item = basePath .. ".cooker.items.item(?)"
    schema:register(XMLValueType.STRING,     item .. "#name",       "Name (l10n or text)")
    schema:register(XMLValueType.FLOAT,      item .. "#hungerGain", "Hunger %+")
    schema:register(XMLValueType.FLOAT,      item .. "#vigorGain",  "Vigor %+")
    schema:register(XMLValueType.INT,        item .. "#cookSec",    "Cook seconds")
    schema:register(XMLValueType.FLOAT,      item .. "#price",      "Price to cook (can be 0)", 0)
    schema:register(XMLValueType.INT,        item .. "#readyCap",   "Ready storage cap for this recipe (0 = infinite)", 0)
    schema:register(XMLValueType.STRING,     item .. "#icon",       "Icon path (dds/png)")
    schema:register(XMLValueType.STRING,     item .. "#desc",       "Description (l10n or text)")
    schema:register(XMLValueType.STRING,     item .. "#kind",       "Type: food/effect", "food")
    schema:register(XMLValueType.STRING,     item .. "#effectId",   "Effect id (ItemEffects)")
    schema:register(XMLValueType.NODE_INDEX, item .. "#animNode",   "Animation skeleton/node for cooking clip")
    schema:register(XMLValueType.STRING,     item .. "#animClip",   "Animation clip name")
    schema:register(XMLValueType.BOOL,       item .. "#animLoop",   "Loop animation while cooking", true)
    schema:register(XMLValueType.FLOAT,      item .. "#animSpeed",  "Animation speed multiplier", 1.0)
    schema:register(XMLValueType.STRING,     item .. "#visNodes",   "Visibility nodes list (separated by ';' or ',')")
    schema:register(XMLValueType.NODE_INDEX, item .. ".visNode(?)#node", "Visibility node (repeatable)")
end

function PlaceableCooker.initSpecialization()
    local saveSchema = Placeable.xmlSchemaSavegame
    local ns = PlaceableCooker.modName .. ".placeableCooker"
    saveSchema:register(XMLValueType.BOOL,  "placeables.placeable(?)."..ns.."#cooking",      "Is cooking", false)
    saveSchema:register(XMLValueType.INT,   "placeables.placeable(?)."..ns.."#currentIndex", "Current recipe index (1-based, 0=none)", 0)
    saveSchema:register(XMLValueType.INT,   "placeables.placeable(?)."..ns.."#remainingMs",  "Remaining ms for active cooking", 0)
    local base = "placeables.placeable(?)."..ns..".ready"
    saveSchema:register(XMLValueType.INT,   base.."#count", "Ready items count", 0)
    saveSchema:register(XMLValueType.INT,   base..".item(?)#index",      "Recipe index (1-based)")
    saveSchema:register(XMLValueType.STRING,base..".item(?)#name",       "Name")
    saveSchema:register(XMLValueType.STRING,base..".item(?)#icon",       "Icon")
    saveSchema:register(XMLValueType.STRING,base..".item(?)#kind",       "Kind food/effect")
    saveSchema:register(XMLValueType.STRING,base..".item(?)#effectId",   "Effect id")
    saveSchema:register(XMLValueType.STRING,base..".item(?)#desc",       "Desc")
    saveSchema:register(XMLValueType.INT,   base..".item(?)#hungerGain", "Hunger %+")
    saveSchema:register(XMLValueType.INT,   base..".item(?)#vigorGain",  "Vigor %+")
end

function PlaceableCooker.registerFunctions(placeableType)
    SpecializationUtil.registerFunction(placeableType, "pcTriggerCallback",          PlaceableCooker.pcTriggerCallback)
    SpecializationUtil.registerFunction(placeableType, "pcGetIsAccessible",          PlaceableCooker.pcGetIsAccessible)
    SpecializationUtil.registerFunction(placeableType, "pcServerStartCook",          PlaceableCooker.pcServerStartCook)
    SpecializationUtil.registerFunction(placeableType, "pcServerFinalizeIfReady",    PlaceableCooker.pcServerFinalizeIfReady)
    SpecializationUtil.registerFunction(placeableType, "pcServerConsumeCooked",      PlaceableCooker.pcServerConsumeCooked)
    SpecializationUtil.registerFunction(placeableType, "pcServerTakeToInventory",    PlaceableCooker.pcServerTakeToInventory)
    SpecializationUtil.registerFunction(placeableType, "pcServerDiscardCooked",      PlaceableCooker.pcServerDiscardCooked)
end

function PlaceableCooker.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",         PlaceableCooker)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",       PlaceableCooker)
    SpecializationUtil.registerEventListener(placeableType, "onUpdate",       PlaceableCooker)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream",   PlaceableCooker)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream",  PlaceableCooker)
    SpecializationUtil.registerEventListener(placeableType, "saveToXMLFile",  PlaceableCooker) -- ← сохранение
end

--------------------------------------------------------------------------------
-- Help
--------------------------------------------------------------------------------
local function _l10n(raw)
    if not raw or raw=="" then return "" end
    if raw:sub(1,1) ~= "$" then return raw end
    local k = raw:sub(2)
    if g_i18n and g_i18n.hasText and g_i18n:hasText(k) then return g_i18n:getText(k) end
    if k:sub(1,5)=="l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then return g_i18n:getText(k:sub(6)) end
    return k
end

local function _nowMs() return g_time or 0 end
local function _dayIndex()
    local env = g_currentMission and g_currentMission.environment
    return (env and env.currentDay) or 0
end
local function _copyTable(src)
    if type(src) ~= "table" then return src end
    local dst = {}
    for k,v in pairs(src) do dst[k] = (type(v)=="table") and _copyTable(v) or v end
    return dst
end

local function _safeIcon(path)
    if not path or path=="" then return nil end
    if fileExists(path) then return path end
    local alt=nil
    if path:lower():sub(-4)==".dds" then alt = path:sub(1,-5)..".png"
    elseif path:lower():sub(-4)==".png" then alt = path:sub(1,-5)..".dds" end
    if alt and fileExists(alt) then return alt end
    return nil
end

local function _topNotify(title, msg, icon)
    local hud = g_currentMission and g_currentMission.hud
    if hud and hud.topNotification and hud.topNotification.setNotification then
        hud.topNotification:setNotification(title or "", msg or "", "", _safeIcon(icon), nil)
    else
        print(("[PlaceableCooker] %s | %s"):format(tostring(title or ""), tostring(msg or "")))
    end
end

-- ==========================
local function _pcVisParseList(str)
    local out = {}
    if not str or str == "" then return out end
    for token in string.gmatch(str, "[^;,]+") do
        local s = token:gsub("^%s+", ""):gsub("%s+$", "")
        if s ~= "" then table.insert(out, s) end
    end
    return out
end

local function _pcVisResolveForItem(self, it)
    if it._visResolved ~= nil then return it._visResolved end
    it._visResolved = false
    it._visNodes = {}

    if it.visNodes and type(it.visNodes) == "string" then
        for _, idx in ipairs(_pcVisParseList(it.visNodes)) do
            local node = I3DUtil.indexToObject(self.components, idx, self.i3dMappings)
            if node ~= nil and node ~= 0 then
                table.insert(it._visNodes, { node = node, wasVisible = getVisibility(node) })
            end
        end
    end

    if it._visNodesExtra then
        for _, node in ipairs(it._visNodesExtra) do
            if node ~= 0 and node ~= nil then
                table.insert(it._visNodes, { node = node, wasVisible = getVisibility(node) })
            end
        end
    end

    it._visResolved = true
    return true
end

local function _pcVisSet(self, it, visible)
    if not it or not _pcVisResolveForItem(self, it) then return end
    for _, entry in ipairs(it._visNodes or {}) do
        if entry.node ~= nil and entry.node ~= 0 then
            setVisibility(entry.node, visible and true or false)
        end
    end
end

local function _pcVisRestoreOriginal(self, it)
    if not it or not it._visNodes then return end
    for _, entry in ipairs(it._visNodes) do
        if entry.node ~= nil and entry.node ~= 0 then
            setVisibility(entry.node, entry.wasVisible and true or false)
        end
    end
end

-- ==========================
-- animation
-- ==========================
local function _pcAnimResolveForItem(self, it)
    if it._animResolved ~= nil then return it._animResolved end

    it._animResolved = false
    it._anim = nil

    local node = it.animNode
    local clip = it.animClip
    if node == nil or node == 0 or clip == nil or clip == "" then
        return false
    end

    local charSet = getAnimCharacterSet(node)
    if charSet == nil then
        Logging.warning("[PlaceableCooker] anim: characterSet missing for item '%s'", tostring(it.name))
        return false
    end
    local clipIndex = getAnimClipIndex(charSet, clip)
    if clipIndex == nil or clipIndex == -1 then
        Logging.warning("[PlaceableCooker] anim: clip '%s' not found for item '%s'", tostring(clip), tostring(it.name))
        return false
    end

    local track = 0
    assignAnimTrackClip(charSet, track, clipIndex)
    setAnimTrackLoopState(charSet, track, it.animLoop == true)
    setAnimTrackSpeedScale(charSet, track, 0)
    setAnimTrackTime(charSet, track, 0, true)
    disableAnimTrack(charSet, track)

    it._anim = {
        node = node,
        charSet = charSet,
        track = track,
        clipIndex = clipIndex,
        duration = getAnimClipDuration(charSet, clipIndex) or 0
    }
    it._animResolved = true
    return true
end

local function _pcAnimStop(self, reason)
    local spec = self.spec_placeableCooker
    if not spec then return end

    if spec._animItem and spec._animItem._anim then
        local anim = spec._animItem._anim
        setAnimTrackSpeedScale(anim.charSet, anim.track, 0)
        setAnimTrackTime(anim.charSet, anim.track, 0, true)
        disableAnimTrack(anim.charSet, anim.track)
    end
	
    _pcVisSet(self, spec._animItem, false)

    spec._animActive = false
    spec._animItem = nil
    spec._animOneShotEndTime = nil
end


local function _pcAnimStartForItem(self, it)
    local spec = self.spec_placeableCooker
    if not spec then return end

    _pcVisSet(self, it, true)

    if not _pcAnimResolveForItem(self, it) then
        spec._animActive = false
        spec._animItem = it
        spec._animOneShotEndTime = nil
        return
    end

    local anim = it._anim
    setAnimTrackLoopState(anim.charSet, anim.track, it.animLoop == true)
    setAnimTrackTime(anim.charSet, anim.track, 0, true)
    setAnimTrackSpeedScale(anim.charSet, anim.track, math.max(0.0001, tonumber(it.animSpeed or 1.0) or 1.0))
    enableAnimTrack(anim.charSet, anim.track)

    spec._animActive = true
    spec._animItem = it
    spec._animOneShotEndTime = nil

    if it.animLoop ~= true and anim.duration and anim.duration > 0 then
        spec._animOneShotEndTime = (g_time or 0) + math.floor(anim.duration * 1000)
    end
end

-- Клиент/сервер
function PlaceableCooker:_pcApplyCookAnimFromState()
    local spec = self.spec_placeableCooker
    if not spec then return end

    if spec.state.cooking and spec.state.current and self.getIsSynchronized and self:getIsSynchronized() then
        local it = spec.state.current
        if it.index and self.spec_placeableCooker.items[it.index] then
            local src = self.spec_placeableCooker.items[it.index]
            it.animNode   = src.animNode
            it.animClip   = src.animClip
            it.animLoop   = src.animLoop
            it.animSpeed  = src.animSpeed
            it.visNodes   = src.visNodes
            it._visNodesExtra = src._visNodesExtra
        end
        _pcVisSet(self, it, true)
        _pcAnimStartForItem(self, it)
    else
        _pcAnimStop(self, "state")
    end
end

--------------------------------------------------------------------------------
-- Load/Save
--------------------------------------------------------------------------------
function PlaceableCooker:onLoad(savegame)
    self.spec_placeableCooker = self.spec_placeableCooker or {}
    local spec = self.spec_placeableCooker

    local xml = self.xmlFile
    spec.titleRaw = xml:getValue("placeable.cooker#title")
    spec.title    = _l10n(spec.titleRaw) ~= "" and _l10n(spec.titleRaw) or ((g_i18n and g_i18n:getText("ui_pcooker_title")) or "Кухня")
    local effectsRel = xml:getValue("placeable.cooker#effectsFile")
    spec.effectsFile = (effectsRel and effectsRel~="") and Utils.getFilename(effectsRel, PlaceableCooker.modDirectory) or nil

    spec.triggerNode = xml:getValue("placeable.cooker#triggerNode", nil, self.components, self.i3dMappings)

    spec.items = {}
    do
        local i=0
        while true do
            local key = ("placeable.cooker.items.item(%d)"):format(i)
            if not xml:hasProperty(key) then break end
            local name       = xml:getValue(key.."#name") or ("Dish "..tostring(i+1))
            local hungerGain = xml:getValue(key.."#hungerGain", 0)
            local vigorGain  = xml:getValue(key.."#vigorGain",  0)
            local cookSec    = xml:getValue(key.."#cookSec",    30)
            local price      = xml:getValue(key.."#price",      0)
            local readyCap   = xml:getValue(key.."#readyCap",   0)
            local iconRel    = xml:getValue(key.."#icon")
            local desc       = xml:getValue(key.."#desc")
            local kind       = (xml:getValue(key.."#kind") or "food"):lower()
            local effectId   = xml:getValue(key.."#effectId")
            local animNode = xml:getValue(key.."#animNode", nil, self.components, self.i3dMappings)
            local animClip = xml:getValue(key.."#animClip")
            local animLoop = xml:getValue(key.."#animLoop", true)
            local animSpeed= xml:getValue(key.."#animSpeed", 1.0)
            local visNodesStr = xml:getValue(key.."#visNodes")
            local visNodesExtra = {}
            xml:iterate(key .. ".visNode", function(_, nkey)
                local n = xml:getValue(nkey.."#node", nil, self.components, self.i3dMappings)
                if n ~= nil and n ~= 0 then table.insert(visNodesExtra, n) end
            end)

            if effectId and effectId~="" then kind = "effect" end
            local icon = iconRel and Utils.getFilename(iconRel, PlaceableCooker.modDirectory) or nil
            if icon and not fileExists(icon) then icon = nil end

            local item = {
                index      = #spec.items+1,
                name       = name,
                hungerGain = math.floor(hungerGain or 0),
                vigorGain  = math.floor(vigorGain  or 0),
                cookSec    = math.max(1, math.floor(cookSec or 1)),
                price      = math.max(0, math.floor(price or 0)),
                readyCap   = math.max(0, math.floor(readyCap or 0)),
                icon       = icon,
                desc       = desc,
                kind       = kind,
                effectId   = effectId,
                animNode   = animNode,
                animClip   = animClip,
                animLoop   = animLoop == true,
                animSpeed  = tonumber(animSpeed) or 1.0,
                _animResolved = nil,
                _anim = nil,
                visNodes       = visNodesStr,
                _visNodesExtra = visNodesExtra,
                _visResolved   = nil,
                _visNodes      = nil
            }

            table.insert(spec.items, item)
            i=i+1
        end
    end

    spec.state = {
        cooking    = false,
        ready      = false,
        spoiled    = false,
        current    = nil,
        endTimeMs  = 0,
        cookedDay  = -1
    }
    spec.readyItems  = {}
    spec.readyCount  = 0
    spec.readyCounts = {}

    spec._animActive = false
    spec._animItem   = nil
    spec._animOneShotEndTime = nil

    if spec.triggerNode ~= nil then
        addTrigger(spec.triggerNode, "pcTriggerCallback", self)
    else
        Logging.xmlWarning(self.xmlFile, "[PlaceableCooker] Missing trigger node (placeable.cooker#triggerNode)")
    end

    if not self.pcActivatable then
        self.pcActivatable = PC_Activatable.new(self)
    end

    if savegame ~= nil and savegame.xmlFile ~= nil then
        local ns = PlaceableCooker.modName .. ".placeableCooker"
        local keyBase = savegame.key .. "." .. ns

        local wasCooking  = savegame.xmlFile:getValue(keyBase.."#cooking", false)
        local currentIdx  = savegame.xmlFile:getValue(keyBase.."#currentIndex", 0)
        local remainingMs = math.max(0, savegame.xmlFile:getValue(keyBase.."#remainingMs", 0))

        local readyKey  = keyBase..".ready"
        local readyCnt  = math.max(0, savegame.xmlFile:getValue(readyKey.."#count", 0))
        for i=0, readyCnt-1 do
            local ikey = string.format("%s.item(%d)", readyKey, i)
            local item = {
                index      = savegame.xmlFile:getValue(ikey.."#index", 0),
                name       = savegame.xmlFile:getValue(ikey.."#name",  "Food"),
                icon       = savegame.xmlFile:getValue(ikey.."#icon",  ""),
                kind       = (savegame.xmlFile:getValue(ikey.."#kind", "food") or "food"),
                effectId   = savegame.xmlFile:getValue(ikey.."#effectId", ""),
                desc       = savegame.xmlFile:getValue(ikey.."#desc", ""),
                hungerGain = math.floor(savegame.xmlFile:getValue(ikey.."#hungerGain", 0) or 0),
                vigorGain  = math.floor(savegame.xmlFile:getValue(ikey.."#vigorGain",  0) or 0),
            }
            table.insert(spec.readyItems, item)
            if item.index and item.index > 0 then
                spec.readyCounts[item.index] = (spec.readyCounts[item.index] or 0) + 1
            end
        end
        spec.readyCount = #spec.readyItems
        spec.state.ready = spec.readyCount > 0

        if wasCooking and currentIdx > 0 and remainingMs > 0 and spec.items[currentIdx] then
            spec.state.cooking   = true
            spec.state.current   = _copyTable(spec.items[currentIdx]); spec.state.current.index = currentIdx
            spec.state.endTimeMs = _nowMs() + remainingMs
            spec.state.cookedDay = -1
            self:raiseActive()
            _pcVisSet(self, spec.state.current, true)
            _pcAnimStartForItem(self, spec.state.current)
        end
    end

    if g_gui and (not (g_gui.guis and g_gui.guis["PlaceableCookerHUD"])) then
        if PlaceableCookerHUD and PlaceableCookerHUD.register then PlaceableCookerHUD.register() end
    end
end

function PlaceableCooker:saveToXMLFile(xmlFile, key)
    local spec = self.spec_placeableCooker
    if not spec then return end
    local base = key
    local cooking = spec.state.cooking == true
    local currentIndex = (spec.state.current and spec.state.current.index) or 0
    local remainingMs = 0
    if cooking then
        remainingMs = math.max(0, (spec.state.endTimeMs or 0) - (g_time or 0))
    end
    xmlFile:setValue(base.."#cooking",      cooking)
    xmlFile:setValue(base.."#currentIndex", currentIndex)
    xmlFile:setValue(base.."#remainingMs",  remainingMs)
    local rBase = base..".ready"
    local cnt = #spec.readyItems
    xmlFile:setValue(rBase.."#count", cnt)
    for i=1,cnt do
        local it = spec.readyItems[i]
        local iKey = string.format("%s.item(%d)", rBase, i-1)
        xmlFile:setValue(iKey.."#index",      it.index or 0)
        xmlFile:setValue(iKey.."#name",       tostring(it.name or "Food"))
        xmlFile:setValue(iKey.."#icon",       tostring(it.icon or ""))
        xmlFile:setValue(iKey.."#kind",       tostring(it.kind or "food"))
        xmlFile:setValue(iKey.."#effectId",   tostring(it.effectId or ""))
        xmlFile:setValue(iKey.."#desc",       tostring(it.desc or ""))
        xmlFile:setValue(iKey.."#hungerGain", math.floor(tonumber(it.hungerGain) or 0))
        xmlFile:setValue(iKey.."#vigorGain",  math.floor(tonumber(it.vigorGain)  or 0))
    end
end

function PlaceableCooker:onDelete()
    local spec = self.spec_placeableCooker
    if spec and spec.triggerNode then removeTrigger(spec.triggerNode); spec.triggerNode=nil end
    if self.pcActivatable then
        g_currentMission.activatableObjectsSystem:removeActivatable(self.pcActivatable)
        self.pcActivatable = nil
    end
    _pcAnimStop(self, "delete")
end

--------------------------------------------------------------------------------
-- Stream (для MP)
--------------------------------------------------------------------------------
function PlaceableCooker:onWriteStream(streamId, connection)
    local spec = self.spec_placeableCooker
    if not spec then return end

    local st = spec.state or {}

    streamWriteBool(streamId, st.cooking == true)
    streamWriteBool(streamId, st.ready   == true)
    streamWriteBool(streamId, st.spoiled == true)
    streamWriteInt32(streamId, st.endTimeMs or 0)
    streamWriteInt16(streamId, st.cookedDay or -1)

    local curIndex = (st.current and st.current.index) or -1
    streamWriteInt16(streamId, curIndex)

    local readyCount = spec.readyCount or (#spec.readyItems or 0)
    streamWriteInt16(streamId, readyCount)

    local numRecipes = (spec.items and #spec.items) or 0
    streamWriteUInt16(streamId, numRecipes)
    if numRecipes > 0 then
        for i = 1, numRecipes do
            local c = (spec.readyCounts and spec.readyCounts[i]) or 0
            streamWriteUInt16(streamId, c)
        end
    end
end

function PlaceableCooker:onReadStream(streamId, connection)
    local spec = self.spec_placeableCooker
    if not spec then return end

    local st = spec.state or {}
    spec.state = st

    st.cooking   = streamReadBool(streamId)
    st.ready     = streamReadBool(streamId)
    st.spoiled   = streamReadBool(streamId)
    st.endTimeMs = streamReadInt32(streamId)
    st.cookedDay = streamReadInt16(streamId)

    local curIndex = streamReadInt16(streamId)
    if curIndex and curIndex > 0 and spec.items and spec.items[curIndex] then
        st.current = spec.items[curIndex]
    else
        st.current = nil
    end

    spec.readyCount  = streamReadInt16(streamId) or 0
    local numRecipes = streamReadUInt16(streamId) or 0

    spec.readyCounts = spec.readyCounts or {}
    for i = 1, numRecipes do
        spec.readyCounts[i] = streamReadUInt16(streamId) or 0
    end

    spec.readyItems = spec.readyItems or {}

    st.ready = (spec.readyCount or 0) > 0

    if self._pcApplyCookAnimFromState then
        self:_pcApplyCookAnimFromState()
    end
end


--------------------------------------------------------------------------------
local function _getReadyCount(spec, recipeIndex)
    return (spec.readyCounts and spec.readyCounts[recipeIndex]) or 0
end
local function _incReadyCount(spec, recipeIndex)
    spec.readyCounts[recipeIndex] = (_getReadyCount(spec, recipeIndex) + 1)
end
local function _decReadyCount(spec, recipeIndex)
    local v = _getReadyCount(spec, recipeIndex)
    spec.readyCounts[recipeIndex] = math.max(0, v - 1)
end

--------------------------------------------------------------------------------
-- Update
--------------------------------------------------------------------------------
function PlaceableCooker:onUpdate(dt)
    local spec = self.spec_placeableCooker; if not spec then return end

    if spec._animActive and spec._animOneShotEndTime and (g_time or 0) >= spec._animOneShotEndTime then
        _pcAnimStop(self, "oneShotEnd")
        spec._animOneShotEndTime = nil
    end

    if spec.state.cooking and _nowMs() >= (spec.state.endTimeMs or 0) then
        local cookedPayload = PlaceableCooker._pcMakePayloadFromCurrent(self)
        if cookedPayload then
            table.insert(spec.readyItems, cookedPayload)
            _incReadyCount(spec, cookedPayload.index or 0)
            spec.readyCount = #spec.readyItems
            spec.state.ready = spec.readyCount > 0
        end
        spec.state.cooking = false
        spec.state.current = nil
        spec.state.endTimeMs = 0
        spec.state.cookedDay = _dayIndex()
        _topNotify(spec.title, (g_i18n and g_i18n:getText("ui_pcooker_ready")) or "Готово", cookedPayload and cookedPayload.icon)
        PCC_StateSyncEvent.send(self)

        _pcAnimStop(self, "cookEnd")

        if _G.PlaceableCookerHUD and PlaceableCookerHUD.INSTANCE
            and PlaceableCookerHUD.INSTANCE.placeable == self
            and g_gui and g_gui:getIsGuiVisible() then
            local hud = PlaceableCookerHUD.INSTANCE
            hud:pullData(true)
            hud:updateStatusText()
            hud:updateButtons()
        end
    end
    if spec.state.cooking then self:raiseActive() end
end

--------------------------------------------------------------------------------
-- Trigger
--------------------------------------------------------------------------------
function PlaceableCooker:pcTriggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
    if not (onEnter or onLeave) then return end
    if g_localPlayer == nil then return end
    if otherActorId ~= g_localPlayer.rootNode then return end

    if onEnter then
        if Platform.gameplay and Platform.gameplay.autoActivateTrigger and self.pcActivatable:getIsActivatable() then
            self.pcActivatable:run()
        else
            g_currentMission.activatableObjectsSystem:addActivatable(self.pcActivatable)
        end
    end
    if onLeave then
        g_currentMission.activatableObjectsSystem:removeActivatable(self.pcActivatable)
    end
end

function PlaceableCooker:pcGetIsAccessible(farmId)
    return self:getOwnerFarmId() == farmId or farmId == 0
end

--------------------------------------------------------------------------------
-- Server
--------------------------------------------------------------------------------
function PlaceableCooker:pcServerStartCook(itemIndex, farmId)
    local spec = self.spec_placeableCooker; if not spec then _topNotify("Cooker", "Нет спека") return false, "noSpec" end
    local it = spec.items[itemIndex]; if not it then _topNotify(spec.title, "Не выбрано блюдо") return false, "noItem" end
    if spec.state.cooking then _topNotify(spec.title, "Уже готовим") return false, "busy" end

    local cap = math.max(0, tonumber(it.readyCap) or 0)
    local cur = _getReadyCount(spec, itemIndex)
    if cap > 0 and cur >= cap then
        local msg = string.format("%s: %d/%d", (g_i18n and g_i18n:getText("ui_pcooker_noSpace") or "Нет места для блюда"), cur, cap)
        _topNotify(_l10n(it.name), msg, it.icon)
        return false, "noSpace"
    end

    local price = math.max(0, tonumber(it.price) or 0)
    if price > 0 then
        local realFarmId = farmId
        if realFarmId == nil or realFarmId == 0 then
            realFarmId = (g_currentMission and g_currentMission:getFarmId()) or 1
        end
        local farm = g_farmManager and g_farmManager:getFarmById(realFarmId)
        if not farm then _topNotify(spec.title, "Нет фермы") return false, "noFarm" end
        if (farm.money or 0) < price then _topNotify(spec.title, "Недостаточно денег") return false, "noMoney" end

        if g_currentMission and g_currentMission.addMoney then
            g_currentMission:addMoney(-price, realFarmId, MoneyType.OTHER or 0, true, true)
        elseif farm.changeBalance then
            farm:changeBalance(-price)
        end
    end

    Logging.info("[PlaceableCooker] start cook item=%s price=%s", tostring(it.name), tostring(price))
    spec.state.cooking   = true
    spec.state.ready     = (#spec.readyItems > 0)
    spec.state.current   = _copyTable(it)
    spec.state.current.index = itemIndex
    spec.state.endTimeMs = _nowMs() + math.max(1, it.cookSec or 1) * 1000
    spec.state.cookedDay = -1

    _pcVisSet(self, spec.state.current, true)
    _pcAnimStartForItem(self, spec.state.current)

    self:raiseActive()
    PCC_StateSyncEvent.send(self)
    return true, price
end

function PlaceableCooker:pcServerFinalizeIfReady()
    local spec = self.spec_placeableCooker
    if not spec or not spec.state.cooking then return end
    if _nowMs() >= (spec.state.endTimeMs or 0) then
        local cookedPayload = PlaceableCooker._pcMakePayloadFromCurrent(self)
        if cookedPayload then
            table.insert(spec.readyItems, cookedPayload)
            _incReadyCount(spec, cookedPayload.index or 0)
            spec.readyCount = #spec.readyItems
            spec.state.ready = true
        end
        spec.state.cooking = false
        spec.state.current = nil
        spec.state.endTimeMs = 0
        spec.state.cookedDay = _dayIndex()

        _pcAnimStop(self, "finalize")

        PCC_StateSyncEvent.send(self)
    end
end

function PlaceableCooker._pcMakePayloadFromCurrent(self)
    local spec = self.spec_placeableCooker; if not spec then return nil end
    local it = spec.state.current; if not it then return nil end

    local hungerGain = math.floor(tonumber(it.hungerGain) or 0)
    local vigorGain  = math.floor(tonumber(it.vigorGain)  or 0)
    local kind       = (it.effectId and it.effectId~="") and "effect" or "food"

    local iconRel = it.icon or ""
    iconRel = tostring(iconRel or "")
    if iconRel ~= "" then
        iconRel = iconRel:gsub("\\", "/")

        local baseDir = tostring(PlaceableCooker.modDirectory or ""):gsub("\\", "/")
        if baseDir ~= "" and iconRel:sub(1, #baseDir) == baseDir then
            iconRel = iconRel:sub(#baseDir + 1)
            if iconRel:sub(1, 1) == "/" then
                iconRel = iconRel:sub(2)
            end
        end

        if iconRel:lower():sub(1, 6) == "datas/" then
            iconRel = "data/" .. iconRel:sub(7)
        end
    end

    return {
        index          = it.index or 0,
        hungerGain     = hungerGain,
        vigorGain      = vigorGain,
        kind           = kind,
        icon           = iconRel,
        name           = it.name or "",
        desc           = it.desc or "",
        effectId       = it.effectId or "",
        effectsFile    = spec.effectsFile or "",
        placeableTitle = spec.title or ((g_i18n and g_i18n:getText("ui_pcooker_title")) or "Кухня")
    }
end


local function _popFirstReady(spec)
    if #spec.readyItems <= 0 then return nil end
    local payload = table.remove(spec.readyItems, 1)
    if payload and payload.index and payload.index > 0 then
        _decReadyCount(spec, payload.index)
    end
    spec.readyCount = #spec.readyItems
    spec.state.ready = spec.readyCount > 0
    return payload
end

local function _popReadyByRecipe(spec, recipeIndex)
    if #spec.readyItems <= 0 then return nil end

    local function normalizeIcon(payload)
        if not payload then return payload end
        local iconRel = payload.icon or ""
        iconRel = tostring(iconRel or "")
        if iconRel ~= "" then
            iconRel = iconRel:gsub("\\", "/")

            local baseDir = tostring(PlaceableCooker.modDirectory or ""):gsub("\\", "/")
            if baseDir ~= "" and iconRel:sub(1, #baseDir) == baseDir then
                iconRel = iconRel:sub(#baseDir + 1)
                if iconRel:sub(1, 1) == "/" then
                    iconRel = iconRel:sub(2)
                end
            end

            if iconRel:lower():sub(1, 6) == "datas/" then
                iconRel = "data/" .. iconRel:sub(7)
            end
        end
        payload.icon = iconRel
        return payload
    end

    if recipeIndex and recipeIndex > 0 then
        for i = 1, #spec.readyItems do
            local p = spec.readyItems[i]
            if (p.index or 0) == recipeIndex then
                table.remove(spec.readyItems, i)
                if p.index and p.index > 0 then
                    spec.readyCounts[p.index] = math.max(0, (spec.readyCounts[p.index] or 0) - 1)
                end
                spec.readyCount  = #spec.readyItems
                spec.state.ready = spec.readyCount > 0
                return normalizeIcon(p)
            end
        end
        return nil
    else
        local payload = table.remove(spec.readyItems, 1)
        if payload and payload.index and payload.index > 0 then
            spec.readyCounts[payload.index] = math.max(0, (spec.readyCounts[payload.index] or 0) - 1)
        end
        spec.readyCount  = #spec.readyItems
        spec.state.ready = spec.readyCount > 0
        return normalizeIcon(payload)
    end
end


function PlaceableCooker:pcServerConsumeCooked(farmId, recipeIndex)
    local spec = self.spec_placeableCooker
    if not spec then return false, "noSpec" end
    if #spec.readyItems <= 0 then return false, "notReady" end

    local payload = _popReadyByRecipe(spec, recipeIndex)
    if not payload then return false, "notReadyRecipe" end

    PCC_StateSyncEvent.send(self)
    return true, payload
end

function PlaceableCooker:pcServerTakeToInventory(farmId, recipeIndex)
    local spec = self.spec_placeableCooker
    if not spec then return false, "noSpec" end
    if #spec.readyItems <= 0 then return false, "notReady" end

    local payload = _popReadyByRecipe(spec, recipeIndex)
    if not payload then return false, "notReadyRecipe" end

    PCC_StateSyncEvent.send(self)
    return true, payload
end

function PlaceableCooker:pcServerDiscardCooked(farmId)
    local spec = self.spec_placeableCooker; if not spec then return false, "noSpec" end
    if (#spec.readyItems <= 0) and (not spec.state.cooking) then return false, "notReady" end

    if #spec.readyItems > 0 then
        for _, p in ipairs(spec.readyItems) do
            if p.index and p.index > 0 then _decReadyCount(spec, p.index) end
        end
        spec.readyItems = {}
        spec.readyCount = 0
        spec.state.ready = false
    end
    if spec.state.cooking then
        spec.state.cooking   = false
        spec.state.current   = nil
        spec.state.endTimeMs = 0
        spec.state.cookedDay = -1
        _pcAnimStop(self, "discard")
    end
    _topNotify(spec.title, (g_i18n and g_i18n:getText("ui_pcooker_discard")) or "Выкинуто", nil)
    PCC_StateSyncEvent.send(self)
    return true, "ok"
end

--------------------------------------------------------------------------------
-- Activatable
--------------------------------------------------------------------------------
PC_Activatable = {}
local PC_Activatable_mt = Class(PC_Activatable)
function PC_Activatable.new(placeable)
    local self = setmetatable({}, PC_Activatable_mt)
    self.placeable = placeable
    self.activateText = (g_i18n and g_i18n:getText("ui_pcooker_pressOpen")) or "Открыть кухню"
    return self
end
function PC_Activatable:getIsActivatable()
    if g_localPlayer == nil then return false end
    if g_localPlayer.getIsInVehicle and g_localPlayer:getIsInVehicle() then return false end
    local farmId = g_currentMission and g_currentMission:getFarmId() or 1
    return self.placeable and self.placeable.pcGetIsAccessible and self.placeable:pcGetIsAccessible(farmId) or false
end
function PC_Activatable:run()
    if not g_gui then return end
    if not (g_gui.guis and g_gui.guis["PlaceableCookerHUD"]) then
        if PlaceableCookerHUD and PlaceableCookerHUD.register then
            if not PlaceableCookerHUD.register() then print("[PlaceableCooker] HUD register failed"); return end
        else
            print("[PlaceableCooker] HUD missing"); return
        end
    end
    PlaceableCookerHUD.INSTANCE.placeable = self.placeable
    g_gui:showDialog("PlaceableCookerHUD")
end
