-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

PickupFilter = {}
PickupFilter.DEBUG = false

-- Белый список:
--   1) типы из <vehicle type="...">  (например "baseMobileStorageBox")
--   2) базовые имена xml/i3d (например "mobileStorageBox")
PickupFilter.ALLOW = {
    baseMobileStorageBox = true,
    mobileStorageBox     = true,
}

PickupFilter.FORCE_TARGETER  = true
PickupFilter.PICKUP_DISTANCE = 10.0
PickupFilter.TARGET_MIN_DOT  = 0.5

local function dbg(fmt, ...)
    if PickupFilter.DEBUG then
        print(("[PickupFilter] " .. fmt):format(...))
    end
end

-- -------- utils ----------
local function strOrNil(x) return (type(x)=="string" and x~="") and x or nil end

local function basenameNoExt(path)
    local s = strOrNil(path); if not s then return nil end
    local base = s:match("([^/\\]+)%.%w+$"); if base and base~="" then return base end
    base = s:match("([^/\\]+)$"); return base
end

local function isVehicle(obj)
    if obj == nil then return false end
    if type(obj.isa) == "function" then
        local ok, res = pcall(obj.isa, obj, Vehicle)
        if ok and res == true then return true end
    end
    if obj.spec_motorized or obj.spec_attachable or obj.typeName ~= nil then
        return true
    end
    return false
end

local function extractTypeKeys(vehicle)
    local typeKey = nil
    if vehicle then
        typeKey = strOrNil(vehicle.typeName) or typeKey
        if not typeKey and vehicle.type then
            typeKey = strOrNil(vehicle.type) or strOrNil(vehicle.type.name)
        end
    end
    local fileKey = nil
    if vehicle then
        fileKey = basenameNoExt(vehicle.configFileName)
               or basenameNoExt(vehicle.xmlFilename)
               or basenameNoExt(vehicle.i3dFilename)
    end
    return typeKey, fileKey
end

local function isAllowedVehicle(vehicle)
    local typeKey, fileKey = extractTypeKeys(vehicle)
    if typeKey and PickupFilter.ALLOW[typeKey] then
        dbg("allowed by type: %s", typeKey); return true
    end
    if fileKey and PickupFilter.ALLOW[fileKey] then
        dbg("allowed by file: %s (cfg=%s)", fileKey, tostring(vehicle and vehicle.configFileName)); return true
    end
    dbg("not in whitelist  type=%s  file=%s  cfg=%s", tostring(typeKey), tostring(fileKey), tostring(vehicle and vehicle.configFileName))
    return false
end

local function patchCanPickUp()
    if HandToolHands == nil or HandToolHands.getCanPickUpNode == nil then
        print("[PickupFilter] ERROR: HandToolHands.getCanPickUpNode not found — patch skipped")
        return
    end

    HandToolHands.getCanPickUpNode = Utils.overwrittenFunction(
        HandToolHands.getCanPickUpNode,
        function(self, superFunc, node)
            local vanillaOk = superFunc(self, node)

            local obj = g_currentMission and g_currentMission:getNodeObject(node) or nil
            if obj == nil then
                return vanillaOk
            end

            if isVehicle(obj) then
                if vanillaOk then
                    return true
                else
                    if isAllowedVehicle(obj) then
                        return true
                    else
                        return false
                    end
                end
            end

            return vanillaOk
        end
    )

    print("[PickupFilter] Patched HandToolHands:getCanPickUpNode (whitelist only relaxes vanilla NO)")
end

local TargeterBooster = { _registeredFor = {} }

function TargeterBooster:tryRegisterForPlayer(player)
    if not PickupFilter.FORCE_TARGETER then return end
    if not player or not player.targeter then return end
    local pid = tostring(player)
    if self._registeredFor[pid] then return end

    local minDot = PickupFilter.TARGET_MIN_DOT
    local dist   = PickupFilter.PICKUP_DISTANCE
    local mask   = HandToolHands and HandToolHands.TARGET_MASK or 0

    if type(player.targeter.addTargetType) == "function" then
        player.targeter:addTargetType(HandToolHands, mask, minDot, dist)
        self._registeredFor[pid] = true
        dbg("Targeter added for player=%s mask=%s dist=%.2f", pid, tostring(mask), dist)
    end
end

function TargeterBooster:update(dt)
    if not PickupFilter.FORCE_TARGETER then return end
    local mission = g_currentMission
    if not mission then return end
    if mission.player then self:tryRegisterForPlayer(mission.player) end
    if mission.players then
        for _, p in pairs(mission.players) do
            self:tryRegisterForPlayer(p)
        end
    end
end

local function registerUpdateable()
    if not PickupFilter.FORCE_TARGETER then return end
    if g_currentMission and not TargeterBooster._added then
        g_currentMission:addUpdateable(TargeterBooster)
        TargeterBooster._added = true
        print("[PickupFilter] Targeter booster active (no EasyDev needed)")
    end
end

local function init()
    patchCanPickUp()
    registerUpdateable()
end

init()

function PickupFilter.addAllowed(key)
    if strOrNil(key) then
        PickupFilter.ALLOW[key] = true
        print(("[PickupFilter] Added allow key: %s"):format(key))
    end
end

return PickupFilter
