-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 23.04.2025
-- Отображение текста над или в указанной ноде...

TextMarkers = {}
local TextMarkers_mt = Class(TextMarkers)

local function parseColorString(colorStr)
    local r, g, b, a = 1, 1, 1, 1 -- значения по умолчанию (белый)
    if colorStr then
        local components = {}
        for component in colorStr:gmatch("[^,]+") do
            table.insert(components, tonumber(component) or 1)
        end
        r = components[1] or 1
        g = components[2] or 1
        b = components[3] or 1
        a = components[4] or 1
    end
    return { r, g, b, a }
end

function TextMarkers.prerequisitesPresent(specializations)
    return true
end

function TextMarkers.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad", TextMarkers)
    SpecializationUtil.registerEventListener(placeableType, "onDelete", TextMarkers)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement", TextMarkers)
    SpecializationUtil.registerEventListener(placeableType, "onUpdate", TextMarkers)
    SpecializationUtil.registerEventListener(placeableType, "onDraw", TextMarkers)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream", TextMarkers)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream", TextMarkers)
end

function TextMarkers.registerXMLPaths(schema, basePath)
    schema:setXMLSpecializationType("TextMarkers")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".textMarkers.textMarker(?)#node", "Node to display text above")
    schema:register(XMLValueType.STRING, basePath .. ".textMarkers.textMarker(?)#text", "Text to display (supports $l10n_ for localization)")
    schema:register(XMLValueType.FLOAT,  basePath .. ".textMarkers.textMarker(?)#textSize", "Text size", 0.05)
    schema:register(XMLValueType.STRING, basePath .. ".textMarkers.textMarker(?)#textColor", "Text color (e.g., '1,1,1,1')", "1,1,1,1")
    schema:register(XMLValueType.FLOAT,  basePath .. ".textMarkers.textMarker(?)#offsetY", "Vertical offset above node", 3.0)
    schema:register(XMLValueType.FLOAT,  basePath .. ".textMarkers.textMarker(?)#maxDistance", "Max distance to display text (meters)", 50.0)
    schema:setXMLSpecializationType()
end

function TextMarkers:onLoad(savegame)
    self.spec_textMarkers = self.spec_textMarkers or {}
    local spec = self.spec_textMarkers
    local xmlFile = self.xmlFile
    spec.textMarkers = {}

    Logging.info("TextMarkers: Loading text markers for '%s' (isServer: %s, isClient: %s)",
        self:getName(), tostring(self.isServer), tostring(self.isClient))

    for _, key in xmlFile:iterator("placeable.textMarkers.textMarker") do
        local node = xmlFile:getValue(key .. "#node", nil, self.components, self.i3dMappings)
        if node == nil then
            Logging.xmlWarning(xmlFile, "TextMarkers: Missing text marker node for '%s'", key)
        else
            local text = xmlFile:getValue(key .. "#text")
            local textSize = xmlFile:getValue(key .. "#textSize", 0.05)
            local textColor = parseColorString(xmlFile:getValue(key .. "#textColor", "1,1,1,1"))
            local offsetY = xmlFile:getValue(key .. "#offsetY", 3.0)
            local maxDistance = xmlFile:getValue(key .. "#maxDistance", 50.0)

            if textSize > 1.0 then
                Logging.xmlWarning(xmlFile, "TextMarkers: Text size %f for '%s' is too large, clamping to 0.05", textSize, key)
                textSize = 0.05
            end

            local displayText = nil
            if text ~= nil then
                if text:startsWith("$l10n_") then
                    local l10nKey = text:sub(7)
                    displayText = g_i18n:getText(l10nKey) or text
                    Logging.info("TextMarkers: Localized text for '%s': '%s' (key: %s)", key, displayText, l10nKey)
                else
                    displayText = text
                    Logging.info("TextMarkers: Direct text for '%s': '%s'", key, displayText)
                end
            end

            if displayText == nil then
                Logging.xmlWarning(xmlFile, "TextMarkers: Missing or invalid text for '%s'", key)
            else
                table.insert(spec.textMarkers, {
                    node        = node,
                    text        = displayText,
                    textSize    = textSize,
                    textColor   = textColor,
                    offsetY     = offsetY,
                    maxDistance = maxDistance
                })
                Logging.info("TextMarkers: Added marker for '%s' with text '%s', size %f, offsetY %f, maxDistance %f",
                    key, displayText, textSize, offsetY, maxDistance)
            end
        end
    end

    Logging.info("TextMarkers: Loaded %d text markers for '%s'", #spec.textMarkers, self:getName())
end

function TextMarkers:onFinalizePlacement()
    Logging.info("TextMarkers: Finalizing placement for '%s'", self:getName())
    if self.isClient then
        g_currentMission:addDrawable(self)
        Logging.info("TextMarkers: Added '%s' to drawables", self:getName())
    end
    self:raiseActive()
end

function TextMarkers:onUpdate(dt)
    self:raiseActive()
end

function TextMarkers:onDelete()
    local spec = self.spec_textMarkers
    if spec then
        if self.isClient then
            g_currentMission:removeDrawable(self)
            Logging.info("TextMarkers: Removed '%s' from drawables", self:getName())
        end
        spec.textMarkers = nil
        Logging.info("TextMarkers: Deleted text markers for '%s'", self:getName())
    end
end

function TextMarkers:onDraw()
    if not self.isClient then
        return
    end

    local spec = self.spec_textMarkers
    if spec == nil or spec.textMarkers == nil or #spec.textMarkers == 0 then
        return
    end

    local cam = getCamera()
    if cam == nil then
        return
    end

    local cx, cy, cz = getWorldTranslation(cam)

    for i = #spec.textMarkers, 1, -1 do
        local marker = spec.textMarkers[i]

        if marker == nil or marker.node == nil then
            table.remove(spec.textMarkers, i)
        else
            local ok, x, y, z = pcall(getWorldTranslation, marker.node)
            if not ok or x == nil or y == nil or z == nil then
                Logging.warning("TextMarkers: Invalid node for marker '%s', removing (MP desync / deleted node)", tostring(marker.text))
                table.remove(spec.textMarkers, i)
            else
                -- Доп. проверка диапазона по высоте
                if y >= -100 and y <= 1000 then
                    local offsetY  = marker.offsetY or 0
                    local worldY   = y + offsetY
                    local dx, dy, dz = x - cx, worldY - cy, z - cz
                    local distance = MathUtil.vector3Length(dx, dy, dz)
                    local maxDist  = marker.maxDistance or 50

                    if distance <= maxDist then
                        local textSize = getCorrectTextSize(marker.textSize or 0.05)

                        Utils.renderTextAtWorldPosition(
                            x, worldY, z,
                            marker.text,
                            textSize,
                            0,
                            unpack(marker.textColor or {1,1,1,1})
                        )
                    end
                end
            end
        end
    end
end


function TextMarkers:onWriteStream(streamId, connection)
    Logging.devInfo("TextMarkers: onWriteStream for '%s' (nothing to sync)", self:getName())
end

function TextMarkers:onReadStream(streamId, connection)
    self.spec_textMarkers = self.spec_textMarkers or {}
    Logging.devInfo("TextMarkers: onReadStream for '%s' (markers loaded from XML)", self:getName())
end
