-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

PlaceableTimeWeatherDisplay = {}

local MONTH_FROM_PERIOD = {
    [1]=3,  [2]=4,  [3]=5,   -- Early/Mid/Late Spring  -> Mar/Apr/May
    [4]=6,  [5]=7,  [6]=8,   -- Early/Mid/Late Summer  -> Jun/Jul/Aug
    [7]=9,  [8]=10, [9]=11,  -- Early/Mid/Late Autumn  -> Sep/Oct/Nov
    [10]=12,[11]=1, [12]=2   -- Early/Mid/Late Winter  -> Dec/Jan/Feb
}

function PlaceableTimeWeatherDisplay.prerequisitesPresent(specializations)
    return true
end

-- ==== helpers =============================================================

local function countMaskZeros(mask)
    local w, s = 0, tostring(mask or "")
    for i = 1, #s do
        if s:sub(i,i) == "0" then w = w + 1 end
    end
    return w
end

local function formatByMaskIntSpaces(num, mask)
    local width = countMaskZeros(mask or "")
    if width <= 0 then width = 1 end
    local n = math.floor((tonumber(num) or 0) + 0.5)
    local s = tostring(math.abs(n))
    if #s < width then
        s = string.rep(" ", width - #s) .. s
    elseif #s > width then
        s = s:sub(#s - width + 1)
    end
    return s
end

local function formatHHMMSpaces(totalMinutes, mask)
    local m = math.max(0, math.floor(tonumber(totalMinutes) or 0))
    local hh = math.floor(m / 60)
    local mm = m % 60
    mask = mask or "00:00"
    local colon = mask:find(":", 1, true) or 3
    local maskH = mask:sub(1, colon - 1)
    local maskM = mask:sub(colon + 1)
    local wH = countMaskZeros(maskH); if wH <= 0 then wH = 2 end
    local wM = countMaskZeros(maskM); if wM <= 0 then wM = 2 end
    local hStr = tostring(hh)
    if #hStr < wH then
        hStr = string.rep(" ", wH - #hStr) .. hStr
    elseif #hStr > wH then
        hStr = hStr:sub(#hStr - wH + 1)
    end
    local mStr = tostring(mm)
    if #mStr < wM then
        mStr = string.rep("0", wM - #mStr) .. mStr
    elseif #mStr > wM then
        mStr = mStr:sub(#mStr - wM + 1)
    end
    return hStr .. ":" .. mStr
end

local function getEnv()
    return g_currentMission and g_currentMission.environment
end

local function getGameMinutes()
    local env = getEnv()
    local dayTimeMs = (env and env.dayTime) or 0
    return math.floor(dayTimeMs / 60000) % 1440
end

local function getOutsideTempC()
    local env = getEnv()
    if env and env.weather and env.weather.getCurrentTemperature then
        return math.floor((env.weather:getCurrentTemperature() or 0) + 0.5)
    end
    if env and env.temperature then
        return math.floor((env.temperature or 0) + 0.5)
    end
    return 0
end

local function getWeatherShort()
    local env = getEnv()
    local wx = env and env.weather
    local okR, raining = pcall(function() return wx and wx.getIsRaining and wx:getIsRaining() end)
    if okR and raining then return "RAIN" end
    local okS, snowing = pcall(function() return wx and wx.getIsSnowing and wx:getIsSnowing() end)
    if okS and snowing then return "SNOW" end
    local okC, clouds = pcall(function() return wx and wx.getCloudCoverage and wx:getCloudCoverage() end)
    if okC and type(clouds) == "number" and clouds >= 0.66 then
        return "CLOUD"
    end
    return "SUN"
end

local function getGameDMY()
    local env = getEnv()
    if not env then return "00.00" end

    local dpp = env.daysPerPeriod
                or (g_currentMission and g_currentMission.missionInfo and g_currentMission.missionInfo.daysPerPeriod)
                or 1
    if dpp < 1 then dpp = 1 end

    local period1
    if env.season and env.season.getCurrentPeriod then
        local ok, p = pcall(function() return env.season:getCurrentPeriod() end)
        if ok and type(p) == "number" then
            if p >= 0 and p <= 11 then period1 = p + 1
            elseif p >= 1 and p <= 12 then period1 = p end
        end
    end
    if not period1 and env.getSeasonPeriod then
        local ok, p = pcall(function() return env:getSeasonPeriod() end)
        if ok and type(p) == "number" then
            if p >= 0 and p <= 11 then period1 = p + 1
            elseif p >= 1 and p <= 12 then period1 = p end
        end
    end

    if not period1 then
        local rawPeriodFields = { env.currentSeasonPeriod, env.seasonPeriod, env.currentPeriod, env.period }
        for _, pf in ipairs(rawPeriodFields) do
            if type(pf) == "number" then
                if pf >= 0 and pf <= 11 then period1 = pf + 1; break
                elseif pf >= 1 and pf <= 12 then period1 = pf; break end
            end
        end
    end

    local dayInPeriod
    if env.season and env.season.getDayInPeriod then
        local ok, dip = pcall(function() return env.season:getDayInPeriod() end)
        if ok and type(dip) == "number" and dip >= 1 then dayInPeriod = dip end
    end
    if (not dayInPeriod) and env.getDayInSeasonPeriod then
        local ok, dip = pcall(function() return env:getDayInSeasonPeriod() end)
        if ok and type(dip) == "number" and dip >= 1 then dayInPeriod = dip end
    end
    if (not dayInPeriod) and type(env.dayInPeriod) == "number" and env.dayInPeriod >= 1 then
        dayInPeriod = env.dayInPeriod
    end

    local dayInYear = env.dayInYear
    if type(dayInYear) ~= "number" then
        local cd = env.currentDay
        if type(cd) == "number" and cd >= 1 then
            if period1 and cd <= dpp then
                dayInYear = (period1 - 1) * dpp + cd
            else
                dayInYear = cd
            end
        else
            dayInYear = 1
        end
    end

    if not period1 then
        period1 = math.floor((math.max(1, dayInYear) - 1) / dpp) + 1
        period1 = ((period1 - 1) % 12) + 1
    end

    if not dayInPeriod then
        dayInPeriod = ((math.max(1, dayInYear) - 1) % dpp) + 1
    end

    local month = MONTH_FROM_PERIOD[period1] or 1

    return string.format("%02d.%02d", dayInPeriod, month)
end

local function getFont(name, customEnv)
    local nm = (name or "DIGIT"):upper()
    local envName = customEnv or g_currentModName

    if g_materialManager and g_materialManager.getFontMaterial then
        local ok, res = pcall(function() return g_materialManager:getFontMaterial(nm, envName) end)
        if ok and res then return res end
    end
    if g_materialManager and g_materialManager.getFont then
        local ok, res = pcall(function() return g_materialManager:getFont(nm, envName) end)
        if ok and res then return res end
    end
    return nil
end

-- ==== message center (опционально, но полезно) ===========================

local function _subscribeEnvTicks(self)
    if not g_messageCenter then return end
    local spec = self.spec_timeWeatherDisplay
    if not spec then return end

    if MessageType and g_messageCenter.subscribe then
        local cb = PlaceableTimeWeatherDisplay._twd_onEnvTick

        spec.msgHour   = g_messageCenter:subscribe(MessageType.HOUR_CHANGED,   cb, self)
        spec.msgDay    = g_messageCenter:subscribe(MessageType.DAY_CHANGED,    cb, self)
        spec.msgPeriod = g_messageCenter:subscribe(MessageType.PERIOD_CHANGED, cb, self)
        -- spec.msgMinute = g_messageCenter:subscribe(MessageType.MINUTE_CHANGED, cb, self)
    end
end

function PlaceableTimeWeatherDisplay._twd_onEnvTick(self, ...)
    local spec = self.spec_timeWeatherDisplay
    if spec and spec.displays and #spec.displays > 0 then
        updateDisplays(spec.displays)
    end
end

local function _unsubscribeEnvTicks(self)
    if not g_messageCenter then return end
    local spec = self.spec_timeWeatherDisplay
    if not spec then return end
    local function u(h) if h and g_messageCenter.unsubscribe then g_messageCenter:unsubscribe(h) end end
    u(spec.msgHour);   spec.msgHour = nil
    u(spec.msgDay);    spec.msgDay = nil
    u(spec.msgPeriod); spec.msgPeriod = nil
    -- u(spec.msgMinute); spec.msgMinute = nil
end

-- ==== XML schema ==========================================================
function PlaceableTimeWeatherDisplay.registerXMLPaths(schema, basePath)
    local base = basePath .. ".timeWeatherDisplay"
    schema:register(XMLValueType.INT,  base .. "#refreshMs", "Интервал обновления (мс)", 200)

    local p = base .. ".display(?)"
    schema:register(XMLValueType.STRING,     p .. "#type",         "time|outsideTemp|weather|date|text", "time")
    schema:register(XMLValueType.NODE_INDEX, p .. "#node",         "CharacterLine-нод")
    schema:register(XMLValueType.STRING,     p .. "#font",         "Имя шрифта (FontMaterial)", "DIGIT")
    schema:register(XMLValueType.STRING,     p .. "#alignment",    "LEFT|CENTER|RIGHT", "CENTER")
    schema:register(XMLValueType.FLOAT,      p .. "#size",         "Размер символа", 0.13)
    schema:register(XMLValueType.FLOAT,      p .. "#scaleX",       "Текстурный масштаб X", 1)
    schema:register(XMLValueType.FLOAT,      p .. "#scaleY",       "Текстурный масштаб Y", 1)
    schema:register(XMLValueType.STRING,     p .. "#mask",         "Маска для чисел/времени/даты (00:00/000/00.00)", nil)
    schema:register(XMLValueType.FLOAT,      p .. "#emissiveScale","Яркость эмиссии", 1)
    schema:register(XMLValueType.COLOR,      p .. "#color",        "Цвет текста", "1 1 1 1")
    schema:register(XMLValueType.COLOR,      p .. "#hiddenColor",  "Цвет скрытого", "")

    schema:register(XMLValueType.STRING,     p .. "#textMask",     "Статический текст (type='text')", "")
    schema:register(XMLValueType.COLOR,      p .. "#textColor",    "Цвет для текста", nil)
    schema:register(XMLValueType.FLOAT,      p .. "#textSize",     "Размер для текста", nil)
    schema:register(XMLValueType.STRING,     p .. "#textAlignment","Выравнивание для текста", nil)
end

function PlaceableTimeWeatherDisplay.registerSavegameXMLPaths(schema, basePath)
end

function PlaceableTimeWeatherDisplay.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",               PlaceableTimeWeatherDisplay)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",             PlaceableTimeWeatherDisplay)
    SpecializationUtil.registerEventListener(placeableType, "onUpdate",             PlaceableTimeWeatherDisplay)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement",  PlaceableTimeWeatherDisplay)
    SpecializationUtil.registerEventListener(placeableType, "onPostLoad",           PlaceableTimeWeatherDisplay)
end

-- ==== lifecycle ==========================================================

function PlaceableTimeWeatherDisplay:onLoad(savegame)
    local spec = {}
    self.spec_timeWeatherDisplay = spec

    local xmlFile = self.xmlFile
    local baseKey = "placeable.timeWeatherDisplay"

    spec.refreshMs = xmlFile:getValue(baseKey .. "#refreshMs", 200)
    spec.timerMs   = 0
    spec.displays  = {}

    local supported = {
        time = true,
        outsidetemp = true,
        outsideTemp = true,
        weather = true,
        text = true,
        date = true
    }

    xmlFile:iterate(baseKey .. ".display", function(_, dKey)
        local node = xmlFile:getValue(dKey .. "#node", nil, self.components, self.i3dMappings)
        if node == nil then return end

        local kindRaw = (xmlFile:getValue(dKey .. "#type", "time") or "time")
        local kind = string.lower(kindRaw)
        if not supported[kind] then return end
        if kind == "outsidetemp" then kind = "outsideTemp" end

        local fontName = (xmlFile:getValue(dKey .. "#font", "DIGIT") or "DIGIT"):upper()
        local fm = getFont(fontName, self.customEnvironment or g_currentModName)
        if not fm then
            Logging.warning("[TimeWeatherDisplay] font '%s' not found for '%s', skipping", fontName, getName(node))
            return
        end

        local alignStr = (xmlFile:getValue(dKey .. "#alignment", "CENTER") or "CENTER"):upper()
        local align = RenderText["ALIGN_" .. alignStr] or RenderText.ALIGN_CENTER

        local size  = xmlFile:getValue(dKey .. "#size", 0.13)
        local sx    = xmlFile:getValue(dKey .. "#scaleX", 1)
        local sy    = xmlFile:getValue(dKey .. "#scaleY", 1)
        local mask  = xmlFile:getValue(dKey .. "#mask")
        local emi   = xmlFile:getValue(dKey .. "#emissiveScale", 1)
        local color = xmlFile:getValue(dKey .. "#color", {1,1,1,1}, true)
        local hiddenColor = xmlFile:getValue(dKey .. "#hiddenColor", nil, true)

        local textMask = xmlFile:getValue(dKey .. "#textMask", "", false)
        local textColor = xmlFile:getValue(dKey .. "#textColor", nil, true)
        local textSize = xmlFile:getValue(dKey .. "#textSize", nil)
        local textAlign = xmlFile:getValue(dKey .. "#textAlignment", nil)

        if kind ~= "text" and ((not mask) or mask == "") then
            if kind == "time" then
                mask = "00:00"
            elseif kind == "date" then
                mask = "00.00"
            else
                mask = "000"
            end
        end

        local charCount
        if kind == "text" then
            local txt = tostring(textMask or "")
            if txt == "" then txt = " " end
            charCount = #txt
        else
            charCount = (mask and #mask) or 3
        end

        local ok, cl = pcall(function() return CharacterLine.new(node, fm, charCount) end)
        if not ok or not cl then
            Logging.warning("[TimeWeatherDisplay] can't create CharacterLine for node '%s', skipping", getName(node))
            return
        end

        if kind == "text" and textSize then
            size = textSize
        end
        cl:setSizeAndScale(size, sx, sy)

        if kind == "text" and textAlign then
            local t = (textAlign or ""):upper()
            local ta = RenderText["ALIGN_" .. t]
            if ta ~= nil then align = ta end
        end
        cl:setTextAlignment(align)

        local useColor = (kind == "text" and textColor) or color
        cl:setColor(useColor, hiddenColor, emi)

        if kind == "text" then
            local txt = tostring(textMask or "")
            if txt == "" then txt = " " end
            cl:setText(txt)
        else
            local initStr = (mask or ""):gsub("0", " ")
            if initStr == "" then initStr = string.rep(" ", charCount) end
            cl:setText(initStr)
        end

        table.insert(spec.displays, {
            kind = kind,
            mask = mask,
            characterLine = cl,
            isText = (kind == "text"),
            textMask = textMask
        })
    end)

    if self.isClient then
        self:raiseActive()
    end
end

function PlaceableTimeWeatherDisplay:onPostLoad(savegame)
    if self.isClient then
        _subscribeEnvTicks(self)
    end
end

function PlaceableTimeWeatherDisplay:onDelete()
    local spec = self.spec_timeWeatherDisplay
    if not spec then return end
    _unsubscribeEnvTicks(self)
    for _, d in ipairs(spec.displays or {}) do
        if d.characterLine and d.characterLine.delete then
            d.characterLine:delete()
        end
    end
    spec.displays = {}
end

function PlaceableTimeWeatherDisplay:onFinalizePlacement()
    local spec = self.spec_timeWeatherDisplay
    if spec and spec.displays and #spec.displays > 0 then
        for _, d in ipairs(spec.displays) do
            if not d.isText then
                d.characterLine:setText("") -- сброс для гарантии перерисовки
            end
        end
        local ok, err = pcall(function() updateDisplays(spec.displays) end)
        if not ok then
            Logging.devWarning("[TimeWeatherDisplay] initial update failed: %s", tostring(err))
        end
    end
end

-- ==== dynamic read & update ==============================================

local function readValue(kind)
    local k = tostring(kind):lower()
    if k == "outsidetemp" then k = "outsideTemp" end

    if k == "time" then
        return getGameMinutes()
    elseif k == "outsideTemp" then
        return getOutsideTempC()
    elseif k == "weather" then
        return getWeatherShort()
    elseif k == "date" then
        return getGameDMY()
    else
        return 0
    end
end

local function padToMaskWidth(txt, mask)
    if not mask or mask == "" then return txt end
    local w = math.max(1, #mask)
    if #txt < w then
        return string.rep(" ", w - #txt) .. txt
    elseif #txt > w then
        return txt:sub(#txt - w + 1)
    end
    return txt
end

function updateDisplays(list)
    if not list or #list == 0 then return end

    for _, d in ipairs(list) do
        if d.isText then
        else
            local s
            if d.kind == "time" then
                local minutes = readValue("time") or 0
                s = formatHHMMSpaces(minutes, d.mask)

            elseif d.kind == "outsideTemp" then
                local v = readValue("outsideTemp") or 0
                s = formatByMaskIntSpaces(v, d.mask)

            elseif d.kind == "weather" then
                local txt = tostring(readValue("weather") or "SUN")
                s = padToMaskWidth(txt, d.mask)

            elseif d.kind == "date" then
                local txt = tostring(readValue("date") or "00.00")
                s = padToMaskWidth(txt, d.mask)

            else
                s = ""
            end

            d.characterLine:setText(s)
        end
    end
end

function PlaceableTimeWeatherDisplay:onUpdate(dt)
    local spec = self.spec_timeWeatherDisplay
    if not spec or (not spec.displays) or #spec.displays == 0 then return end

    if self.isClient then
        self:raiseActive()
    end

    spec.timerMs = (spec.timerMs or 0) + dt
    if spec.timerMs >= (spec.refreshMs or 200) then
        spec.timerMs = 0
        updateDisplays(spec.displays)
    end
end
