-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

PlaceableNodeAnimator = {}

local NodeAnimatorSetStateEvent = {}
local NodeAnimatorSetStateEvent_mt = Class(NodeAnimatorSetStateEvent, Event)
InitEventClass(NodeAnimatorSetStateEvent, "NodeAnimatorSetStateEvent")

local function _clamp01(x)
    x = tonumber(x) or 0
    if x < 0 then return 0 end
    if x > 1 then return 1 end
    return x
end

function NodeAnimatorSetStateEvent.emptyNew()
    return Event.new(NodeAnimatorSetStateEvent_mt)
end

function NodeAnimatorSetStateEvent.new(object, index, wantOpen, resetProgress)
    local self = NodeAnimatorSetStateEvent.emptyNew()
    self.object = object
    self.index = index
    self.wantOpen = wantOpen
    self.resetProgress = resetProgress and true or false
    return self
end

function NodeAnimatorSetStateEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.object)
    streamWriteUInt8(streamId, self.index or 0)
    streamWriteBool(streamId, self.wantOpen)
    streamWriteBool(streamId, self.resetProgress)
end

function NodeAnimatorSetStateEvent:readStream(streamId, connection)
    self.object = NetworkUtil.readNodeObject(streamId)
    self.index = streamReadUInt8(streamId)
    self.wantOpen = streamReadBool(streamId)
    self.resetProgress = streamReadBool(streamId)
    self:run(connection)
end

function NodeAnimatorSetStateEvent:run(connection)
    if self.object ~= nil and self.object.setNodeAnimatorTarget ~= nil then
        self.object:setNodeAnimatorTarget(self.index, self.wantOpen, self.resetProgress, true)
    end
end

-- ======================================================================

function PlaceableNodeAnimator.prerequisitesPresent(_)
    return true
end

function PlaceableNodeAnimator.registerFunctions(placeableType)
    SpecializationUtil.registerFunction(placeableType, "setNodeAnimatorTarget", PlaceableNodeAnimator.setNodeAnimatorTarget)
    SpecializationUtil.registerFunction(placeableType, "toggleNodeAnimator",    PlaceableNodeAnimator.toggleNodeAnimator)
end

function PlaceableNodeAnimator.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",               PlaceableNodeAnimator)
    SpecializationUtil.registerEventListener(placeableType, "onPostLoad",           PlaceableNodeAnimator)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",             PlaceableNodeAnimator)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream",         PlaceableNodeAnimator)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream",        PlaceableNodeAnimator)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement",  PlaceableNodeAnimator)
    SpecializationUtil.registerEventListener(placeableType, "onUpdate",             PlaceableNodeAnimator)
end

function PlaceableNodeAnimator.registerOverwrittenFunctions(placeableType) end

function PlaceableNodeAnimator.registerXMLPaths(schema, basePath)
    local base = basePath .. ".nodeAnimator"
    schema:register(XMLValueType.STRING, base.."#consolePrefix", "Console prefix for commands", "NodeAnim")

    local p = base..".item(?)"
    schema:register(XMLValueType.NODE_INDEX,   p.."#node",       "Node to animate")
    schema:register(XMLValueType.VECTOR_TRANS, p.."#posClosed",  "Closed position (x y z)", "0 0 0")
    schema:register(XMLValueType.VECTOR_TRANS, p.."#posOpen",    "Open position (x y z)",   "0 0 0")

    schema:register(XMLValueType.VECTOR_3,     p.."#rotClosed",  "Closed rotation DEGREES (x y z)", "0 0 0")
    schema:register(XMLValueType.VECTOR_3,     p.."#rotOpen",    "Open rotation DEGREES (x y z)",   "0 0 0")

    schema:register(XMLValueType.BOOL,         p.."#translate",  "Animate translation", true)
    schema:register(XMLValueType.BOOL,         p.."#rotate",     "Animate rotation",    true)

    schema:register(XMLValueType.NODE_INDEX,   p.."#showNodeOpen",   "Node visible when OPEN")
    schema:register(XMLValueType.NODE_INDEX,   p.."#showNodeClosed", "Node visible when CLOSED")

    schema:register(XMLValueType.FLOAT,        p.."#openSec",    "Open duration, seconds", 1.0)
    schema:register(XMLValueType.FLOAT,        p.."#closeSec",   "Close duration, seconds", 1.0)
    schema:register(XMLValueType.INT,          p.."#openAtMin",  "Auto open at game minute (0..1439)", -1)
    schema:register(XMLValueType.INT,          p.."#closeAtMin", "Auto close at game minute (0..1439)", -1)
    schema:register(XMLValueType.STRING,       p.."#easing",     "linear|smooth", "linear")
    schema:register(XMLValueType.STRING,       p.."#startState", "closed|open", "closed")
end

function PlaceableNodeAnimator.registerSavegameXMLPaths(schema, basePath)
    schema:register(XMLValueType.BOOL,  basePath..".nodeAnimator#present",          "Marker")
    schema:register(XMLValueType.BOOL,  basePath..".nodeAnimator.item(?)#isOpen",   "Current state")
    schema:register(XMLValueType.FLOAT, basePath..".nodeAnimator.item(?)#progress", "Current progress")
end

-- ======================================================================

local function _rad(x) return x * 0.017453292519943 end
local function _deg3_to_rad3(v) return { _rad(v[1] or 0), _rad(v[2] or 0), _rad(v[3] or 0) } end
local function _lerp(a,b,t) return a + (b-a)*t end
local function _mix3(a,b,t) return { _lerp(a[1],b[1],t), _lerp(a[2],b[2],t), _lerp(a[3],b[3],t) } end
local function _smoothstep(t) return t*t*(3-2*t) end

local function _applyNode(item, t)
    local node = item.node
    if node ~= nil then
        if item.translate then
            local pos = _mix3(item.posClosed, item.posOpen, t)
            setTranslation(node, pos[1], pos[2], pos[3])
        end
        if item.rotate then
            local rot = _mix3(item.rotClosed, item.rotOpen, t)
            setRotation(node, rot[1], rot[2], rot[3])
        end
    end

    if item.showNodeOpen ~= nil or item.showNodeClosed ~= nil then
        local openVis, closedVis
        if t <= 0.001 then
            openVis, closedVis = false, true
        elseif t >= 0.999 then
            openVis, closedVis = true, false
        else
            openVis = (t >= 0.5)
            closedVis = not openVis
        end
        if item.showNodeOpen   ~= nil then setVisibility(item.showNodeOpen,   openVis and true or false) end
        if item.showNodeClosed ~= nil then setVisibility(item.showNodeClosed, closedVis and true or false) end
    end
end

local function _getGameMinutes()
    local env = g_currentMission and g_currentMission.environment
    if env and env.getMinuteOfDay then
        return env:getMinuteOfDay() % 1440
    end
    local dayTimeMs = (env and env.dayTime) or 0
    return math.floor(dayTimeMs / 60000) % 1440
end

local function _isInOpenWindow(now, openAt, closeAt)
    if openAt == nil or closeAt == nil or openAt < 0 or closeAt < 0 then
        return false
    end
    if openAt == closeAt then
        return false
    end
    if openAt < closeAt then
        return now >= openAt and now < closeAt
    else
        return (now >= openAt) or (now < closeAt)
    end
end

-- ======================================================================

function PlaceableNodeAnimator:onLoad(savegame)
    local spec = {}
    self.spec_nodeAnimator = spec
    spec.items = {}
    spec.consolePrefix = self.xmlFile:getValue("placeable.nodeAnimator#consolePrefix", "NodeAnim")

    self.xmlFile:iterate("placeable.nodeAnimator.item", function(ix, key)
        local node = self.xmlFile:getValue(key.."#node", nil, self.components, self.i3dMappings)
        if node == nil then
            Logging.xmlWarning(self.xmlFile, "[NodeAnimator] Missing node at %s", key)
            return
        end

        local posClosed = self.xmlFile:getValue(key.."#posClosed", {0,0,0}, true)
        local posOpen   = self.xmlFile:getValue(key.."#posOpen",   {0,0,0}, true)

        local rotClosedDeg = self.xmlFile:getValue(key.."#rotClosed", {0,0,0}, true)
        local rotOpenDeg   = self.xmlFile:getValue(key.."#rotOpen",   {0,0,0}, true)
        local rotClosed = _deg3_to_rad3(rotClosedDeg)
        local rotOpen   = _deg3_to_rad3(rotOpenDeg)

        local translate = self.xmlFile:getValue(key.."#translate", true)
        local rotate    = self.xmlFile:getValue(key.."#rotate",    true)

        local openSec   = math.max(0.001, self.xmlFile:getValue(key.."#openSec",  1.0))
        local closeSec  = math.max(0.001, self.xmlFile:getValue(key.."#closeSec", 1.0))

        local openAtMin  = self.xmlFile:getValue(key.."#openAtMin",  -1)
        local closeAtMin = self.xmlFile:getValue(key.."#closeAtMin", -1)

        local easingStr    = (self.xmlFile:getValue(key.."#easing", "linear") or "linear"):lower()
        local easingSmooth = (easingStr == "smooth")

        local startState = (self.xmlFile:getValue(key.."#startState", "closed") or "closed"):lower()
        local isOpen = (startState == "open")

        local showNodeOpen   = self.xmlFile:getValue(key.."#showNodeOpen",   nil, self.components, self.i3dMappings)
        local showNodeClosed = self.xmlFile:getValue(key.."#showNodeClosed", nil, self.components, self.i3dMappings)

        local item = {
            node = node,
            translate = translate,
            rotate = rotate,

            posClosed = posClosed, posOpen = posOpen,
            rotClosed = rotClosed, rotOpen = rotOpen,

            showNodeOpen = showNodeOpen,
            showNodeClosed = showNodeClosed,

            openSec = openSec, closeSec = closeSec,
            openAtMin = openAtMin, closeAtMin = closeAtMin,
            easingSmooth = easingSmooth,

            isOpen = isOpen,
            targetOpen = isOpen,
            progress = isOpen and 1.0 or 0.0,
            lastMinute = -1
        }
        table.insert(spec.items, item)

        local t0 = item.easingSmooth and _smoothstep(item.progress) or item.progress
        _applyNode(item, t0)
    end)

    local nowMin = _getGameMinutes()
    for _, it in ipairs(spec.items) do
        if it.openAtMin and it.closeAtMin and it.openAtMin >= 0 and it.closeAtMin >= 0 then
            local shouldOpen = _isInOpenWindow(nowMin, it.openAtMin, it.closeAtMin)
            it.isOpen = shouldOpen
            it.targetOpen = shouldOpen
            it.progress = shouldOpen and 1.0 or 0.0
            local t = it.easingSmooth and _smoothstep(it.progress) or it.progress
            _applyNode(it, t)
        end
    end

    if self.isClient then
        self:raiseActive()
    end

    -- console helpers
    if g_consoleCommandManager and self.isClient then
        local pref = spec.consolePrefix
        addConsoleCommand("gs"..pref.."Open",   "Open animator by index",  function(_,i) i=tonumber(i or 1) self:setNodeAnimatorTarget(i, true,  true,  false) end)
        addConsoleCommand("gs"..pref.."Close",  "Close animator by index", function(_,i) i=tonumber(i or 1) self:setNodeAnimatorTarget(i, false, true,  false) end)
        addConsoleCommand("gs"..pref.."Toggle", "Toggle animator by index",function(_,i) i=tonumber(i or 1) self:toggleNodeAnimator(i,                    false) end)
    end
end

function PlaceableNodeAnimator:onPostLoad(savegame)
end

function PlaceableNodeAnimator:onDelete()
    local spec = self.spec_nodeAnimator
    if g_consoleCommandManager and spec and spec.consolePrefix and self.isClient then
        local pref = spec.consolePrefix
        removeConsoleCommand("gs"..pref.."Open")
        removeConsoleCommand("gs"..pref.."Close")
        removeConsoleCommand("gs"..pref.."Toggle")
    end
end

function PlaceableNodeAnimator:onFinalizePlacement()
    if self.isClient then
        self:raiseActive()
    end
end

function PlaceableNodeAnimator:onReadStream(streamId, connection)
    local spec = self.spec_nodeAnimator
    local count = streamReadUInt8(streamId)
    for i=1,count do
        local isOpen   = streamReadBool(streamId)
        local progress = streamReadFloat32(streamId)
        local it = spec.items[i]
        if it then
            it.isOpen = isOpen
            it.targetOpen = isOpen
            it.progress = progress
        end
    end
end

function PlaceableNodeAnimator:onWriteStream(streamId, connection)
    local spec = self.spec_nodeAnimator
    local count = math.min(#spec.items, 255)
    streamWriteUInt8(streamId, count)
    for i=1,count do
        local it = spec.items[i]
        streamWriteBool(streamId, it.isOpen)
        streamWriteFloat32(streamId, it.progress or 0)
    end
end

-- API

function PlaceableNodeAnimator:setNodeAnimatorTarget(index, wantOpen, resetProgress, noEventSend)
    local spec = self.spec_nodeAnimator
    local it = spec and spec.items and spec.items[index]
    if not it then return end

    if self.isServer then
        if noEventSend ~= true then
            g_server:broadcastEvent(NodeAnimatorSetStateEvent.new(self, index, wantOpen, resetProgress), true, nil, self)
        end
    else
        if noEventSend ~= true then
            g_client:getServerConnection():sendEvent(NodeAnimatorSetStateEvent.new(self, index, wantOpen, resetProgress))
        end
    end

    it.targetOpen = wantOpen and true or false
    self:raiseActive()
end

function PlaceableNodeAnimator:toggleNodeAnimator(index, noEventSend)
    local spec = self.spec_nodeAnimator
    local it = spec and spec.items and spec.items[index]
    if not it then return end
    self:setNodeAnimatorTarget(index, not it.targetOpen, false, noEventSend)
end

-- Save/Load

function PlaceableNodeAnimator:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self.spec_nodeAnimator
    if not spec or not spec.items or #spec.items == 0 then return end
    xmlFile:setValue(key..".nodeAnimator#present", true)
    for i,it in ipairs(spec.items) do
        local k = string.format("%s.nodeAnimator.item(%d)", key, i-1)
        xmlFile:setValue(k.."#isOpen", it.isOpen)
        xmlFile:setValue(k.."#progress", it.progress or 0)
    end
end

function PlaceableNodeAnimator:loadFromXMLFile(xmlFile, key)
    local spec = self.spec_nodeAnimator
    if not spec or not spec.items then return end
    if not xmlFile:getValue(key..".nodeAnimator#present") then return end

    for i,it in ipairs(spec.items) do
        local k = string.format("%s.nodeAnimator.item(%d)", key, i-1)

        local isOpen   = xmlFile:getValue(k.."#isOpen")
        local progress = xmlFile:getValue(k.."#progress")

        if isOpen ~= nil then
            it.isOpen = (isOpen == true) or (tonumber(isOpen) == 1)
            it.targetOpen = it.isOpen
        end

        if progress ~= nil then
            it.progress = _clamp01(progress)
        end

        local p = it.progress ~= nil and it.progress or (it.isOpen and 1 or 0)
        local t = it.easingSmooth and _smoothstep(p) or p
        _applyNode(it, t)
    end
end

-- ======================================================================

function PlaceableNodeAnimator:onUpdate(dt)
    local spec = self.spec_nodeAnimator
    if not spec or not spec.items or #spec.items == 0 then return end

    if self.isClient then
        self:raiseActive()
    end

    local nowMin = _getGameMinutes()
    for i,it in ipairs(spec.items) do
        local hasSchedule = (it.openAtMin and it.openAtMin >= 0) and (it.closeAtMin and it.closeAtMin >= 0)
        if hasSchedule then
            local wantOpen = _isInOpenWindow(nowMin, it.openAtMin, it.closeAtMin)
            if wantOpen ~= it.targetOpen then
                self:setNodeAnimatorTarget(i, wantOpen, false, false)
            end
        else
            if it.openAtMin and it.openAtMin >= 0 then
                if it.lastMinute ~= nowMin and nowMin == it.openAtMin then
                    self:setNodeAnimatorTarget(i, true,  false, false)
                end
            end
            if it.closeAtMin and it.closeAtMin >= 0 then
                if it.lastMinute ~= nowMin and nowMin == it.closeAtMin then
                    self:setNodeAnimatorTarget(i, false, false, false)
                end
            end
        end
        it.lastMinute = nowMin
    end

    local anyActive = false
    for _,it in ipairs(spec.items) do
        local target = it.targetOpen and 1 or 0
        local cur = it.progress or 0
        if math.abs(target - cur) > 0.0001 then
            anyActive = true
            local dur = (it.targetOpen and it.openSec) or it.closeSec or 1.0
            local step = (dt / 1000.0) / math.max(0.001, dur)
            if target > cur then
                cur = math.min(1.0, cur + step)
            else
                cur = math.max(0.0, cur - step)
            end
            it.progress = cur

            local t = it.easingSmooth and _smoothstep(cur) or cur
            _applyNode(it, t)

            if cur == 0.0 or cur == 1.0 then
                it.isOpen = (cur == 1.0)
            end
        end
    end

    if anyActive then
        self:raiseActive()
    end
end
