-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 18.11.2025 (исправлена ошибка компиляции + MP Eat/Take)

IngredientCookerHUD = {}
local IngredientCookerHUD_mt = Class(IngredientCookerHUD, MessageDialog)

IngredientCookerHUD.modDirectory     = g_currentModDirectory or ""
IngredientCookerHUD.consoleRegistered = false
IngredientCookerHUD.INSTANCE = nil

-- ===== helpers =====
local function _l10n(s)
    if not s or s == "" then return "" end
    if s:sub(1,1) ~= "$" then return s end
    local k = s:sub(2)
    if g_i18n and g_i18n:hasText(k) then return g_i18n:getText(k) end
    if k:sub(1,5) == "l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then
        return g_i18n:getText(k:sub(6))
    end
    return s
end

local function _invCounts()
    local c = {}
    if not Inventory or not Inventory.items then return c end
    for i=1, Inventory.maxSlots do
        local it = Inventory.items[i]
        if it and it.id and it.id ~= "" then
            c[it.id] = (c[it.id] or 0) + 1
        end
    end
    return c
end

local function _ingName(id)
    if not id or id == "" then return "?" end
    local tail = id:match("[^%.]+$") or id
    local key  = "l10n_ing_" .. tail
    if g_i18n and g_i18n:hasText(key) then return g_i18n:getText(key) end
    return _l10n("$" .. key)
end

-- ===== helpers: baseDir для LiveFarmer =====
local function _getLiveFarmerBaseDir_PIC()
    local candidates = {}

    if PlaceableIngredientCooker and PlaceableIngredientCooker.modDirectory and PlaceableIngredientCooker.modDirectory ~= "" then
        table.insert(candidates, PlaceableIngredientCooker.modDirectory)
    end
    if IngredientCookerHUD.modDirectory and IngredientCookerHUD.modDirectory ~= "" then
        table.insert(candidates, IngredientCookerHUD.modDirectory)
    end

    if g_modManager ~= nil then
        if g_modManager.getModByName ~= nil then
            local m = g_modManager:getModByName("FS25_liveFarmer")
            if m ~= nil then
                local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                if dir and dir ~= "" then
                    table.insert(candidates, dir)
                end
            end
        end

        if g_modManager.mods ~= nil then
            for _, m in pairs(g_modManager.mods) do
                local name     = m.modName or m.name or ""
                local fileName = m.modFileName or ""
                if name == "FS25_liveFarmer" or fileName:lower():find("fs25_livefarmer") ~= nil then
                    local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                    if dir and dir ~= "" then
                        table.insert(candidates, dir)
                    end
                end
            end
        end
    end

    if debug and debug.getinfo then
        local info = debug.getinfo(_getLiveFarmerBaseDir_PIC, "S")
        if info and info.source then
            local src = info.source
            if src:sub(1,1) == "@" then src = src:sub(2) end
            src = src:gsub("\\", "/")
            local idx = src:lower():find("/fs25_livefarmer/")
            if idx then
                local dir = src:sub(1, idx + string.len("/FS25_liveFarmer/") - 1)
                table.insert(candidates, dir)
            end
        end
    end

    if g_modsDirectory ~= nil then
        table.insert(candidates, g_modsDirectory .. "/FS25_liveFarmer/")
    end

    local tried = {}
    for _, baseDir in ipairs(candidates) do
        if type(baseDir) == "string" and baseDir ~= "" then
            baseDir = baseDir:gsub("\\", "/")
            if baseDir:sub(-1) ~= "/" then baseDir = baseDir .. "/" end
            if not tried[baseDir] then
                tried[baseDir] = true
                local testPath = Utils.getFilename("gui/IngredientCookerHUD.xml", baseDir)
                if testPath and fileExists(testPath) then
                    return baseDir
                end
            end
        end
    end

    return IngredientCookerHUD.modDirectory or (g_currentModDirectory or "")
end

function IngredientCookerHUD.register()
    if not g_gui then return false end
    IngredientCookerHUD.registerConsole()
    if g_gui.guis and g_gui.guis["IngredientCookerHUD"] then
        IngredientCookerHUD.INSTANCE = g_gui.guis["IngredientCookerHUD"].target
        return true
    end

    local inst = IngredientCookerHUD.new()

    local baseDir = _getLiveFarmerBaseDir_PIC()
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/IngredientCookerHUD.xml", baseDir)
    local ok = false

    if xmlPath and xmlPath ~= "" then
        ok = pcall(function()
            g_gui:loadGui(xmlPath, "IngredientCookerHUD", inst)
        end)
    end

    if (not ok) or not (g_gui.guis and g_gui.guis["IngredientCookerHUD"]) then
        local path1 = baseDir .. "gui/IngredientCookerHUD.xml"
        local path2 = baseDir .. "IngredientCookerHUD.xml"

        for _, p in ipairs({path1, path2}) do
            if p and p ~= "" then
                local loaded = pcall(function()
                    g_gui:loadGui(p, "IngredientCookerHUD", inst)
                end)
                if loaded and g_gui.guis and g_gui.guis["IngredientCookerHUD"] then
                    ok = true
                    xmlPath = p
                    break
                end
            end
        end
    end

    if ok and g_gui.guis and g_gui.guis["IngredientCookerHUD"] then
        IngredientCookerHUD.INSTANCE = g_gui.guis["IngredientCookerHUD"].target
        return true
    else
        Logging.error("[PIC HUD] XML not found (baseDir=%s, xmlPath=%s)", tostring(baseDir), tostring(xmlPath))
        pcall(function() inst:delete() end)
        return false
    end
end

function IngredientCookerHUD.new(target, custom_mt)
    local self = MessageDialog.new(target, custom_mt or IngredientCookerHUD_mt)
    self.placeable     = nil
    self.items         = {}
    self.selectedIndex = 0

    self.itemsTable    = nil
    self.itemsSlider   = nil
    self.btnCook       = nil
    self.btnEat        = nil
    self.btnTake       = nil
    self.btnDiscard    = nil
    self.titleEl       = nil
    self.descTitleEl   = nil
    self.descTextEl    = nil
    self.statusText    = nil
    self.recipeTextEl  = nil

    self._tick         = 0
    self._finalizeSent = false
    return self
end

-------------------------------------------------------------------------

function IngredientCookerHUD.registerConsole()
    if IngredientCookerHUD.consoleRegistered then return end
    if addConsoleCommand ~= nil then
        addConsoleCommand("gsReloadIngredientCookerHUD", "Reload IngredientCooker HUD xml (close/reopen if open)", "reloadHUD", IngredientCookerHUD)
        addConsoleCommand("gsReloadPICHUD", "Reload IngredientCooker HUD xml (alias)", "reloadHUD", IngredientCookerHUD)
        IngredientCookerHUD.consoleRegistered = true
        print("[PIC HUD] Console command registered: gsReloadIngredientCookerHUD")
    end
end

function IngredientCookerHUD.reloadHUD()
    if not g_gui then print("[PIC HUD] g_gui is not ready") return end

    local baseDir = _getLiveFarmerBaseDir_PIC and _getLiveFarmerBaseDir_PIC() or (IngredientCookerHUD.modDirectory or g_currentModDirectory or "")
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then baseDir = baseDir .. "/" end

    local xmlPath = Utils.getFilename("gui/IngredientCookerHUD.xml", baseDir)
    if not (xmlPath and fileExists(xmlPath)) then
        print("[PIC HUD] HUD XML not found to reload (baseDir=" .. tostring(baseDir) .. ")")
        return
    end

    local keepPlaceable, wasOpen = nil, false
    if IngredientCookerHUD.INSTANCE then keepPlaceable = IngredientCookerHUD.INSTANCE.placeable end

    if g_gui.closeDialogByName then
        if g_gui.guis and g_gui.guis["IngredientCookerHUD"] then wasOpen = true end
        pcall(function() g_gui:closeDialogByName("IngredientCookerHUD") end)
    elseif g_gui.currentGuiName == "IngredientCookerHUD" then
        wasOpen = true
        pcall(function() g_gui:showGui("") end)
    end

    if g_gui.guis and g_gui.guis["IngredientCookerHUD"] then
        pcall(function() g_gui:unloadGui("IngredientCookerHUD") end)
        g_gui.guis["IngredientCookerHUD"] = nil
    end
    if IngredientCookerHUD.INSTANCE then
        pcall(function() IngredientCookerHUD.INSTANCE:delete() end)
        IngredientCookerHUD.INSTANCE = nil
    end

    local ok = IngredientCookerHUD.register()
    print("[PIC HUD] Reload result: " .. tostring(ok))

    if ok and wasOpen and keepPlaceable and IngredientCookerHUD.INSTANCE then
        IngredientCookerHUD.INSTANCE.placeable = keepPlaceable
        if g_gui.showDialog then g_gui:showDialog("IngredientCookerHUD")
        else g_gui:showGui("IngredientCookerHUD") end
    end
end

-- ===== lifecycle =====
function IngredientCookerHUD:onOpen()
    local sc = self.superClass and self:superClass()
    if sc and sc.onOpen then
        sc.onOpen(self)
    end

    if g_server == nil and g_client ~= nil
        and self.placeable ~= nil
        and PIC_StateSyncRequestEvent ~= nil then

        local conn = g_client:getServerConnection()
        if conn ~= nil then
            conn:sendEvent(PIC_StateSyncRequestEvent.new(self.placeable))
        end
    end

    ----------------------------------------------------------------
    -- инициализация элементов GUI
    ----------------------------------------------------------------
    if g_inputBinding and g_inputBinding.setShowMouseCursor then
        g_inputBinding:setShowMouseCursor(true)
    end

    self.itemsTable   = self:getDescendantById("itemsTable")
    self.itemsSlider  = self:getDescendantById("itemsTableSlider")
    self.btnCook      = self:getDescendantById("cookButton")
    self.btnEat       = self:getDescendantById("eatButton")
    self.btnTake      = self:getDescendantById("takeButton")
    self.btnDiscard   = self:getDescendantById("discardButton")
    self.titleEl      = self:getDescendantById("dialogTitleElement")
    self.descTitleEl  = self:getDescendantById("itemDescTitle")
    self.descTextEl   = self:getDescendantById("itemDesc")
    self.statusText   = self:getDescendantById("itemStatus")
    self.recipeTextEl = self:getDescendantById("itemRecipe")

    local old = self:getDescendantById("statusText")
    if old and old.setVisible then
        old:setVisible(false)
    end

    FocusManager:setFocus(self.itemsTable)

    self:pullData()

    if self.selectedIndex == 0 and #self.items > 0 then
        self.selectedIndex = 1
    end

    if self.itemsTable and #self.items > 0 then
        self.itemsTable:reloadData(true)
        self.itemsTable:setSelectedIndex(self.selectedIndex, true, true)
    end

    self:updateButtons()
    self:updateStatus()
    self:updateRecipeInfo()

    self._tick         = 0
    self._finalizeSent = false

    local spec = self.placeable and self.placeable.spec_ingredientCooker
    if spec then
        self._lastReadyCount = spec.readyCount
            or (spec.readyItems and #spec.readyItems)
            or 0
    else
        self._lastReadyCount = 0
    end
end

function IngredientCookerHUD:onClose()
    if g_inputBinding and g_inputBinding.setShowMouseCursor then
        g_inputBinding:setShowMouseCursor(false)
    end
    local sc = self.superClass and self:superClass()
    if sc and sc.onClose then sc.onClose(self) end
end

function IngredientCookerHUD:update(dt)
    local sc = self.superClass and self:superClass()
    if sc and sc.update then
        sc.update(self, dt)
    end

    local spec = self.placeable and self.placeable.spec_ingredientCooker
    if not spec then
        return
    end

    self._tick = (self._tick or 0) - (dt or 0)
    if self._tick <= 0 then
        self._tick = 250

        if spec.state and spec.state.cooking then
            local remainMs = math.max(0, (spec.state.endTimeMs or 0) - (g_time or 0))

            if remainMs <= 0 and not self._finalizeSent then
                self._finalizeSent = true

                if g_client ~= nil then
                    local conn = g_client:getServerConnection()
                    if conn ~= nil then
                        conn:sendEvent(PIC_FinalizeEvent.new(self.placeable))
                    end
                else
                    if self.placeable.picServerFinalizeIfReady then
                        self.placeable:picServerFinalizeIfReady()
                    end
                end
            elseif remainMs > 0 then
                self._finalizeSent = false
            end
        else
            self._finalizeSent = false
        end

        local curReady = spec.readyCount
            or (spec.readyItems and #spec.readyItems)
            or 0

        if self._lastReadyCount ~= curReady then
            self._lastReadyCount = curReady

            self:pullData()

            if self.itemsTable then
                self.itemsTable:reloadData(true)

                if (self.selectedIndex or 0) == 0 and #self.items > 0 then
                    self.selectedIndex = 1
                end

                if self.itemsTable.setSelectedIndex and self.selectedIndex > 0 then
                    self.itemsTable:setSelectedIndex(self.selectedIndex, true, true)
                end
            end

            self:updateStatus()
            self:updateButtons()
            self:updateRecipeInfo()
        end
    end
end


function IngredientCookerHUD:pullData()
    self.items = {}
    local spec = self.placeable and self.placeable.spec_ingredientCooker
    if not spec then return end

    if self.titleEl then
        self.titleEl:setText(spec.title or _l10n("$ui_pic_title"))
    end

    local firstReadySlotMap = {}
    for slotIdx, r in ipairs(spec.readyItems or {}) do
        if r.index and not firstReadySlotMap[r.index] then
            firstReadySlotMap[r.index] = slotIdx
        end
    end

    local counts = _invCounts()

    for i, it in ipairs(spec.items or {}) do
        local miss, parts = {}, {}
        for _, ing in ipairs(it.recipe or {}) do
            local have = counts[ing.id] or 0
            local need = math.max(1, ing.count or 1)
            local seg = string.format("%s ×%d (%d/%d)", _ingName(ing.id), need, have, need)
            if have >= need then
                table.insert(parts, seg)
            else
                table.insert(miss, seg)
            end
        end
        local recipeStr = table.concat(parts, ", ")
        if #miss > 0 then
            recipeStr = recipeStr .. (recipeStr ~= "" and " | " or "") .. _l10n("$ui_pic_missing") .. ": " .. table.concat(miss, ", ")
        end

        local readyCount = spec.readyCounts[i] or 0
        local cap = it.readyCap or 0
        local capText = cap > 0 and string.format("%d/%d", readyCount, cap) or tostring(readyCount)

        local gainText = ""
        if (it.hungerGain or 0) > 0 or (it.vigorGain or 0) > 0 then
            gainText = string.format(" (+%d сытость, +%d бодрость)", it.hungerGain or 0, it.vigorGain or 0)
        end

        table.insert(self.items, {
            index          = i,
            name           = _l10n(it.name),
            desc           = _l10n(it.desc),
            icon           = it.icon,
            recipeStr      = recipeStr,
            hasMissing     = #miss > 0,
            canCook        = #miss == 0,
            readyCount     = readyCount,
            firstReadySlot = firstReadySlotMap[i] or 0,
            displayTxt     = string.format("%s — %ds — %s%s", _l10n(it.name), it.cookSec or 0, capText, gainText)
        })
    end

    if self.itemsTable then
        self.itemsTable:reloadData(true)
    end

    local sel = self.items[self.selectedIndex]
    if sel then
        if self.descTitleEl then self.descTitleEl:setText(sel.name or "") end
        if self.descTextEl  then self.descTextEl:setText(sel.desc or "") end
    end

    self:updateRecipeInfo()
    self:updateButtons()
    self:updateStatus()
end

function IngredientCookerHUD:updateButtons()
    local spec = self.placeable and self.placeable.spec_ingredientCooker
    if not spec then return end

    local cooking = spec.state and spec.state.cooking
    local sel = self.items[self.selectedIndex]

    local canCook = sel and sel.canCook and not cooking
    local hasReady = sel and sel.firstReadySlot > 0

    if self.btnCook     then self.btnCook:setDisabled(not canCook) end
    if self.btnEat      then self.btnEat:setDisabled(not hasReady or cooking) end
    if self.btnTake     then self.btnTake:setDisabled(not hasReady or cooking) end
    if self.btnDiscard  then self.btnDiscard:setDisabled(not hasReady or cooking) end
end

function IngredientCookerHUD:updateStatus()
    local spec = self.placeable and self.placeable.spec_ingredientCooker
    if not spec then return end

    local statusStr
    if spec.state.cooking then
        local leftMs = math.max(0, (spec.state.endTimeMs or 0) - (g_time or 0))
        statusStr = string.format("%s: %ds",
            (g_i18n and g_i18n:getText("ui_pic_status_cooking")) or "Готовится",
            math.floor(leftMs / 1000))
    elseif (spec.readyCount or 0) > 0 then
        statusStr = string.format("%s (x%d)",
            (g_i18n and g_i18n:getText("ui_pic_status_ready")) or "Готово — можно съесть или забрать",
            spec.readyCount or 1)
    else
        statusStr = (g_i18n and g_i18n:getText("ui_pic_status_idle")) or "Ожидает начала готовки"
    end

    if self.statusText and self.statusText.setText then
        self.statusText:setText(statusStr)
    end

    self:updateButtons()
end

function IngredientCookerHUD:getNumberOfSections(list) return 1 end
function IngredientCookerHUD:getNumberOfItemsInSection(list, section) return #self.items end
function IngredientCookerHUD:getCellTypeForItemInSection(list, section, index) return "rowTemplate" end

function IngredientCookerHUD:populateCellForItemInSection(list, section, index, cell)
    local it = self.items[index]; if not it then return end
    local textEl = cell:getDescendantByName("rowText")
    local iconEl = cell:getDescendantByName("rowIcon")

    if textEl then textEl:setText(it.displayTxt or "") end
    if iconEl then
        if it.icon and fileExists(it.icon) then
            iconEl:setVisible(true)
            iconEl:setImageFilename(it.icon)
        else
            iconEl:setVisible(false)
        end
    end
end

function IngredientCookerHUD:onListSelectionChanged(list, section, index)
    self.selectedIndex = index or 0
    local it = self.items[self.selectedIndex]
    if it then
        if self.descTitleEl then self.descTitleEl:setText(it.name or "") end
        if self.descTextEl  then self.descTextEl:setText(it.desc or "") end
    else
        if self.descTitleEl then self.descTitleEl:setText("") end
        if self.descTextEl  then self.descTextEl:setText("") end
    end
    self:updateRecipeInfo()
    self:updateStatus()
end

function IngredientCookerHUD:updateRecipeInfo()
    if not self.recipeTextEl then return end
    local it = self.items[self.selectedIndex]
    if not it then
        self.recipeTextEl:setText("")
        return
    end

    local label = _l10n("$ui_pic_recipe")
    local txt = (label ~= "" and (label .. ": ") or "") .. (it.recipeStr or "")
    self.recipeTextEl:setText(txt)

    if self.recipeTextEl.setTextColor then
        if it.hasMissing then
            self.recipeTextEl:setTextColor(1.0, 0.65, 0.65, 1.0)
        else
            self.recipeTextEl:setTextColor(0.85, 0.95, 0.85, 1.0)
        end
    end
end

-- ===== buttons =====
function IngredientCookerHUD:onClickCook()
    if not self.placeable then return end

    local sel = self.items[self.selectedIndex]
    if not sel or not sel.canCook then return end

    local idx = sel.index
    local spec = self.placeable.spec_ingredientCooker
    local recipe = spec.items[idx]
    if not recipe then return end

    if not Inventory or not Inventory.items or not Inventory.maxSlots then
        g_currentMission:showBlinkingWarning("Ошибка инвентаря", 3000)
        return
    end

    local counts = {}
    for i = 1, Inventory.maxSlots do
        local item = Inventory.items[i]
        if item and item.id then
            counts[item.id] = (counts[item.id] or 0) + 1
        end
    end

    for _, ing in ipairs(recipe.recipe or {}) do
        local need = math.max(1, ing.count or 1)
        if (counts[ing.id] or 0) < need then
            g_currentMission:showBlinkingWarning("Не хватает: " .. _ingName(ing.id), 3000)
            return
        end
    end

    local function removeOne(id)
        for i = 1, Inventory.maxSlots do
            if Inventory.items[i] and Inventory.items[i].id == id then
                Inventory.items[i] = nil
                return true
            end
        end
        return false
    end

    for _, ing in ipairs(recipe.recipe or {}) do
        local n = math.max(1, ing.count or 1)
        for i = 1, n do
            if not removeOne(ing.id) then
                g_currentMission:showBlinkingWarning("Ошибка списания ингредиентов", 3000)
                return
            end
        end
    end

    self:pullData()
    self:updateButtons()

    if g_client then
        g_client:getServerConnection():sendEvent(PIC_StartEvent.new(self.placeable, idx))
    else
        self.placeable:picServerStart(idx)
    end
end

function IngredientCookerHUD:onClickTake()
    if not self.placeable then return end
    local sel = self.items[self.selectedIndex]
    if not sel or sel.firstReadySlot < 1 then return end

    local slot = sel.firstReadySlot
    if g_client then
        g_client:getServerConnection():sendEvent(PIC_TakeEvent.new(self.placeable, slot))
    else
        self.placeable:picServerTakeToInventory(nil, slot)
    end
end

function IngredientCookerHUD:onClickEat()
    if not self.placeable then return end
    local sel = self.items[self.selectedIndex]
    if not sel or sel.firstReadySlot < 1 then return end

    local slot = sel.firstReadySlot
    if g_client then
        g_client:getServerConnection():sendEvent(PIC_ConsumeEvent.new(self.placeable, slot))
    else
        self.placeable:picServerConsumeReady(nil, slot)
    end
end

function IngredientCookerHUD:onClickDiscard()
    if not self.placeable then return end
    local sel = self.items[self.selectedIndex]
    if not sel or sel.firstReadySlot < 1 then return end

    local slot = sel.firstReadySlot
    if g_client then
        g_client:getServerConnection():sendEvent(PIC_DiscardEvent.new(self.placeable, slot))
    else
        self.placeable:picServerDiscard(nil, slot)
    end
end

function IngredientCookerHUD:close()
    g_gui:showGui("")
end