-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

PlaceableFoodVendor = {}
PlaceableFoodVendor.modName      = g_currentModName or "HungerSystem"
PlaceableFoodVendor.modDirectory = g_currentModDirectory or ""

local function FV_clamp(v, minV, maxV)
    v = tonumber(v) or 0
    if v < minV then return minV end
    if v > maxV then return maxV end
    return v
end

function PlaceableFoodVendor.prerequisitesPresent(_)
    return true
end

function PlaceableFoodVendor.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.STRING, basePath .. ".foodVendor#title", "HUD title (l10n or plain)")
    schema:register(XMLValueType.STRING, basePath .. ".foodVendor#effectsFile", "Item effects XML (relative path)")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".foodVendor#triggerNode", "Trigger node for vendor")

    local itemKey = basePath .. ".foodVendor.items.item(?)"
    schema:register(XMLValueType.STRING, itemKey .. "#name",       "Display name (l10n or plain)")
    schema:register(XMLValueType.FLOAT,  itemKey .. "#price",      "Price", 0)
    schema:register(XMLValueType.INT,    itemKey .. "#hungerGain", "Hunger +/- in %", 0)
    schema:register(XMLValueType.INT,    itemKey .. "#vigorGain",  "Vigor  +/- in %", 0)
    schema:register(XMLValueType.STRING, itemKey .. "#icon",       "Relative icon path")
    schema:register(XMLValueType.STRING, itemKey .. "#desc",       "Description")
	
    schema:register(XMLValueType.STRING, itemKey .. "#kind",       "food|effect", "food")
    schema:register(XMLValueType.STRING, itemKey .. "#effectId",   "Effect id from effectsFile")
end

function PlaceableFoodVendor.initSpecialization() end

function PlaceableFoodVendor.registerFunctions(placeableType)
    SpecializationUtil.registerFunction(placeableType, "foodVendorTriggerCallback", PlaceableFoodVendor.foodVendorTriggerCallback)
    SpecializationUtil.registerFunction(placeableType, "getIsAccessible",            PlaceableFoodVendor.getIsAccessible)
    SpecializationUtil.registerFunction(placeableType, "serverBuyFood",              PlaceableFoodVendor.serverBuyFood)
end

function PlaceableFoodVendor.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",              PlaceableFoodVendor)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",            PlaceableFoodVendor)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement", PlaceableFoodVendor)
end

local function _topNotify(title, msg, icon)
    local hud = g_currentMission and g_currentMission.hud
    if hud and hud.topNotification and hud.topNotification.setNotification then
        hud.topNotification:setNotification(title or "", msg or "", "", icon or "", nil)
    else
        print(("[FoodVendor] %s %s"):format(tostring(title or ""), tostring(msg or "")))
    end
end

local function _getCurrentHunger()
    if _G.HungerSystem and HungerSystem.hunger ~= nil then
        return tonumber(HungerSystem.hunger) or 100
    end
    local p = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
              or (g_currentMission and g_currentMission.player)
    if p and p.hunger ~= nil then
        return tonumber(p.hunger) or 100
    end
    return 100
end

local function _resolveIconPath(path)
    if not path or path == "" then return nil end
    if fileExists(path) then return path end
    if path:lower():sub(-4) == ".dds" then
        local alt = path:sub(1, -5) .. ".png"
        if fileExists(alt) then return alt end
    elseif path:lower():sub(-4) == ".png" then
        local alt = path:sub(1, -5) .. ".dds"
        if fileExists(alt) then return alt end
    end
    return nil
end

local function _resolveL10nStrict(raw)
    if raw == nil or raw == "" then return nil end
    local s = tostring(raw)
    local candidates = {}

    local function add(k) if k and k ~= "" then table.insert(candidates, k) end end

    if s:sub(1,1) == "$" then
        s = s:sub(2)                    
        add(s)
        if s:sub(1,5) == "l10n_"   then add(s:sub(6)) end   
        if s:sub(1,8) == "l10n_ui_" then add(s:sub(6)) end  
    else
        return s
    end

    for _, k in ipairs(candidates) do
        if g_i18n and g_i18n.hasText and g_i18n:hasText(k) then
            local txt = g_i18n:getText(k)
            if txt and txt ~= "" and not string.find(txt, "Missing '", 1, true) then
                return txt
            end
        end
    end

    local last = candidates[#candidates] or s
    last = last:gsub("^ui_", ""):gsub("_", " ")
    return last
end

--------------------------------------------------------------------------------
-- Жизненный цикл
--------------------------------------------------------------------------------
function PlaceableFoodVendor:onLoad(savegame)
    self.spec_foodVendor = self.spec_foodVendor or {}
    local spec = self.spec_foodVendor

    spec.farmId       = self:getOwnerFarmId()
    spec.triggerNode  = nil
    spec.items        = {}
    spec.guiLoadError = false

    local titleFromVendor = self.xmlFile:getValue("placeable.foodVendor#title")
    local fallbackTitle   = (g_i18n and g_i18n:getText("ui_foodVendor_title")) or "Кафе"
    spec.displayNameRaw   = titleFromVendor
    spec.displayName      = _resolveL10nStrict and _resolveL10nStrict(spec.displayNameRaw) or titleFromVendor or fallbackTitle
    if (spec.displayName or "") == "" then spec.displayName = fallbackTitle end

    -- эффекты
    do
        local effectsRel = self.xmlFile:getValue("placeable.foodVendor#effectsFile")
        if effectsRel ~= nil and effectsRel ~= "" then
            spec.effectsFileRel = effectsRel
            spec.effectsFileAbs = Utils.getFilename(effectsRel, PlaceableFoodVendor.modDirectory)
            if not fileExists(spec.effectsFileAbs) then
                Logging.warning("FoodVendor effectsFile not found: %s", tostring(spec.effectsFileAbs))
                spec.effectsFileAbs = nil
            end
        else
            spec.effectsFileRel = nil
            spec.effectsFileAbs = nil
        end
    end

    -- триггер
    local triggerNode = self.xmlFile:getValue("placeable.foodVendor#triggerNode", nil, self.components, self.i3dMappings)
    if triggerNode ~= nil then
        addTrigger(triggerNode, "foodVendorTriggerCallback", self)
        spec.triggerNode = triggerNode
    else
        Logging.xmlWarning(self.xmlFile, "FoodVendor trigger node not found or invalid")
    end

    -- активатор
    if not self.foodVendorActivatable then
        self.foodVendorActivatable = FoodVendorActivatable.new(self)
    end

    -- ===================== АССОРТИМЕНТ =====================
    local i = 0
    while true do
        local key = string.format("placeable.foodVendor.items.item(%d)", i)
        if not self.xmlFile:hasProperty(key) then break end

        local name       = self.xmlFile:getValue(key .. "#name") or ("Food "..tostring(i+1))
        local price      = self.xmlFile:getValue(key .. "#price", 100)
        local hungerGain = self.xmlFile:getValue(key .. "#hungerGain", 0)
        local vigorGain  = self.xmlFile:getValue(key .. "#vigorGain",  0)
        local iconRelXml = self.xmlFile:getValue(key .. "#icon")
        local desc       = self.xmlFile:getValue(key .. "#desc")
        local kind       = (self.xmlFile:getValue(key .. "#kind") or "food"):lower()
        local effectId   = self.xmlFile:getValue(key .. "#effectId")

        if effectId and effectId ~= "" then
            kind = "effect"
        end

        local iconRel = nil
        local iconAbs = nil
        if iconRelXml and iconRelXml ~= "" then
            iconRel = iconRelXml
            iconAbs = Utils.getFilename(iconRelXml, PlaceableFoodVendor.modDirectory)
            if not fileExists(iconAbs) then
                iconAbs = iconRelXml
            end
        end

        local item = {
            name       = name,
            price      = math.max(0, math.floor(price or 0)),
            hungerGain = math.floor(hungerGain or 0),
            vigorGain  = math.floor(vigorGain or 0),
            icon       = iconAbs,
            iconRel    = iconRel,
            desc       = desc,
            kind       = kind,
            effectId   = effectId
        }
        table.insert(spec.items, item)

        i = i + 1
    end

    if g_gui and (not (g_gui.guis and g_gui.guis["FoodVendorHUD"])) then
        if FoodVendorHUD and FoodVendorHUD.register then
            FoodVendorHUD.register()
        end
    end
end

FoodVendorActivatable = {}
local FoodVendorActivatable_mt = Class(FoodVendorActivatable)

function FoodVendorActivatable.new(placeable)
    local self = setmetatable({}, FoodVendorActivatable_mt)
    self.placeable    = placeable
    self.activateText = (g_i18n and g_i18n:getText("ui_foodVendor_pressOpen")) or (g_i18n and g_i18n:getText("action_activateShop")) or "Открыть"
    return self
end

function FoodVendorActivatable:getIsActivatable()
    if g_localPlayer == nil then return false end
    if g_localPlayer:getIsInVehicle() then return false end

    local farmId = g_currentMission:getFarmId()
    if not self.placeable:getIsAccessible(farmId) then return false end

    if g_gui then
        if g_gui.getIsGuiVisible and g_gui:getIsGuiVisible("FoodVendorHUD") then
            return false
        end
        if g_gui.getIsDialogVisible and g_gui:getIsDialogVisible("FoodVendorHUD") then
            return false
        end
    end
    return true
end

function FoodVendorActivatable:run()
    if g_gui and (not (g_gui.guis and g_gui.guis["FoodVendorHUD"])) then
        if FoodVendorHUD and FoodVendorHUD.register then
            local ok = FoodVendorHUD.register()
            if not ok then
                print("[FoodVendor] Cannot open HUD: register() failed (check FoodVendorHUD.xml path)")
                return
            end
        else
            print("[FoodVendor] Cannot open HUD: FoodVendorHUD.register is missing")
            return
        end
    end

    if not (FoodVendorHUD and FoodVendorHUD.INSTANCE) then
        if FoodVendorHUD and FoodVendorHUD.register then
            local ok = FoodVendorHUD.register()
            if not ok or not FoodVendorHUD.INSTANCE then
                print("[FoodVendor] Cannot open HUD: INSTANCE missing after register()")
                return
            end
        else
            print("[FoodVendor] Cannot open HUD: FoodVendorHUD not found")
            return
        end
    end

    FoodVendorHUD.INSTANCE.vendor = self.placeable
    g_gui:showDialog("FoodVendorHUD")
end

function PlaceableFoodVendor:onFinalizePlacement() end

function PlaceableFoodVendor:onDelete()
    local spec = self.spec_foodVendor
    if spec and spec.triggerNode then
        removeTrigger(spec.triggerNode)
        spec.triggerNode = nil
    end
    if self.foodVendorActivatable then
        g_currentMission.activatableObjectsSystem:removeActivatable(self.foodVendorActivatable)
        self.foodVendorActivatable = nil
    end
end


function PlaceableFoodVendor:getIsAccessible(farmId)
    return true
end

--------------------------------------------------------------------------------
-- Триггер → открыть HUD
--------------------------------------------------------------------------------

function PlaceableFoodVendor:foodVendorTriggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
    if not (onEnter or onLeave) then return end
    if g_localPlayer == nil then return end
    if otherActorId ~= g_localPlayer.rootNode then return end

    if onEnter then
        if Platform.gameplay.autoActivateTrigger and self.foodVendorActivatable:getIsActivatable() then
            self.foodVendorActivatable:run()
        else
            g_currentMission.activatableObjectsSystem:addActivatable(self.foodVendorActivatable)
        end
    end

    if onLeave then
        g_currentMission.activatableObjectsSystem:removeActivatable(self.foodVendorActivatable)
    end
end

--------------------------------------------------------------------------------
-- Покупка (сервер)
--------------------------------------------------------------------------------
function PlaceableFoodVendor:serverBuyFood(itemIndex, farmId)
    local spec = self.spec_foodVendor
    if spec == nil or spec.items == nil then
        return false, { error = "NO_ITEMS" }
    end

    local it = spec.items[itemIndex]
    if it == nil then
        return false, { error = "BAD_INDEX" }
    end

    ------------------------------------------------------------------
    -- 1) Цена и фермы (покупатель / владелец)
    ------------------------------------------------------------------
    local price = tonumber(it.price or 0) or 0
    if price < 0 then price = 0 end

    -- ферма-покупатель
    local buyerFarmId = farmId or 1
    local buyerFarm = g_farmManager and g_farmManager:getFarmById(buyerFarmId) or nil
    if buyerFarm == nil then
        return false, { error = "NO_FARM" }
    end

    -- ферма-владелец ресторана (плейсабл поставила эта ферма)
    local ownerFarmId = spec.farmId or self:getOwnerFarmId() or 0
    if ownerFarmId == nil then ownerFarmId = 0 end

    -- хватает ли денег у покупателя?
    if buyerFarm.getBalance and buyerFarm:getBalance() < price then
        return false, { error = "NO_MONEY" }
    end

    local moneyTypeOther = (MoneyType and MoneyType.OTHER) or 0

    if g_currentMission ~= nil and g_currentMission.addMoney ~= nil then
        g_currentMission:addMoney(-price, buyerFarmId, moneyTypeOther, true, true)
    elseif buyerFarm.changeBalance ~= nil then
        pcall(function()
            buyerFarm:changeBalance(-price, moneyTypeOther)
        end)
    else
        buyerFarm.money = (buyerFarm.money or 0) - price
    end

    ------------------------------------------------------------------
    -- 3) Комиссия владельцу ресторана
    --    Если покупаем в СВОЁМ ресторане (buyer == owner) – ничего не начисляем.
    ------------------------------------------------------------------
    if ownerFarmId ~= 0 and ownerFarmId ~= buyerFarmId then
        local ownerFarm = g_farmManager and g_farmManager:getFarmById(ownerFarmId)
        if ownerFarm ~= nil then
            local commission = math.floor(price * 0.25) -- 25% от стоимости
            if commission > 0 then
                if g_currentMission ~= nil and g_currentMission.addMoney ~= nil then
                    g_currentMission:addMoney(commission, ownerFarmId, moneyTypeOther, true, true)
                elseif ownerFarm.changeBalance ~= nil then
                    pcall(function()
                        ownerFarm:changeBalance(commission, moneyTypeOther)
                    end)
                else
                    ownerFarm.money = (ownerFarm.money or 0) + commission
                end
            end
        end
    end

    ------------------------------------------------------------------
    -- 4) Параметры сытости/бодрости
    ------------------------------------------------------------------
    local rawHunger = it.hungerGain or it.hunger or 0
    local rawVigor  = it.vigorGain  or it.vigor  or 0

    local hungerGain = FV_clamp(math.floor(tonumber(rawHunger) or 0), -100, 100)
    local vigorGain  = FV_clamp(math.floor(tonumber(rawVigor)  or 0), -100, 100)

    ------------------------------------------------------------------
    -- 5) Эффекты
    ------------------------------------------------------------------
    local effectId    = it.effectId or ""
    local effectsFile = spec.effectsFileAbs or spec.effectsFile or ""

    local iconForNet = it.iconRel or it.icon or ""

    local payload = {
        itemIndex   = itemIndex,
        price       = price,

        name        = it.name,
        desc        = it.desc or it.description,
        icon        = iconForNet,
        kind        = it.kind or "food",

        hungerGain  = hungerGain,
        vigorGain   = vigorGain,

        effectId    = effectId,
        effectsFile = effectsFile,

        placeableId = NetworkUtil.getObjectId(self)
    }

    print(("[FoodVendor] serverBuyFood idx=%d name=%s hungerGain=%d vigorGain=%d effectId=%s icon=%s buyerFarm=%s ownerFarm=%s")
        :format(itemIndex, tostring(it.name), hungerGain, vigorGain,
                tostring(effectId), tostring(iconForNet),
                tostring(buyerFarmId), tostring(ownerFarmId)))

    return true, payload
end

--------------------------------------------------------------------------------
-- Событие применения/получения еды/энергетика
--------------------------------------------------------------------------------
FoodVendorGainHungerEvent = {}
local FoodVendorGainHungerEvent_mt = Class(FoodVendorGainHungerEvent, Event)
InitEventClass(FoodVendorGainHungerEvent, "FoodVendorGainHungerEvent")

function FoodVendorGainHungerEvent.emptyNew()
    return Event.new(FoodVendorGainHungerEvent_mt)
end

function FoodVendorGainHungerEvent.new(payload)
    local self = FoodVendorGainHungerEvent.emptyNew()
    self.hungerGain = math.floor((payload and payload.hungerGain) or 0)
    self.vigorGain  = math.floor((payload and payload.vigorGain)  or 0)
    self.kind       = (payload and payload.kind) or "food"
    self.icon       = (payload and payload.icon) or nil
    self.effectId   = (payload and payload.effectId) or nil
    self.effectsFile= (payload and payload.effectsFile) or nil
    self.placeableTitle = (payload and payload.placeableTitle) or nil
    return self
end


function FoodVendorGainHungerEvent:writeStream(streamId, connection)
    streamWriteInt8(streamId,  FV_clamp(self.hungerGain or 0, -100, 100))
	streamWriteInt8(streamId,  FV_clamp(self.vigorGain  or 0, -100, 100))

    streamWriteString(streamId, tostring(self.kind or "food"))
    streamWriteString(streamId, tostring(self.icon or ""))
    streamWriteUInt16(streamId, math.min(self.boostSec or 0, 65535))
    streamWriteFloat32(streamId, self.boostWalk or 7.5)
    streamWriteFloat32(streamId, self.boostRun  or 13.0)
    streamWriteString(streamId, tostring(self.effectId or ""))
    streamWriteString(streamId, tostring(self.effectsFile or ""))
end

function FoodVendorGainHungerEvent:readStream(streamId, connection)
    self.hungerGain = streamReadInt8(streamId)
    self.vigorGain  = streamReadInt8(streamId)
    self.kind       = streamReadString(streamId)
    local iconStr   = streamReadString(streamId)
    self.icon       = (iconStr ~= "" and iconStr) or nil
    self.boostSec   = streamReadUInt16(streamId)
    self.boostWalk  = streamReadFloat32(streamId)
    self.boostRun   = streamReadFloat32(streamId)
    self.effectId   = streamReadString(streamId)
    self.effectsFile= streamReadString(streamId)

    self:run(connection)
end

local function _resolveIconPath(path)
    if not path or path == "" then return nil end

    if fileExists(path) then
        return path
    end

    local baseDir = (HungerSystem and HungerSystem.modDirectory)
                 or (PlaceableFoodVendor and PlaceableFoodVendor.modDirectory)
                 or g_currentModDirectory
                 or ""
    if baseDir ~= "" then
        local candidate = Utils.getFilename(path, baseDir)
        if fileExists(candidate) then
            return candidate
        end
    end

    if path:lower():sub(-4) == ".dds" then
        local alt = path:sub(1, -5) .. ".png"
        if fileExists(alt) then return alt end
        if baseDir ~= "" then
            local alt2 = Utils.getFilename(alt, baseDir)
            if fileExists(alt2) then return alt2 end
        end
    elseif path:lower():sub(-4) == ".png" then
        local alt = path:sub(1, -5) .. ".dds"
        if fileExists(alt) then return alt end
        if baseDir ~= "" then
            local alt2 = Utils.getFilename(alt, baseDir)
            if fileExists(alt2) then return alt2 end
        end
    end

    return nil
end

local function _topNotify(title, msg, icon)
    local safeIcon = nil
    if icon ~= nil and icon ~= "" then
        if fileExists(icon) then
            safeIcon = icon
        else
            local alt = nil
            if icon:lower():sub(-4) == ".dds" then
                alt = icon:sub(1, -5)..".png"
            elseif icon:lower():sub(-4) == ".png" then
                alt = icon:sub(1, -5)..".dds"
            end
            if alt and fileExists(alt) then safeIcon = alt end
        end
    end

    local hud = g_currentMission and g_currentMission.hud
    if hud and hud.topNotification and hud.topNotification.setNotification then
        hud.topNotification:setNotification(title or "", msg or "", "", safeIcon, nil)
    else
        print(("[FoodVendor] %s %s"):format(tostring(title or ""), tostring(msg or "")))
    end
end

-- ====================================================================
-- Событие: применить эффект покупки
-- ====================================================================
function FoodVendorGainHungerEvent:run(connection)
    local hs = HungerSystem
    if hs ~= nil and hs.instance ~= nil then
        hs = hs.instance
    elseif g_currentMission and (g_currentMission.hungerSystem or g_currentMission.hungerSystemCore) then
        hs = g_currentMission.hungerSystem or g_currentMission.hungerSystemCore
    end

    local kind = tostring(self.kind or "food")
    local hG   = self.hungerGain or 0
    local vG   = self.vigorGain  or 0

    -- локальный игрок
    local player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)
    if not player then
        return
    end

    local function _safeIcon(iconPath)
        if not iconPath or iconPath == "" then return nil end
        if fileExists(iconPath) then return iconPath end
        return nil
    end

    local function _resolveEffectsFile(rawPath)
        if rawPath == nil or rawPath == "" then
            return nil
        end

        if fileExists(rawPath) then
            return rawPath
        end

        local baseDir = (HungerSystem and HungerSystem.modDirectory)
                     or (PlaceableFoodVendor and PlaceableFoodVendor.modDirectory)
                     or (ItemEffects and ItemEffects.modDirectory)
                     or g_currentModDirectory
                     or ""
        if baseDir ~= "" then
            local candidate = Utils.getFilename(rawPath, baseDir)
            if fileExists(candidate) then
                return candidate
            end
        end

        return nil
    end

    ------------------------------------------------------------------
    -- 1) ЭФФЕКТЫ (itemEffects.xml) — ТОЛЬКО если kind == "effect"
    ------------------------------------------------------------------
    if kind == "effect" then
        local effectId = tostring(self.effectId or "")
        local title = (self.placeableTitle)
                    or (g_i18n and g_i18n:getText("ui_foodVendor_title"))
                    or "Кафе"

        if effectId == "" then
            local msg = (g_i18n and g_i18n:getText("ui_effect_notfound")) or "Эффект не найден"
            _topNotify(title, msg, _safeIcon(self.icon))
            return
        end

        local effectsFileArg = _resolveEffectsFile(self.effectsFile)

        local ok, reason

        if hs and hs.applyItemEffect then
            ok, reason = hs:applyItemEffect(effectId, effectsFileArg)
        end

        if (not ok) and (reason == "notFound" or reason == nil) and ItemEffects and ItemEffects.startEffect then
            local xmlAbs = effectsFileArg
            if xmlAbs == nil or xmlAbs == "" then
                local baseDir = (HungerSystem and HungerSystem.modDirectory)
                             or (PlaceableFoodVendor and PlaceableFoodVendor.modDirectory)
                             or (ItemEffects and ItemEffects.modDirectory)
                             or g_currentModDirectory
                             or ""
                xmlAbs = Utils.getFilename("itemEffects.xml", baseDir)
            end

            ok, reason = ItemEffects:startEffect(hs or HungerSystem or ItemEffects, effectId, xmlAbs)
        end

        if not ok then
            local msg =
                (reason == "busy")         and ((g_i18n and g_i18n:getText("ui_effect_busy"))      or "Другой эффект уже действует")
             or (reason == "atCap")        and ((g_i18n and g_i18n:getText("ui_effect_cap"))       or "Достигнут лимит")
             or (reason == "energyActive") and ((g_i18n and g_i18n:getText("ui_energy_denied"))    or "Нельзя использовать")
             or (reason == "notFound")     and ((g_i18n and g_i18n:getText("ui_effect_notfound"))  or "Эффект не найден")
             or (g_i18n and g_i18n:getText("ui_effect_denied")) or "Эффект не сработал"

            local icon = _safeIcon(self.icon)
            _topNotify(title, msg, icon)
        end

        return
    end

    if kind == "energy" and hs and hs.applyEnergyDrink then
        local ok, reason = hs:applyEnergyDrink({
            hungerDelta = hG,
            vigorDelta  = vG,
            boostSec    = self.boostSec,
            walk        = self.boostWalk,
            run         = self.boostRun
        })

        if not ok then
            local title = (self.placeableTitle)
                        or (g_i18n and g_i18n:getText("ui_foodVendor_title"))
                        or "Кафе"
            local msg = (g_i18n and g_i18n:getText("ui_energy_denied")) or "Нельзя использовать энергетик"
            local icon = _safeIcon(self.icon)
            _topNotify(title, msg, icon)
        end

        return
    end

    if hs then
        local appliedH = hs.addHungerDelta and hs:addHungerDelta(hG) or 0
        local appliedV = hs.addVigorDelta  and hs:addVigorDelta(vG)  or 0

        print(("[FoodVendor] apply from vendor: hunger %+d, vigor %+d (kind=%s)")
            :format(appliedH, appliedV, tostring(kind)))
    end
end

function FoodVendorGainHungerEvent.sendToConnection(connection, payload)
    if connection ~= nil then
        connection:sendEvent(FoodVendorGainHungerEvent.new(payload))
    end
end

--------------------------------------------------------------------------------
-- Событие покупки (клиент->сервер)
--------------------------------------------------------------------------------

FoodVendorBuyEvent = {}
local FoodVendorBuyEvent_mt = Class(FoodVendorBuyEvent, Event)
InitEventClass(FoodVendorBuyEvent, "FoodVendorBuyEvent")

function FoodVendorBuyEvent.emptyNew() return Event.new(FoodVendorBuyEvent_mt) end

function FoodVendorBuyEvent.new(placeable, index)
    local self = FoodVendorBuyEvent.emptyNew()
    self.placeable = placeable
    self.index     = index
    return self
end

function FoodVendorBuyEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.placeable)
    streamWriteUInt8(streamId, math.min(self.index or 1, 255))
end

function FoodVendorBuyEvent:readStream(streamId, connection)
    self.placeable = NetworkUtil.readNodeObject(streamId)
    self.index     = streamReadUInt8(streamId)
    self:run(connection)
end

function FoodVendorBuyEvent:run(connection)
    local buyerFarmId = 1
    if HungerSystem and HungerSystem.getFarmIdFromConnection then
        buyerFarmId = HungerSystem.getFarmIdFromConnection(connection)
    elseif g_currentMission
       and g_currentMission.player
       and g_currentMission.player.farmId then
        buyerFarmId = g_currentMission.player.farmId
    end

    if self.placeable ~= nil and self.placeable.serverBuyFood ~= nil then
        local ok, result = self.placeable:serverBuyFood(self.index, buyerFarmId)
        if ok then
            if self.placeable.spec_foodVendor
               and self.placeable.spec_foodVendor.displayName then
                result.placeableTitle = self.placeable.spec_foodVendor.displayName
            end

            if FoodVendorGainHungerEvent ~= nil then
                if connection ~= nil and FoodVendorGainHungerEvent.sendToConnection ~= nil then
                    FoodVendorGainHungerEvent.sendToConnection(connection, result)
                elseif FoodVendorGainHungerEvent.new ~= nil then
                    FoodVendorGainHungerEvent.new(result):run(nil)
                end
            end
        end
    end
end

FoodVendorBuyToInventoryEvent = {}
local FoodVendorBuyToInventoryEvent_mt = Class(FoodVendorBuyToInventoryEvent, Event)
InitEventClass(FoodVendorBuyToInventoryEvent, "FoodVendorBuyToInventoryEvent")

function FoodVendorBuyToInventoryEvent.emptyNew() return Event.new(FoodVendorBuyToInventoryEvent_mt) end
function FoodVendorBuyToInventoryEvent.new(placeable, index)
    local self = FoodVendorBuyToInventoryEvent.emptyNew()
    self.placeable = placeable
    self.index     = index
    return self
end

function FoodVendorBuyToInventoryEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.placeable)
    streamWriteUInt8(streamId, math.min(self.index or 1, 255))
end

function FoodVendorBuyToInventoryEvent:readStream(streamId, connection)
    self.placeable = NetworkUtil.readNodeObject(streamId)
    self.index     = streamReadUInt8(streamId)
    self:run(connection)
end

function FoodVendorBuyToInventoryEvent:run(connection)
    local buyerFarmId = 1
    if HungerSystem and HungerSystem.getFarmIdFromConnection then
        buyerFarmId = HungerSystem.getFarmIdFromConnection(connection)
    elseif g_currentMission
       and g_currentMission.player
       and g_currentMission.player.farmId then
        buyerFarmId = g_currentMission.player.farmId
    end

    if self.placeable ~= nil and self.placeable.serverBuyFood ~= nil then
        local ok, payload = self.placeable:serverBuyFood(self.index, buyerFarmId)
        if ok then
            FoodVendorBuyToInventoryClientEvent.sendToConnection(
                connection,
                payload or {}
            )
        end
    end
end

FoodVendorBuyToInventoryClientEvent = {}
local FoodVendorBuyToInventoryClientEvent_mt = Class(FoodVendorBuyToInventoryClientEvent, Event)
InitEventClass(FoodVendorBuyToInventoryClientEvent, "FoodVendorBuyToInventoryClientEvent")

function FoodVendorBuyToInventoryClientEvent.emptyNew() return Event.new(FoodVendorBuyToInventoryClientEvent_mt) end
function FoodVendorBuyToInventoryClientEvent.new(payload)
    local self = FoodVendorBuyToInventoryClientEvent.emptyNew()
    self.payload = payload or {}
    return self
end

function FoodVendorBuyToInventoryClientEvent:writeStream(streamId, connection)
    local p = self.payload or {}
	streamWriteString(streamId, tostring(p.name or "Food"))
	streamWriteString(streamId, tostring(p.icon or ""))
	streamWriteInt8 (streamId, FV_clamp(p.hungerGain or 0, -100, 100))
	streamWriteInt8 (streamId, FV_clamp(p.vigorGain  or 0, -100, 100))
	streamWriteString(streamId, tostring((p.kind == "effect") and "effect" or "food"))
	streamWriteString(streamId, tostring(p.effectId or ""))
	streamWriteString(streamId, tostring(p.effectsFile or ""))

end

function FoodVendorBuyToInventoryClientEvent:readStream(streamId, connection)
    local p = {}
    p.name       = streamReadString(streamId)
    p.icon       = streamReadString(streamId)
    p.hungerGain = streamReadInt8(streamId)
    p.vigorGain  = streamReadInt8(streamId)
    p.kind       = streamReadString(streamId)
    p.effectId   = streamReadString(streamId)
    p.effectsFile= streamReadString(streamId)
    self.payload = p
    self:run(connection)
end

-- ====================================================================
-- Клиент: пришёл payload для покупки "в инвентарь"
-- ====================================================================
function FoodVendorBuyToInventoryClientEvent:run(connection)
    if g_currentMission == nil then
        return
    end

    if not _G.Inventory or not Inventory.API or not Inventory.API.addItemInstance then
        print("[FoodVendor] Inventory API unavailable in FoodVendorBuyToInventoryClientEvent")
        return
    end

    local p = self.payload or {}

    local iconRel = p.icon or ""
    local iconAbs = iconRel

    if iconRel ~= "" then
        if _resolveIconPath ~= nil then
            local resolved = _resolveIconPath(iconRel)
            if resolved ~= nil then
                iconAbs = resolved
            end
        end

        if (iconAbs == iconRel or not fileExists(iconAbs)) and Inventory and Inventory.modDir then
            local candidate = Utils.getFilename(iconRel, Inventory.modDir)
            if candidate and fileExists(candidate) then
                iconAbs = candidate
            end
        end

        if (iconAbs == iconRel or not fileExists(iconAbs)) and g_currentModDirectory then
            local candidate = Utils.getFilename(iconRel, g_currentModDirectory)
            if candidate and fileExists(candidate) then
                iconAbs = candidate
            end
        end
    end

    local item = {
        name       = p.name or "Food",
        icon       = iconAbs,
        kind       = (p.kind == "effect") and "effect" or "food",
        hungerGain = tonumber(p.hungerGain or 0) or 0,
        vigorGain  = tonumber(p.vigorGain  or 0) or 0,
        effectId   = p.effectId or "",
        effectsFile= p.effectsFile or ""
    }

    Inventory.API.addItemInstance(item)
end

function FoodVendorBuyToInventoryClientEvent.sendToConnection(connection, payload)
    if connection ~= nil then connection:sendEvent(FoodVendorBuyToInventoryClientEvent.new(payload)) end
end


