-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

FoodVendorHUD = {}
FoodVendorHUD.modDirectory = g_currentModDirectory or ""
FoodVendorHUD.consoleRegistered = false
FoodVendorHUD.__index = FoodVendorHUD

local FoodVendorHUD_mt = Class(FoodVendorHUD, MessageDialog)

local function _resolveIcon(pathRelOrAbs)
    if not pathRelOrAbs or pathRelOrAbs == "" then return nil end
    local path = pathRelOrAbs
    if not fileExists(path) then
        local baseDir = (PlaceableFoodVendor and PlaceableFoodVendor.modDirectory) or FoodVendorHUD.modDirectory or ""
        path = Utils.getFilename(pathRelOrAbs, baseDir)
    end
    if not fileExists(path) then
        if path:lower():sub(-4) == ".dds" then
            local alt = path:sub(1, -5) .. ".png"
            if fileExists(alt) then path = alt end
        elseif path:lower():sub(-4) == ".png" then
            local alt = path:sub(1, -5) .. ".dds"
            if fileExists(alt) then path = alt end
        end
    end
    if fileExists(path) then return path end
    return nil
end

local function _topNotify(title, msg, icon)
    local safeIcon = nil
    if icon ~= nil and icon ~= "" then
        if fileExists(icon) then
            safeIcon = icon
        else
            local alt = nil
            if icon:lower():sub(-4) == ".dds" then
                alt = icon:sub(1, -5)..".png"
            elseif icon:lower():sub(-4) == ".png" then
                alt = icon:sub(1, -5)..".dds"
            end
            if alt and fileExists(alt) then
                safeIcon = alt
            end
        end
    end

    local hud = g_currentMission and g_currentMission.hud
    if hud and hud.topNotification and hud.topNotification.setNotification then
        hud.topNotification:setNotification(title or "", msg or "", "", safeIcon, nil)
    else
        print(("[FoodVendorHUD] %s %s"):format(tostring(title or ""), tostring(msg or "")))
    end
end

local function _getLiveFarmerBaseDir_FV()
    local candidates = {}

    if PlaceableFoodVendor and PlaceableFoodVendor.modDirectory and PlaceableFoodVendor.modDirectory ~= "" then
        table.insert(candidates, PlaceableFoodVendor.modDirectory)
    end
    if FoodVendorHUD.modDirectory and FoodVendorHUD.modDirectory ~= "" then
        table.insert(candidates, FoodVendorHUD.modDirectory)
    end

    if g_modManager ~= nil then
        if g_modManager.getModByName ~= nil then
            local m = g_modManager:getModByName("FS25_liveFarmer")
            if m ~= nil then
                local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                if dir and dir ~= "" then
                    table.insert(candidates, dir)
                end
            end
        end
        if g_modManager.mods ~= nil then
            for _, m in pairs(g_modManager.mods) do
                local name     = m.modName or m.name or ""
                local fileName = m.modFileName or ""
                if name == "FS25_liveFarmer" or fileName:lower():find("fs25_livefarmer") ~= nil then
                    local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                    if dir and dir ~= "" then
                        table.insert(candidates, dir)
                    end
                end
            end
        end
    end

    if debug and debug.getinfo then
        local info = debug.getinfo(_getLiveFarmerBaseDir_FV, "S")
        if info and info.source then
            local src = info.source
            if src:sub(1,1) == "@" then src = src:sub(2) end
            src = src:gsub("\\", "/")
            local idx = src:lower():find("/fs25_livefarmer/")
            if idx then
                local dir = src:sub(1, idx + string.len("/FS25_liveFarmer/") - 1)
                table.insert(candidates, dir)
            end
        end
    end

    if g_modsDirectory ~= nil then
        table.insert(candidates, g_modsDirectory .. "/FS25_liveFarmer/")
    end

    local tried = {}
    for _, baseDir in ipairs(candidates) do
        if type(baseDir) == "string" and baseDir ~= "" then
            baseDir = baseDir:gsub("\\", "/")
            if baseDir:sub(-1) ~= "/" then
                baseDir = baseDir .. "/"
            end
            if not tried[baseDir] then
                tried[baseDir] = true
                local testPath = Utils.getFilename("gui/FoodVendorHUD.xml", baseDir)
                if testPath and fileExists(testPath) then
                    return baseDir
                end
            end
        end
    end

    return FoodVendorHUD.modDirectory or (g_currentModDirectory or "")
end

function FoodVendorHUD.registerConsole()
    if FoodVendorHUD.consoleRegistered then return end
    if addConsoleCommand ~= nil then
        addConsoleCommand(
            "gsReloadFoodVendorHUD",
            "Reload FoodVendorHUD.xml (close/reopen if open)",
            "reloadHUD",
            FoodVendorHUD
        )
        FoodVendorHUD.consoleRegistered = true
        print("[FoodVendorHUD] Console command registered: gsReloadFoodVendorHUD")
    end
end

function FoodVendorHUD.reloadHUD()
    if not g_gui then
        print("[FoodVendorHUD] g_gui is not ready")
        return
    end

    local baseDir = _getLiveFarmerBaseDir_FV and _getLiveFarmerBaseDir_FV() or (FoodVendorHUD.modDirectory or g_currentModDirectory or "")
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/FoodVendorHUD.xml", baseDir)
    if not (xmlPath and fileExists(xmlPath)) then
        print("[FoodVendorHUD] HUD XML not found to reload (baseDir=" .. tostring(baseDir) .. ")")
        return
    end

    local keepVendor, wasOpen = nil, false
    if FoodVendorHUD.INSTANCE and FoodVendorHUD.INSTANCE.vendor then
        keepVendor = FoodVendorHUD.INSTANCE.vendor
    end

    if g_gui.closeDialogByName then
        if g_gui.guis and g_gui.guis["FoodVendorHUD"] then
            wasOpen = true
        end
        pcall(function()
            g_gui:closeDialogByName("FoodVendorHUD")
        end)
    elseif g_gui.currentGuiName == "FoodVendorHUD" then
        wasOpen = true
        pcall(function() g_gui:showGui("") end)
    end

    if g_gui.guis and g_gui.guis["FoodVendorHUD"] then
        pcall(function() g_gui:unloadGui("FoodVendorHUD") end)
        g_gui.guis["FoodVendorHUD"] = nil
    end
    if FoodVendorHUD.INSTANCE then
        pcall(function() FoodVendorHUD.INSTANCE:delete() end)
        FoodVendorHUD.INSTANCE = nil
    end

    local ok = FoodVendorHUD.register()
    print("[FoodVendorHUD] Reload result: " .. tostring(ok))

    if ok and wasOpen and keepVendor and FoodVendorHUD.INSTANCE then
        FoodVendorHUD.INSTANCE.vendor = keepVendor
        if g_gui.showDialog then
            g_gui:showDialog("FoodVendorHUD")
        else
            g_gui:showGui("FoodVendorHUD")
        end
    end
end

local function _ensureConsoleCommand()
    if FoodVendorHUD.consoleRegistered then return end
    if addConsoleCommand ~= nil then
        addConsoleCommand("gsReloadFoodVendorHUD", "Reload FoodVendor HUD xml on the fly", "reloadHUD", FoodVendorHUD)
        FoodVendorHUD.consoleRegistered = true
        print("[FoodVendorHUD] Console command registered: gsReloadFoodVendorHUD")
    end
end

function FoodVendorHUD.register()
    if not g_gui then return false end

    FoodVendorHUD.registerConsole()

    if g_gui.guis and g_gui.guis["FoodVendorHUD"] then
        FoodVendorHUD.INSTANCE = g_gui.guis["FoodVendorHUD"].target
        return true
    end

    local inst = FoodVendorHUD.new()

    local baseDir = _getLiveFarmerBaseDir_FV()
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/FoodVendorHUD.xml", baseDir)
    local ok = false

    if xmlPath and xmlPath ~= "" then
        ok = pcall(function()
            g_gui:loadGui(xmlPath, "FoodVendorHUD", inst)
        end)
    end

    if (not ok) or not (g_gui.guis and g_gui.guis["FoodVendorHUD"]) then
        local path1 = baseDir .. "gui/FoodVendorHUD.xml"
        local path2 = baseDir .. "FoodVendorHUD.xml"
        for _, p in ipairs({path1, path2}) do
            if p and p ~= "" then
                local loaded = pcall(function()
                    g_gui:loadGui(p, "FoodVendorHUD", inst)
                end)
                if loaded and g_gui.guis and g_gui.guis["FoodVendorHUD"] then
                    ok = true
                    xmlPath = p
                    break
                end
            end
        end
    end

    if not ok or not (g_gui.guis and g_gui.guis["FoodVendorHUD"]) then
        pcall(function() inst:delete() end)
        print(string.format("[FoodVendorHUD] GUI XML not found (baseDir=%s, xmlPath=%s)", tostring(baseDir), tostring(xmlPath)))
        return false
    end

    FoodVendorHUD.INSTANCE = g_gui.guis["FoodVendorHUD"].target
    return true
end


function FoodVendorHUD.new(target, custom_mt)
    local self = MessageDialog.new(target, custom_mt or FoodVendorHUD_mt)
    self.vendor         = nil
    self.itemsData      = {}
    self.selectedIndex  = 0
    self.itemsTable     = nil
    self.itemsSlider    = nil
    self.buyButton      = nil
    self.descText       = nil
    self.headerElement  = nil
    return self
end

function FoodVendorHUD:onOpen()
    local sc = self.superClass and self:superClass()
    if sc and sc.onOpen then sc.onOpen(self) end

    self.itemsTable     = self:getDescendantById("itemsTable")
    self.itemsSlider    = self:getDescendantById("itemsTableSlider")
    self.buyButton      = self:getDescendantById("buyButton")
    self.buyToInvButton = self:getDescendantById("buyToInvButton") -- может быть nil
    self.descText       = self:getDescendantById("itemDesc")
    self.headerElement  = self:getDescendantById("dialogTitleElement")

    self.hvPanel      = self:getDescendantById("hvPanel")
    self.hvHungerText = self:getDescendantById("hvHungerText")
    self.hvVigorText  = self:getDescendantById("hvVigorText")
    self.hvHungerIcon = self:getDescendantById("hvHungerIcon")
    self.hvVigorIcon  = self:getDescendantById("hvVigorIcon")

    self:updateHeaderTitle()
    self:updateData()

    if self.buyToInvButton and self.buyToInvButton.setText then
        local txt = (g_i18n and g_i18n:getText("ui_foodVendor_buyToInv")) or "В инвентарь"
        self.buyToInvButton:setText(txt)
    end

    local modDir = (PlaceableFoodVendor and PlaceableFoodVendor.modDirectory)
                or  FoodVendorHUD.modDirectory
                or  g_currentModDirectory
                or  ""
    if modDir ~= "" then
        local hungerPath = modDir .. "gui/icon/hunger.dds"
        local vigorPath  = modDir .. "gui/icon/vigor.dds"
        if self.hvHungerIcon and self.hvHungerIcon.setImageFilename and hungerPath ~= "" and fileExists(hungerPath) then
            self.hvHungerIcon:setImageFilename(hungerPath)
            if self.hvHungerIcon.setUseNineSliceShader then self.hvHungerIcon:setUseNineSliceShader(false) end
        end
        if self.hvVigorIcon and self.hvVigorIcon.setImageFilename and vigorPath ~= "" and fileExists(vigorPath) then
            self.hvVigorIcon:setImageFilename(vigorPath)
            if self.hvVigorIcon.setUseNineSliceShader then self.hvVigorIcon:setUseNineSliceShader(false) end
        end
    end

    self._hvUpdateTimer = 0
end

function FoodVendorHUD:onClose()
    self.vendor        = nil
    self.itemsData     = {}
    self.selectedIndex = 0
    MessageDialog.onClose(self)
end

function FoodVendorHUD:update(dt)
    local sc = self.superClass and self:superClass()
    if sc and sc.update then sc.update(self, dt) end

    if not (self._hvUpdateTimer and self.hvPanel and self.hvHungerText and self.hvVigorText) then
        return
    end

    self._hvUpdateTimer = (self._hvUpdateTimer or 0) - (dt or 0)
    if self._hvUpdateTimer > 0 then return end
    self._hvUpdateTimer = 228

    local core = (g_currentMission and (g_currentMission.hungerSystem or HungerSystem)) or HungerSystem
    local player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)

    local hunger = (player and player.hunger) or (core and core.hunger) or 100
    local vigor  = (player and player.vigor)  or (core and core.vigor)  or 100
    hunger = math.floor(math.max(0, math.min(100, hunger)))
    vigor  = math.floor(math.max(0, math.min(100, vigor)))

    if self.hvHungerText.setText then self.hvHungerText:setText(("%d%%"):format(hunger)) end
    if self.hvVigorText.setText  then self.hvVigorText :setText(("%d%%"):format(vigor))  end

    local function col(v, hi, mid, lo)
        if v > 60 then return hi[1],hi[2],hi[3],1
        elseif v > 20 then return mid[1],mid[2],mid[3],1
        else return lo[1],lo[2],lo[3],1 end
    end
    local r,g,b,a = col(hunger, {0.55,1.00,0.55},{1.00,0.93,0.55},{1.00,0.55,0.55})
    if self.hvHungerText.setTextColor then self.hvHungerText:setTextColor(r,g,b,a) end
    r,g,b,a = col(vigor,  {0.55,1.00,0.55},{1.00,0.93,0.55},{1.00,0.55,0.55})
    if self.hvVigorText.setTextColor then self.hvVigorText:setTextColor(r,g,b,a) end
end

function FoodVendorHUD:updateHeaderTitle()
    local spec = self.vendor and self.vendor.spec_foodVendor
    local fallback = (g_i18n and g_i18n:getText("ui_foodVendor_title")) or "Кафе"
    local title = fallback

    if spec and spec.displayName and spec.displayName ~= "" then
        title = spec.displayName
        if string.find(title, "Missing '", 1, true) then title = fallback end
    end

    if self.headerElement and self.headerElement.setText then
        self.headerElement:setText(title)
    end
end

local function _normL10n(raw)
    if not raw or raw == "" then return "" end
    if raw:sub(1,1) ~= "$" then return raw end
    local k = raw:sub(2)
    if g_i18n and g_i18n.hasText and g_i18n:hasText(k) then
        return g_i18n:getText(k)
    end
    if k:sub(1,5) == "l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then
        return g_i18n:getText(k:sub(6))
    end
    return k:gsub("^ui_", ""):gsub("_", " ")
end

local function _setDesc(self, text)
    if not self.descText then return end
    self.descText:setText(text or "")
end

function FoodVendorHUD:updateData()
    self.itemsData = {}

    if self.vendor and self.vendor.spec_foodVendor then
        for _, it in ipairs(self.vendor.spec_foodVendor.items) do
            local name   = _normL10n(it.name or "")
            local desc   = _normL10n(it.desc or "")
            local kind   = (it.kind or "food")

            local hGain  = tonumber(it.hungerGain) or 0
            local vGain  = tonumber(it.vigorGain)  or 0

            local labelH = (g_i18n and g_i18n:getText("ui_foodVendor_hungerShort")) or "Сытость"
            local labelV = (g_i18n and g_i18n:getText("ui_foodVendor_vigorShort"))  or "Бодрость"
            local labelP = (g_i18n and g_i18n:getText("ui_foodVendor_priceShort"))  or "Цена"

            local parts  = {}
            table.insert(parts, string.format("%s: %+d%%", labelH, hGain))
            if vGain ~= 0 then
                table.insert(parts, string.format("%s: %+d%%", labelV, vGain))
            end
            local effectsStr = table.concat(parts, ", ")

            table.insert(self.itemsData, {
                display    = string.format("%s — %s, %s: %d $", name, effectsStr, labelP, it.price or 0),
                name       = name,
                desc       = desc,
                price      = it.price or 0,
                hunger     = hGain,
                vigor      = vGain,
                hungerGain = hGain,
                vigorGain  = vGain,
                icon       = it.icon,
                kind       = kind,
                effectId   = it.effectId or "",
                effectsFile= (self.effectsFile or it.effectsFile)
            })
        end
    end

    if self.itemsTable then
        self.itemsTable:reloadData(true)
        if #self.itemsData > 0 then
            self.selectedIndex = 1
            self.itemsTable:setSelectedIndex(1, true, true)
            self.itemsTable:scrollTo(0, false)
            _setDesc(self, self.itemsData[1].desc or "")
        else
            self.selectedIndex = 0
            _setDesc(self, "")
        end
    end

    self:updateBuyButton()
end

function FoodVendorHUD:getNumberOfSections(list) return 1 end
function FoodVendorHUD:getNumberOfItemsInSection(list, sectionIndex) return #self.itemsData end
function FoodVendorHUD:getCellTypeForItemInSection(list, sectionIndex, indexInSection) return "rowTemplate" end

function FoodVendorHUD:populateCellForItemInSection(list, sectionIndex, indexInSection, cell)
    local data = self.itemsData[indexInSection]
    if not data then return end

    local rowText = cell:getAttribute("rowText") or cell:getDescendantByName("rowText") or cell:getDescendantById("rowText")
    if rowText then rowText:setText(data.display) end

    local iconElem = cell:getAttribute("rowIcon") or cell:getDescendantByName("rowIcon") or cell:getDescendantById("rowIcon")
    if iconElem then
        local full = _resolveIcon(data.icon)
        if full and iconElem.setImageFilename then
            if iconElem.setImageColor then iconElem:setImageColor(1,1,1,1) end
            iconElem:setImageFilename(full)
            iconElem:setVisible(true)
        else
            iconElem:setVisible(false)
        end
    end
end

function FoodVendorHUD:onListSelectionChanged(list, sectionIndex, indexInSection)
    self.selectedIndex = indexInSection or 0
    self:updateBuyButton()
    if self.selectedIndex > 0 and self.itemsData[self.selectedIndex] then
        _setDesc(self, self.itemsData[self.selectedIndex].desc or "")
    else
        _setDesc(self, "")
    end
end

function FoodVendorHUD:updateBuyButton()
    local enable = (self.selectedIndex > 0 and self.itemsData[self.selectedIndex] ~= nil)

    if self.buyButton then
        if enable then
            local price = self.itemsData[self.selectedIndex].price or 0
            local label = (g_i18n and g_i18n:getText("ui_foodVendor_buy")) or "Купить (%d $)"
            self.buyButton:setDisabled(false)
            self.buyButton:setText(string.format(label, price))
        else
            self.buyButton:setDisabled(true)
            self.buyButton:setText((g_i18n and g_i18n:getText("ui_foodVendor_buy")) or "Купить")
        end
    end

    if self.buyToInvButton then
        self.buyToInvButton:setDisabled(not enable)
        local label = (g_i18n and g_i18n:getText("ui_foodVendor_buyToInv")) or "В инвентарь"
        self.buyToInvButton:setText(label)
    end
end

local function _getLocalHunger()
    local p = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
              or (g_currentMission and g_currentMission.player)
    if p and p.hunger ~= nil then return tonumber(p.hunger) or 100 end
    if _G.HungerSystem and HungerSystem.hunger ~= nil then return tonumber(HungerSystem.hunger) or 100 end
    return 100
end

function FoodVendorHUD:onClickBuy()
    if not (self.vendor and self.itemsData[self.selectedIndex]) then return end
    local idx = self.selectedIndex
    local it  = self.itemsData[idx]

    if it and (it.kind == "food") then
        local function _getLocalHunger()
            local p = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                   or (g_currentMission and g_currentMission.player)
            if p and p.hunger ~= nil then return tonumber(p.hunger) or 100 end
            if _G.HungerSystem and HungerSystem.hunger ~= nil then return tonumber(HungerSystem.hunger) or 100 end
            return 100
        end
        if _getLocalHunger() >= 100 and (it.hunger or 0) > 0 then
            local ttl = (self.vendor and self.vendor.spec_foodVendor and self.vendor.spec_foodVendor.displayName)
                        or (g_i18n and g_i18n:getText("ui_foodVendor_title")) or "Кафе"
            local hud = g_currentMission and g_currentMission.hud
            if hud and hud.topNotification and hud.topNotification.setNotification then
                hud.topNotification:setNotification(ttl,
                    (g_i18n and g_i18n:getText("ui_foodVendor_hungerFull")) or "Сытость полная",
                    "", it.icon or "", nil)
            end
            return
        end
    end

    if g_client ~= nil then
        local conn = g_client:getServerConnection()
        if conn ~= nil then
            conn:sendEvent(FoodVendorBuyEvent.new(self.vendor, idx))
        end
    else
        local farmId = (g_currentMission and g_currentMission.player and g_currentMission.player.farmId) or 1
        local ok, result = self.vendor:serverBuyFood(idx, farmId)
        if ok then
            if self.vendor.spec_foodVendor and self.vendor.spec_foodVendor.displayName then
                result.placeableTitle = self.vendor.spec_foodVendor.displayName
                FoodVendorGainHungerEvent.new(result):run(nil)
            else
                FoodVendorGainHungerEvent.new(result):run(nil)
            end
        end
    end
end

function FoodVendorHUD:_addItemIndexToInventory(index)
    if not _G.Inventory or not Inventory.API or not Inventory.API.addItemInstance then
        print("[FoodVendorHUD] Inventory API unavailable")
        return false
    end

    local item = self.itemsData[index]
    if not item then return false end

    if not (Inventory.API.firstFreeSlot and Inventory.API.firstFreeSlot()) then
        self:notify(g_i18n:getText("ui_inventory_full") or "Нет места в инвентаре")
        return false
    end

    local payload = {
        id         = item.id,
        name       = item.name,
        desc       = item.desc or item.description,
        icon       = item.icon,
        kind       = item.kind or "food",

        hungerGain = item.hungerGain or item.hunger or 0,
        vigorGain  = item.vigorGain  or item.vigor  or 0,

        effectId   = item.effectId or "",
        effectsFile= self.effectsFile or item.effectsFile or ""
    }

    local ok = select(1, Inventory.API.addItemInstance(payload))
    if ok then
        self:notify(g_i18n:getText("ui_added_to_inventory") or "Добавлено в инвентарь")
    end
    return ok
end

-- Кнопка «Купить в инвентарь»
function FoodVendorHUD:onClickBuyToInventory()
    if not (self.vendor and self.itemsData and self.selectedIndex) then return end

    if not _G.Inventory or not (Inventory.API and Inventory.API.firstFreeSlot and Inventory.API.firstFreeSlot()) then
        self:notify(g_i18n:getText("ui_inventory_full") or "Нет места в инвентаре")
        return
    end

    local idx = self.selectedIndex

    -- КЛИЕНТ → СЕРВЕР
    if g_client ~= nil then
        local conn = g_client:getServerConnection()
        if conn ~= nil then
            conn:sendEvent(FoodVendorBuyToInventoryEvent.new(self.vendor, idx))
        end
        return
    end

    local farmId = (g_currentMission and g_currentMission.player and g_currentMission.player.farmId) or 1
    local ok, payload = self.vendor:serverBuyFood(idx, farmId)
    if not ok then return end

    self:_addItemIndexToInventory(idx)
end

function FoodVendorHUD:notify(text)
    local hud = g_currentMission and g_currentMission.hud
    if hud and hud.topNotification and hud.topNotification.setNotification then
        local ttl = (g_i18n and g_i18n:getText("ui_foodVendor_title")) or "Кафе"
        hud.topNotification:setNotification(ttl, text or "", "", nil, nil)
    else
        print("[FoodVendorHUD] " .. tostring(text))
    end
end

function FoodVendorHUD._addPayloadToInventory(payload)
    if not payload then return end
    if not _G.Inventory or not Inventory.API or not Inventory.API.addItemInstance then
        print("[FoodVendorHUD] Inventory API not found")
        return
    end

    local iconPath = payload.icon or ""
    if iconPath ~= "" then
        local abs = iconPath
        if not fileExists(abs) then
            local baseDir = Inventory.modDir or g_currentModDirectory or ""
            if baseDir ~= "" then
                local candidate = Utils.getFilename(iconPath, baseDir)
                if fileExists(candidate) then
                    abs = candidate
                end
            end
        end
        iconPath = abs
    end

    local item = {
        name       = payload.name or "Food",
        icon       = iconPath,
        kind       = (payload.kind == "effect") and "effect" or "food",
        hungerGain = tonumber(payload.hungerGain or 0) or 0,
        vigorGain  = tonumber(payload.vigorGain  or 0) or 0,
        effectId   = payload.effectId or "",
        effectsFile= payload.effectsFile or ""
    }

    Inventory.API.addItemInstance(item)
end

function FoodVendorHUD:onClickClose()
    self:close()
end
