-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

VehicleStorageHUD = {}
local VehicleStorageHUD_mt = Class(VehicleStorageHUD, MessageDialog)

VehicleStorageHUD.modDirectory = g_currentModDirectory or ""
VehicleStorageHUD.INSTANCE = nil

local GUI_NAME = "VehicleStorageHUD"

local function _l10n(raw)
  if not raw or raw=="" then return "" end
  if raw:sub(1,1) ~= "$" then return raw end
  local k = raw:sub(2)
  if g_i18n and g_i18n.hasText and g_i18n:hasText(k) then return g_i18n:getText(k) end
  if k:sub(1,5)=="l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then return g_i18n:getText(k:sub(6)) end
  return raw
end

local function _getLiveFarmerBaseDir_VS()
    local candidates = {}

    if VehicleStorageHUD.modDirectory and VehicleStorageHUD.modDirectory ~= "" then
        table.insert(candidates, VehicleStorageHUD.modDirectory)
    end
    if g_currentModDirectory and g_currentModDirectory ~= "" then
        table.insert(candidates, g_currentModDirectory)
    end

    if g_modManager ~= nil then
        if g_modManager.getModByName ~= nil then
            local m = g_modManager:getModByName("FS25_liveFarmer")
            if m ~= nil then
                local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                if dir and dir ~= "" then
                    table.insert(candidates, dir)
                end
            end
        end
        if g_modManager.mods ~= nil then
            for _, m in pairs(g_modManager.mods) do
                local name = m.modName or m.name
                if name == "FS25_liveFarmer" then
                    local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                    if dir and dir ~= "" then
                        table.insert(candidates, dir)
                    end
                end
            end
        end
    end

    local tried = {}
    for _, baseDir in ipairs(candidates) do
        if type(baseDir) == "string" and baseDir ~= "" then
            baseDir = baseDir:gsub("\\", "/")
            if baseDir:sub(-1) ~= "/" then
                baseDir = baseDir .. "/"
            end
            if not tried[baseDir] then
                tried[baseDir] = true
                local testPath = Utils.getFilename("gui/VehicleStorageHUD.xml", baseDir)
                if testPath and fileExists(testPath) then
                    return baseDir
                end
            end
        end
    end

    if g_modsDirectory ~= nil then
        local baseDir = (g_modsDirectory .. "/FS25_liveFarmer/"):gsub("\\", "/")
        local testPath = Utils.getFilename("gui/VehicleStorageHUD.xml", baseDir)
        if testPath and fileExists(testPath) then
            return baseDir
        end
    end

    return VehicleStorageHUD.modDirectory or (g_currentModDirectory or "")
end

function VehicleStorageHUD.register()
    if not g_gui then return false end

    if g_gui.guis and g_gui.guis[GUI_NAME] then
        VehicleStorageHUD.INSTANCE = g_gui.guis[GUI_NAME].target
        return true
    end

    local inst = VehicleStorageHUD.new()

    local baseDir = _getLiveFarmerBaseDir_VS()
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/VehicleStorageHUD.xml", baseDir)
    local ok = false

    if xmlPath and xmlPath ~= "" and fileExists(xmlPath) then
        ok = g_gui:loadGui(xmlPath, GUI_NAME, inst)
    end

    if not ok then
        local altPath = Utils.getFilename("gui/VehicleStorageHUD.xml", g_currentModDirectory or "")
        if altPath and altPath ~= "" and altPath ~= xmlPath and fileExists(altPath) then
            ok = g_gui:loadGui(altPath, GUI_NAME, inst)
        end
    end

    if not ok then
        pcall(function() inst:delete() end)
        print("[VehicleStorageHUD] XML not found (baseDir=" .. tostring(baseDir) .. ")")
        return false
    end

    VehicleStorageHUD.INSTANCE = inst
    return true
end


function VehicleStorageHUD.new(target, custom_mt)
  local self = MessageDialog.new(target, custom_mt or VehicleStorageHUD_mt)
  self.vehicle     = nil
  self.inventory   = Inventory
  self.leftData, self.rightData = {}, {}
  self.leftSel,  self.rightSel  = 0, 0
  self.leftList, self.rightList = nil, nil
  self.leftSlider, self.rightSlider = nil, nil
  self.btnToStorage, self.btnToInv, self.btnClose = nil, nil, nil
  self.selTitle, self.selDesc = nil, nil
  self.titleElement = nil
  self.capacityLabel = nil
  return self
end

function VehicleStorageHUD:onOpen()
  local sc = self.superClass and self:superClass()
  if sc and sc.onOpen then sc.onOpen(self) end

  if g_inputBinding and g_inputBinding.setShowMouseCursor then
    g_inputBinding:setShowMouseCursor(true)
  end
  self.leftList      = self:getDescendantById("leftList")
  self.rightList     = self:getDescendantById("rightList")
  self.leftSlider    = self:getDescendantById("leftListSlider")
  self.rightSlider   = self:getDescendantById("rightListSlider")
  self.btnToStorage  = self:getDescendantById("toStorageBtn")
  self.btnToInv      = self:getDescendantById("toInventoryBtn")
  self.btnClose      = self:getDescendantById("closeButton")
  self.selTitle      = self:getDescendantById("selectedTitle")
  self.selDesc       = self:getDescendantById("selectedDesc")
  self.titleElement  = self:getDescendantById("dialogTitleElement")
  self.capacityLabel = self:getDescendantById("capacityLabel")
  if self.leftList then self.leftList:setDataSource(self);  self.leftList:setDelegate(self)  end
  if self.rightList then self.rightList:setDataSource(self); self.rightList:setDelegate(self) end
  if FocusManager and self.leftList then FocusManager:setFocus(self.leftList) end
  self:_applyTitleFromVehicle()
  self:refreshData()
end

function VehicleStorageHUD:_applyTitleFromVehicle()
  if not self.titleElement then return end
  local spec = self.vehicle and self.vehicle.spec_vehicleStorage
  local title = (spec and spec.titleRaw and spec.titleRaw ~= "") and spec.titleRaw or "$l10n_ui_vstore_title"
  self.titleElement:setText(_l10n(title))
end

function VehicleStorageHUD:onClose()
  if g_inputBinding and g_inputBinding.setShowMouseCursor then
    g_inputBinding:setShowMouseCursor(false)
  end
  self.leftData, self.rightData = {}, {}
  self.leftSel, self.rightSel = 0, 0
  MessageDialog.onClose(self)
end

function VehicleStorageHUD:close()
  g_gui:showGui("")
end

function VehicleStorageHUD:getNumberOfSections(list) return 1 end
function VehicleStorageHUD:getNumberOfItemsInSection(list, section)
  if list == self.leftList then return #self.leftData else return #self.rightData end
end
function VehicleStorageHUD:getCellTypeForItemInSection(list, section, index) return "rowTemplate" end
function VehicleStorageHUD:populateCellForItemInSection(list, section, index, cell)
  local item = (list == self.leftList) and self.leftData[index] or self.rightData[index]
  local rowIcon = cell:getDescendantByName("rowIcon") or cell:getDescendantById("rowIcon")
  local rowText = cell:getDescendantByName("rowText") or cell:getDescendantById("rowText")

  if item then
    if rowIcon then
      if item.icon and item.icon ~= "" and fileExists(item.icon) and rowIcon.setImageFilename then
        rowIcon:setImageFilename(item.icon); rowIcon:setVisible(true)
      else rowIcon:setVisible(false) end
    end
    local name = _l10n(item.name or "")
    local h = tonumber(item.hungerGain or 0) or 0
    local v = tonumber(item.vigorGain  or 0) or 0
    if rowText then rowText:setText(string.format("%s — %+dH, %+dV", name, h, v)) end
  else
    if rowIcon then rowIcon:setVisible(false) end
    if rowText then rowText:setText("") end
  end
end
function VehicleStorageHUD:onListSelectionChanged(list, section, index)
  if list == self.leftList then self.leftSel = index or 0;  self:updateSelectedInfo(self.leftData[self.leftSel])
  else self.rightSel = index or 0; self:updateSelectedInfo(self.rightData[self.rightSel]) end
  self:updateButtons()
end

function VehicleStorageHUD:setVehicle(vehicle)
  self.vehicle = vehicle
  self:_applyTitleFromVehicle()
  self:refreshData()
end

local function _safeReload(list)
  if not list then return end
  if list.reloadData then list:reloadData(true)
  elseif list.reloadDataSource then list:reloadDataSource() end
end

local function _countNonNil(tbl, max)
  local c=0
  for i=1, max do if tbl[i] ~= nil then c=c+1 end end
  return c
end

function VehicleStorageHUD:refreshData()
  self.leftData = {}
  if self.inventory and self.inventory.items then
    local maxL = (self.inventory.gridCols or 7) * (self.inventory.gridRows or 5)
    for i=1,maxL do local it=self.inventory.items[i]; if it then table.insert(self.leftData, it) end end
  end
  
  self.rightData = {}
  local spec = self.vehicle and self.vehicle.spec_vehicleStorage
  if spec and spec.items then
    for i=1, spec.maxSlots or 0 do local it = spec.items[i]; if it then table.insert(self.rightData, it) end end
  end

  _safeReload(self.leftList)
  if self.leftList and self.leftList.setSelectedItem then
    self.leftList:setSelectedItem(1, (#self.leftData>0) and 1 or 0, true, true)
  end
  self.leftSel = (#self.leftData>0) and 1 or 0

  _safeReload(self.rightList)
  if self.rightList and self.rightList.setSelectedItem then
    self.rightList:setSelectedItem(1, (#self.rightData>0) and 1 or 0, true, true)
  end
  self.rightSel = (#self.rightData>0) and 1 or 0

  self:updateSelectedInfo(self.leftData[1] or self.rightData[1])
  self:updateButtons()
  self:updateCapacityLabel()
end

function VehicleStorageHUD:updateSelectedInfo(item)
  if not (self.selTitle and self.selDesc) then return end
  if not item then self.selTitle:setText(""); self.selDesc:setText(""); return end
  self.selTitle:setText(_l10n(item.name or ""))
  self.selDesc:setText(_l10n(item.desc or ""))
end

function VehicleStorageHUD:updateButtons()
  local canToStorage   = (self.leftData[self.leftSel]  ~= nil)
  local canToInventory = (self.rightData[self.rightSel] ~= nil)
  if self.btnToStorage  then self.btnToStorage:setDisabled(not canToStorage) end
  if self.btnToInv      then self.btnToInv:setDisabled(not canToInventory) end
end

function VehicleStorageHUD:updateCapacityLabel()
  if not self.capacityLabel then return end
  local spec = self.vehicle and self.vehicle.spec_vehicleStorage
  if not spec then self.capacityLabel:setText(""); return end
  local used = _countNonNil(spec.items or {}, spec.maxSlots or 0)
  local max  = tonumber(spec.maxSlots or 0) or 0
  
  self.capacityLabel:setText(string.format("%d\\%d", used, max))
end

function VehicleStorageHUD:onClickMoveToStorage()
  if not (self.vehicle and self.leftData[self.leftSel]) then return end
  local srcItem = self.leftData[self.leftSel]
  local srcIndex
  if Inventory and Inventory.items then
    local maxL = (Inventory.gridCols or 7) * (Inventory.gridRows or 5)
    for i=1,maxL do if Inventory.items[i] == srcItem then srcIndex = i; break end end
  end
  if not srcIndex then return end
  if self.vehicle.vsPutFromInventory and self.vehicle:vsPutFromInventory(srcItem, srcIndex) then
    self:refreshData()
  end
end

function VehicleStorageHUD:onClickMoveToInventory()
  if not (self.vehicle and self.rightData[self.rightSel]) then return end
  local srcItem = self.rightData[self.rightSel]
  local spec = self.vehicle.spec_vehicleStorage; if not spec then return end
  local rightIndex
  for i=1, spec.maxSlots or 0 do if spec.items[i] == srcItem then rightIndex = i; break end end
  if not rightIndex then return end
  if self.vehicle.vsTakeToInventory and self.vehicle:vsTakeToInventory(rightIndex) then
    self:refreshData()
  end
end

function VehicleStorageHUD:onClickClose()
  self:close()
end

VehicleStorageHUD.consoleRegistered = false

function VehicleStorageHUD.registerConsole()
  if VehicleStorageHUD.consoleRegistered then return end
  if addConsoleCommand ~= nil then
    addConsoleCommand(
      "gsReloadVehicleStorageHUD",
      "Reload VehicleStorageHUD.xml (close/reopen if open)",
      "reloadHUD",
      VehicleStorageHUD
    )
    VehicleStorageHUD.consoleRegistered = true
    print("[VehicleStorageHUD] Console command registered: gsReloadVehicleStorageHUD")
  end
end

function VehicleStorageHUD.reloadHUD()
  if not g_gui then
    print("[VehicleStorageHUD] g_gui is not ready")
    return
  end

  local baseDir = _getLiveFarmerBaseDir_VS and _getLiveFarmerBaseDir_VS() or (VehicleStorageHUD.modDirectory or g_currentModDirectory or "")
  baseDir = baseDir or ""
  baseDir = baseDir:gsub("\\", "/")
  if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
    baseDir = baseDir .. "/"
  end

  local xmlPath = Utils.getFilename("gui/VehicleStorageHUD.xml", baseDir)
  if not (xmlPath and fileExists(xmlPath)) then
    print("[VehicleStorageHUD] HUD XML not found to reload (baseDir=" .. tostring(baseDir) .. ")")
    return
  end

  local keepVehicle, wasOpen = nil, false
  if VehicleStorageHUD.INSTANCE and VehicleStorageHUD.INSTANCE.vehicle then
    keepVehicle = VehicleStorageHUD.INSTANCE.vehicle
    wasOpen = (g_gui.currentGuiName == "VehicleStorageHUD")
    pcall(function() g_gui:showGui("") end)
  end

  if g_gui.guis and g_gui.guis[GUI_NAME] then
    pcall(function() g_gui:unloadGui(GUI_NAME) end)
    g_gui.guis[GUI_NAME] = nil
  end

  if VehicleStorageHUD.INSTANCE then
    pcall(function() VehicleStorageHUD.INSTANCE:delete() end)
    VehicleStorageHUD.INSTANCE = nil
  end

  local ok = VehicleStorageHUD.register()
  print("[VehicleStorageHUD] Reload result: " .. tostring(ok))

  if ok and wasOpen and keepVehicle and VehicleStorageHUD.INSTANCE then
    VehicleStorageHUD.INSTANCE:setVehicle(keepVehicle)
    g_gui:showGui("VehicleStorageHUD")
  end
end

VehicleStorageHUD.registerConsole()
