-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

VehicleStorage = {}
VehicleStorage.MOD_NAME = g_currentModName or "FS25_liveFarmer"

function VehicleStorage.initSpecialization()
  local schema = Vehicle.xmlSchema
  schema:setXMLSpecializationType("VehicleStorage")
  local base = "vehicle.vehicleStorage"
  schema:register(XMLValueType.NODE_INDEX, base .. "#playerTrigger", "Trigger node for activatable")
  schema:register(XMLValueType.INT,        base .. "#itemSlot",      "Max items in vehicle storage (new)", 0)
  schema:register(XMLValueType.STRING,     base .. "#title",         "Custom dialog title (l10n or raw)", "") 
  schema:register(XMLValueType.INT,        base .. "#cols",          "Grid columns (deprecated)", 7)
  schema:register(XMLValueType.INT,        base .. "#rows",          "Grid rows (deprecated)", 5)
  schema:register(XMLValueType.BOOL,       base .. "#onlyOwner",     "Only owner can open", true)
  schema:setXMLSpecializationType()
  local schemaSave = Vehicle.xmlSchemaSavegame
  local nsRoot = "vehicles.vehicle(?)." .. VehicleStorage.MOD_NAME .. ".vehicleStorage"
  schemaSave:register(XMLValueType.INT, nsRoot .. "#itemSlot", "Saved max slots (itemSlot or cols*rows)")
  schemaSave:register(XMLValueType.INT, nsRoot .. "#cols",     "Saved cols (legacy config echo)")
  schemaSave:register(XMLValueType.INT, nsRoot .. "#rows",     "Saved rows (legacy config echo)")
  local nsSlot = nsRoot .. ".slots.slot(?)"
  schemaSave:register(XMLValueType.INT,    nsSlot.."#index",       "Slot index 1..N")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#id",          "Item id/uid")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#name",        "Item name (l10n or raw)")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#desc",        "Item description (l10n or raw)")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#icon",        "Item icon path")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#kind",        "Item kind")
  schemaSave:register(XMLValueType.INT,    nsSlot.."#hungerGain",  "Hunger bonus")
  schemaSave:register(XMLValueType.INT,    nsSlot.."#vigorGain",   "Vigor bonus")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#effectId",    "Effect Id")
  schemaSave:register(XMLValueType.STRING, nsSlot.."#effectsFile", "Effects file")
end

function VehicleStorage.prerequisitesPresent(_)
  return true
end

function VehicleStorage.registerFunctions(vehicleType)
  SpecializationUtil.registerFunction(vehicleType, "vsTriggerCallback",     VehicleStorage.vsTriggerCallback)
  SpecializationUtil.registerFunction(vehicleType, "vsGetIsAccessible",     VehicleStorage.vsGetIsAccessible)
  SpecializationUtil.registerFunction(vehicleType, "vsPutFromInventory",    VehicleStorage.vsPutFromInventory)
  SpecializationUtil.registerFunction(vehicleType, "vsTakeToInventory",     VehicleStorage.vsTakeToInventory)
  SpecializationUtil.registerFunction(vehicleType, "vsOpen",                VehicleStorage.vsOpen)
  SpecializationUtil.registerFunction(vehicleType, "vsClose",               VehicleStorage.vsClose)
  SpecializationUtil.registerFunction(vehicleType, "vsToggle",              VehicleStorage.vsToggle)
  SpecializationUtil.registerFunction(vehicleType, "vsIsOpen",              VehicleStorage.vsIsOpen)
  SpecializationUtil.registerFunction(vehicleType, "vsSaveToXML",           VehicleStorage.vsSaveToXML)
  SpecializationUtil.registerFunction(vehicleType, "vsLoadFromXML",         VehicleStorage.vsLoadFromXML)
  SpecializationUtil.registerFunction(vehicleType, "vsRefreshBothHUDs",     VehicleStorage.vsRefreshBothHUDs)
  SpecializationUtil.registerFunction(vehicleType, "vsMarkDirty",           VehicleStorage.vsMarkDirty)
end

function VehicleStorage.registerEventListeners(vehicleType)
  SpecializationUtil.registerEventListener(vehicleType, "onLoad",        VehicleStorage)
  SpecializationUtil.registerEventListener(vehicleType, "onDelete",      VehicleStorage)
  SpecializationUtil.registerEventListener(vehicleType, "onReadStream",  VehicleStorage)
  SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", VehicleStorage)
  SpecializationUtil.registerEventListener(vehicleType, "saveToXMLFile", VehicleStorage)
end

local function _genId()
  VehicleStorage.__uid = (VehicleStorage.__uid or 0) + 1
  return string.format("vstore_%d_%d", g_time or 0, VehicleStorage.__uid)
end

local function _firstFreeSlot(tbl, max)
  for i=1, max do if tbl[i] == nil then return i end end
  return nil
end

local function _endsWith(s, suf)
  if s == nil or suf == nil then return false end
  local ls, lsuf = #s, #suf
  if lsuf > ls then return false end
  return s:sub(ls - lsuf + 1, ls) == suf
end

function VehicleStorage:onLoad(savegame)
  local spec = self.spec_vehicleStorage or {}
  self.spec_vehicleStorage = spec
  local xml  = self.xmlFile
  local base = "vehicle.vehicleStorage"
  spec.triggerNode = xml:getValue(base .. "#playerTrigger", nil, self.components, self.i3dMappings)
  spec.onlyOwner   = xml:getValue(base .. "#onlyOwner", true)
  spec.itemSlot    = math.max(0, xml:getValue(base .. "#itemSlot", 0))
  spec.gridCols    = xml:getValue(base .. "#cols", 7)
  spec.gridRows    = xml:getValue(base .. "#rows", 5)
  spec.titleRaw    = xml:getValue(base .. "#title", "") or ""
  if spec.itemSlot and spec.itemSlot > 0 then
    spec.maxSlots = spec.itemSlot
  else
    spec.maxSlots = (spec.gridCols or 7) * (spec.gridRows or 5)
  end

  spec.items = {}
  for i=1, spec.maxSlots do spec.items[i] = nil end

  if self.isClient then
    if not self.vsActivatable then
      self.vsActivatable = VS_Activatable.new(self)
    end
    if spec.triggerNode ~= nil then
      addTrigger(spec.triggerNode, "vsTriggerCallback", self)
    else
      Logging.xmlWarning(xml, "[VehicleStorage] Missing playerTrigger (vehicle.vehicleStorage#playerTrigger)")
    end
  end

  if savegame ~= nil and savegame.key ~= nil then
    
    local baseKey = savegame.key
    local newKey  = baseKey .. "." .. VehicleStorage.MOD_NAME .. ".vehicleStorage"
    local oldKey  = baseKey .. ".vehicleStorageData" 

    if savegame.xmlFile:hasProperty(newKey) then
      self:vsLoadFromXML(savegame.xmlFile, newKey)
    elseif savegame.xmlFile:hasProperty(oldKey) then
      self:vsLoadFromXML(savegame.xmlFile, oldKey)
    end
  end
end

function VehicleStorage:saveToXMLFile(xmlFile, key, usedModNames)
  self:vsSaveToXML(xmlFile, key)
end

function VehicleStorage:onDelete()
  local spec = self.spec_vehicleStorage
  if self.isClient and spec then
    if spec.triggerNode then removeTrigger(spec.triggerNode); spec.triggerNode = nil end
    if self.vsActivatable then
      g_currentMission.activatableObjectsSystem:removeActivatable(self.vsActivatable)
      self.vsActivatable = nil
    end
  end
end

function VehicleStorage:vsGetIsAccessible(farmId)
  return (self:getOwnerFarmId() == farmId) or (farmId == 0)
end

function VehicleStorage:vsTriggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
  if not (onEnter or onLeave) then return end
  if g_localPlayer == nil then return end

  local spec = self.spec_vehicleStorage
  if triggerId ~= spec.triggerNode then return end
  if otherActorId ~= g_localPlayer.rootNode then return end

  if onEnter then
    if Platform.gameplay and Platform.gameplay.autoActivateTrigger and self.vsActivatable:getIsActivatable() then
      self.vsActivatable:run()
    else
      g_currentMission.activatableObjectsSystem:addActivatable(self.vsActivatable)
    end
  end

  if onLeave then
    g_currentMission.activatableObjectsSystem:removeActivatable(self.vsActivatable)
    self:vsClose()
  end
end

VS_Activatable = {}
local VS_Activatable_mt = Class(VS_Activatable)
function VS_Activatable.new(vehicle)
  local self = setmetatable({}, VS_Activatable_mt)
  self.vehicle = vehicle
  self.activateText =
      (g_i18n and g_i18n:getText("ui_storage_pressOpen"))
      or (g_i18n and g_i18n:getText("l10n_ui_vstore_title"))
      or "Открыть склад"
  return self
end

function VS_Activatable:getIsActivatable()
  if g_localPlayer == nil then return false end
  if g_localPlayer.getIsInVehicle and g_localPlayer:getIsInVehicle() then return false end
  local farmId = (g_currentMission and g_currentMission:getFarmId()) or 1
  return self.vehicle and self.vehicle.vsGetIsAccessible and self.vehicle:vsGetIsAccessible(farmId) or false
end

function VS_Activatable:run()
  if self.vehicle then
    self.vehicle:vsOpen()
  end
end

function VehicleStorage:vsOpen()
  if not self.isClient then return end

  if VehicleStorageHUD then
    local mod = g_modManager and g_modManager:getModByName(VehicleStorage.MOD_NAME)
    if mod ~= nil and mod.modDir ~= nil then
      VehicleStorageHUD.modDirectory = mod.modDir
    end
  end
  if VehicleStorageHUD and VehicleStorageHUD.register then
    local ok = VehicleStorageHUD.register()
    if not ok then
      print("[VehicleStorage] Cannot open HUD: register() failed (check VehicleStorageHUD.xml path)")
      return
    end
  else
    print("[VehicleStorage] Cannot open HUD: VehicleStorageHUD.register is missing")
    return
  end
  if VehicleStorageHUD.INSTANCE ~= nil then
    VehicleStorageHUD.INSTANCE:setVehicle(self)
    g_gui:showGui("VehicleStorageHUD")
  end
end

function VehicleStorage:vsClose()
  if not self.isClient then return end
  if g_gui.currentGuiName == "VehicleStorageHUD" then
    g_gui:showGui("")
  end
end

function VehicleStorage:vsToggle()
  if self:vsIsOpen() then self:vsClose() else self:vsOpen() end
end

function VehicleStorage:vsIsOpen()
  if not self.isClient then return false end
  return g_gui and g_gui.currentGuiName == "VehicleStorageHUD"
end

function VehicleStorage:vsPutFromInventory(item, leftSlotIndex)
  local spec = self.spec_vehicleStorage
  if not item then return false end
  local free = _firstFreeSlot(spec.items, spec.maxSlots)
  if not free then return false end

  spec.items[free] = {
    uid=_genId(),
    id  = item.id,
    name = item.name or "",
    desc = item.desc or "",
    icon = item.icon or "",
    kind = item.kind or "food",
    hungerGain = tonumber(item.hungerGain or 0) or 0,
    vigorGain  = tonumber(item.vigorGain  or 0) or 0,
    effectId   = item.effectId or "",
    effectsFile= item.effectsFile or ""
  }
  self:vsMarkDirty()

  if Inventory and Inventory.API and Inventory.API.removeAt then
    Inventory.API.removeAt(leftSlotIndex)
  end

  self:vsRefreshBothHUDs()
  return true
end

function VehicleStorage:vsTakeToInventory(rightSlotIndex)
  local spec = self.spec_vehicleStorage
  local it = spec.items[rightSlotIndex]
  if not it then return false end

  local ok = false
  if Inventory and Inventory.API and Inventory.API.addItemInstance then
    ok = Inventory.API.addItemInstance({
      id=it.id, name=it.name, desc=it.desc, icon=it.icon, kind=it.kind,
      hungerGain=it.hungerGain, vigorGain=it.vigorGain,
      effectId=it.effectId, effectsFile=it.effectsFile
    })
  end
  if not ok then return false end
  spec.items[rightSlotIndex] = nil
  self:vsMarkDirty()
  self:vsRefreshBothHUDs()
  return true
end

function VehicleStorage:vsRefreshBothHUDs()
  if Inventory and Inventory.Hud and Inventory.Hud.refresh then
    Inventory.Hud:refresh()
  end
  if VehicleStorageHUD and VehicleStorageHUD.INSTANCE then
    VehicleStorageHUD.INSTANCE:refreshData()
  end
end

function VehicleStorage:vsSaveToXML(xmlFile, key)
  local spec = self.spec_vehicleStorage
  if spec.itemSlot and spec.itemSlot > 0 then
    xmlFile:setValue(key .. "#itemSlot", spec.itemSlot)
  else
    xmlFile:setValue(key .. "#cols", spec.gridCols or 7)
    xmlFile:setValue(key .. "#rows", spec.gridRows or 5)
  end

  local idx = 0
  for i=1, spec.maxSlots do
    local it = spec.items[i]
    if it then
      local sKey = string.format("%s.slots.slot(%d)", key, idx)
      xmlFile:setValue(sKey.."#index",       i)
      xmlFile:setValue(sKey.."#id",          tostring(it.id or it.uid))
      xmlFile:setValue(sKey.."#name",        tostring(it.name or ""))
      xmlFile:setValue(sKey.."#desc",        tostring(it.desc or ""))
      xmlFile:setValue(sKey.."#icon",        tostring(it.icon or ""))
      xmlFile:setValue(sKey.."#kind",        tostring(it.kind or "food"))
      xmlFile:setValue(sKey.."#hungerGain",  tonumber(it.hungerGain or 0))
      xmlFile:setValue(sKey.."#vigorGain",   tonumber(it.vigorGain  or 0))
      xmlFile:setValue(sKey.."#effectId",    tostring(it.effectId or ""))
      xmlFile:setValue(sKey.."#effectsFile", tostring(it.effectsFile or ""))
      idx = idx + 1
    end
  end
end

function VehicleStorage:vsLoadFromXML(xmlFile, key)
  local spec = self.spec_vehicleStorage
  if not xmlFile:hasProperty(key) then return end
  local fromItemSlot = xmlFile:getValue(key .. "#itemSlot", 0)
  if fromItemSlot and fromItemSlot > 0 then
    spec.itemSlot = fromItemSlot
    spec.maxSlots = fromItemSlot
  else
    spec.gridCols = xmlFile:getValue(key .. "#cols", spec.gridCols or 7)
    spec.gridRows = xmlFile:getValue(key .. "#rows", spec.gridRows or 5)
    spec.maxSlots = (spec.gridCols or 7) * (spec.gridRows or 5)
  end
  spec.items = {}
  for i=1, spec.maxSlots do spec.items[i] = nil end
  local i = 0
  while true do
    local sKey = string.format("%s.slots.slot(%d)", key, i)
    if not xmlFile:hasProperty(sKey) then break end
    local slotIndex  = xmlFile:getValue(sKey .. "#index")
    local id         = xmlFile:getValue(sKey .. "#id") or _genId()
    local name       = xmlFile:getValue(sKey .. "#name") or ""
    local desc       = xmlFile:getValue(sKey .. "#desc") or ""
    local icon       = xmlFile:getValue(sKey .. "#icon") or ""
    local kind       = xmlFile:getValue(sKey .. "#kind") or "food"
    local hunger     = xmlFile:getValue(sKey .. "#hungerGain") or 0
    local vigor      = xmlFile:getValue(sKey .. "#vigorGain")  or 0
    local effectId   = xmlFile:getValue(sKey .. "#effectId") or ""
    local effectsFile= xmlFile:getValue(sKey .. "#effectsFile") or ""
    if slotIndex and slotIndex >= 1 and slotIndex <= spec.maxSlots then
      spec.items[slotIndex] = {
        uid=_genId(), id=id, name=name, desc=desc, icon=icon, kind=kind,
        hungerGain=hunger, vigorGain=vigor, effectId=effectId, effectsFile=effectsFile
      }
    end
    i = i + 1
  end
  if VehicleStorageHUD and VehicleStorageHUD.INSTANCE and g_gui and g_gui.currentGuiName == "VehicleStorageHUD" then
    VehicleStorageHUD.INSTANCE:refreshData()
  end
end

function VehicleStorage:onWriteStream(streamId, connection)
  local spec = self.spec_vehicleStorage
  local max = math.min(255, tonumber(spec.maxSlots or 0) or 0)
  streamWriteUInt8(streamId, max)
  for i=1, max do
    local it = spec.items[i]
    streamWriteBool(streamId, it ~= nil)
    if it then
      streamWriteString(streamId, tostring(it.id or ""))
      streamWriteString(streamId, tostring(it.name or ""))
      streamWriteString(streamId, tostring(it.desc or ""))
      streamWriteString(streamId, tostring(it.icon or ""))
      streamWriteString(streamId, tostring(it.kind or "food"))
      streamWriteInt16(streamId, tonumber(it.hungerGain or 0))
      streamWriteInt16(streamId, tonumber(it.vigorGain  or 0))
      streamWriteString(streamId, tostring(it.effectId or ""))
      streamWriteString(streamId, tostring(it.effectsFile or ""))
    end
  end
end

function VehicleStorage:onReadStream(streamId, connection)
  local spec = self.spec_vehicleStorage
  local max = streamReadUInt8(streamId)
  spec.maxSlots = max
  spec.items = {}
  for i=1, max do spec.items[i] = nil end
  for i=1, max do
    local has = streamReadBool(streamId)
    if has then
      local it = {
        uid=_genId(),
        id         = streamReadString(streamId),
        name       = streamReadString(streamId),
        desc       = streamReadString(streamId),
        icon       = streamReadString(streamId),
        kind       = streamReadString(streamId),
        hungerGain = streamReadInt16(streamId) or 0,
        vigorGain  = streamReadInt16(streamId) or 0,
        effectId   = streamReadString(streamId),
        effectsFile= streamReadString(streamId),
      }
      spec.items[i] = it
    end
  end
  if VehicleStorageHUD and VehicleStorageHUD.INSTANCE and g_gui and g_gui.currentGuiName == "VehicleStorageHUD" then
    VehicleStorageHUD.INSTANCE:refreshData()
  end
end

function VehicleStorage:vsMarkDirty()
  if self.setDirty ~= nil then self:setDirty() end
end
