-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 12.11.2025

PlaceableStorage = {}
PlaceableStorage.MOD_NAME = g_currentModName or "FS25_liveFarmer"

-- Регистрация схемы XML
function PlaceableStorage.initSpecialization()
    local schema = Placeable.xmlSchema
    schema:setXMLSpecializationType("PlaceableStorage")
    local base = "placeable.placeableStorage"
    schema:register(XMLValueType.NODE_INDEX, base .. "#playerTrigger", "Триггер для активации GUI")
    schema:register(XMLValueType.INT,        base .. "#itemSlot",      "Максимум слотов (новый)", 0)
    schema:register(XMLValueType.STRING,     base .. "#title",         "Заголовок HUD (raw или $l10n_*)", "")
    schema:register(XMLValueType.INT,        base .. "#cols",          "Колонок (устар.)", 7)
    schema:register(XMLValueType.INT,        base .. "#rows",          "Рядов (устар.)", 5)
    schema:register(XMLValueType.BOOL,       base .. "#onlyOwner",     "Только владелец может открыть", true)
    schema:setXMLSpecializationType()

    local schemaSave = Placeable.xmlSchemaSavegame
    local nsRoot = string.format("placeables.placeable(?).%s.placeableStorage", PlaceableStorage.MOD_NAME)
    schemaSave:register(XMLValueType.INT, nsRoot .. "#itemSlot", "Сохранённый лимит слотов")
    schemaSave:register(XMLValueType.INT, nsRoot .. "#cols",     "Сохранённые колонки (легаси)")
    schemaSave:register(XMLValueType.INT, nsRoot .. "#rows",     "Сохранённые ряды (легаси)")

    local nsSlot = nsRoot .. ".slots.slot(?)"
    schemaSave:register(XMLValueType.INT,    nsSlot .. "#index",       "Индекс слота 1..N")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#id",          "ID/uid предмета")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#name",        "Имя (raw или $l10n)")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#desc",        "Описание")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#icon",        "Путь к иконке")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#kind",        "Вид предмета")
    schemaSave:register(XMLValueType.INT,    nsSlot .. "#hungerGain",  "Сытость +")
    schemaSave:register(XMLValueType.INT,    nsSlot .. "#vigorGain",   "Бодрость +")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#effectId",    "ID эффекта")
    schemaSave:register(XMLValueType.STRING, nsSlot .. "#effectsFile", "Файл эффектов")
end

function PlaceableStorage.prerequisitesPresent(_)
    return true
end

function PlaceableStorage.registerFunctions(placeableType)
    SpecializationUtil.registerFunction(placeableType, "psTriggerCallback",     PlaceableStorage.psTriggerCallback)
    SpecializationUtil.registerFunction(placeableType, "psGetIsAccessible",     PlaceableStorage.psGetIsAccessible)
    SpecializationUtil.registerFunction(placeableType, "psPutFromInventory",    PlaceableStorage.psPutFromInventory)
    SpecializationUtil.registerFunction(placeableType, "psTakeToInventory",     PlaceableStorage.psTakeToInventory)
    SpecializationUtil.registerFunction(placeableType, "psOpen",                PlaceableStorage.psOpen)
    SpecializationUtil.registerFunction(placeableType, "psClose",               PlaceableStorage.psClose)
    SpecializationUtil.registerFunction(placeableType, "psToggle",              PlaceableStorage.psToggle)
    SpecializationUtil.registerFunction(placeableType, "psIsOpen",              PlaceableStorage.psIsOpen)
    SpecializationUtil.registerFunction(placeableType, "psSaveToXML",           PlaceableStorage.psSaveToXML)
    SpecializationUtil.registerFunction(placeableType, "psLoadFromXML",         PlaceableStorage.psLoadFromXML)
    SpecializationUtil.registerFunction(placeableType, "psRefreshBothHUDs",     PlaceableStorage.psRefreshBothHUDs)
    SpecializationUtil.registerFunction(placeableType, "psMarkDirty",           PlaceableStorage.psMarkDirty)
end

function PlaceableStorage.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",        PlaceableStorage)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",      PlaceableStorage)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream",  PlaceableStorage)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream", PlaceableStorage)
    SpecializationUtil.registerEventListener(placeableType, "saveToXMLFile", PlaceableStorage)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement", PlaceableStorage)
end

-- helpers
local function _genId()
    PlaceableStorage.__uid = (PlaceableStorage.__uid or 0) + 1
    return string.format("pstore_%d_%d", g_time or 0, PlaceableStorage.__uid)
end

local function _firstFreeSlot(tbl, max)
    for i = 1, max do
        if tbl[i] == nil then return i end
    end
    return nil
end

-- lifecycle

function PlaceableStorage:onLoad(savegame)
    local spec = self.spec_placeableStorage or {}
    self.spec_placeableStorage = spec

    local xml  = self.xmlFile
    local base = "placeable.placeableStorage"

    spec.triggerNode = xml:getValue(base .. "#playerTrigger", nil, self.components, self.i3dMappings)
    spec.onlyOwner   = xml:getValue(base .. "#onlyOwner", true)
    spec.itemSlot    = math.max(0, xml:getValue(base .. "#itemSlot", 0))
    spec.gridCols    = xml:getValue(base .. "#cols", 7)
    spec.gridRows    = xml:getValue(base .. "#rows", 5)
    spec.titleRaw    = xml:getValue(base .. "#title", "") or ""

    if spec.itemSlot > 0 then
        spec.maxSlots = spec.itemSlot
    else
        spec.maxSlots = (spec.gridCols or 7) * (spec.gridRows or 5)
    end

    spec.items = {}
    for i = 1, spec.maxSlots do spec.items[i] = nil end

    if self.isClient then
        if not self.psActivatable then
            self.psActivatable = PS_Activatable.new(self)
        end
        if spec.triggerNode ~= nil then
            addTrigger(spec.triggerNode, "psTriggerCallback", self)
        else
            Logging.xmlWarning(xml, "[PlaceableStorage] Missing playerTrigger (placeable.placeableStorage#playerTrigger)")
        end
    end

    if savegame ~= nil and savegame.key ~= nil then
        local baseKey = savegame.key
        local newKey  = string.format("%s.%s.placeableStorage", baseKey, PlaceableStorage.MOD_NAME)
        if savegame.xmlFile:hasProperty(newKey) then
            self:psLoadFromXML(savegame.xmlFile, newKey)
        end
    end
end

function PlaceableStorage:onFinalizePlacement()
end

function PlaceableStorage:saveToXMLFile(xmlFile, key, usedModNames)
    self:psSaveToXML(xmlFile, key)
end

function PlaceableStorage:onDelete()
    local spec = self.spec_placeableStorage
    if self.isClient and spec then
        if spec.triggerNode then removeTrigger(spec.triggerNode); spec.triggerNode = nil end
        if self.psActivatable then
            g_currentMission.activatableObjectsSystem:removeActivatable(self.psActivatable)
            self.psActivatable = nil
        end
    end
end

-- access

function PlaceableStorage:psGetIsAccessible(farmId)
    if self.getOwnerFarmId ~= nil then
        return (self:getOwnerFarmId() == farmId) or (farmId == 0)
    end
    return true
end

function PlaceableStorage:psTriggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
    if not (onEnter or onLeave) then return end
    if g_localPlayer == nil then return end

    local spec = self.spec_placeableStorage
    if triggerId ~= spec.triggerNode then return end
    if otherActorId ~= g_localPlayer.rootNode then return end

    if onEnter then
        if Platform.gameplay and Platform.gameplay.autoActivateTrigger and self.psActivatable:getIsActivatable() then
            self.psActivatable:run()
        else
            g_currentMission.activatableObjectsSystem:addActivatable(self.psActivatable)
        end
    end

    if onLeave then
        g_currentMission.activatableObjectsSystem:removeActivatable(self.psActivatable)
        self:psClose()
    end
end

PS_Activatable = {}
local PS_Activatable_mt = Class(PS_Activatable)
function PS_Activatable.new(placeable)
    local self = setmetatable({}, PS_Activatable_mt)
    self.placeable = placeable
    self.activateText =
        (g_i18n and g_i18n:getText("ui_storage_pressOpen"))
        or (g_i18n and g_i18n:getText("l10n_ui_pstore_title"))
        or "Открыть склад"
    return self
end

function PS_Activatable:getIsActivatable()
    if g_localPlayer == nil then return false end
    if g_localPlayer.getIsInVehicle and g_localPlayer:getIsInVehicle() then return false end
    local farmId = (g_currentMission and g_currentMission:getFarmId()) or 1
    return self.placeable and self.placeable.psGetIsAccessible and self.placeable:psGetIsAccessible(farmId) or false
end

function PS_Activatable:run()
    if self.placeable then
        self.placeable:psOpen()
    end
end

-- HUD
function PlaceableStorage:psOpen()
    if not self.isClient then return end

    if PlaceableStorageHUD then
        local mod = g_modManager and g_modManager:getModByName(PlaceableStorage.MOD_NAME)
        if mod ~= nil and mod.modDir ~= nil then
            PlaceableStorageHUD.modDirectory = mod.modDir
        end
    end

    if PlaceableStorageHUD and PlaceableStorageHUD.register then
        local ok = PlaceableStorageHUD.register()
        if not ok then
            print("[PlaceableStorage] Cannot open HUD: register() failed (check PlaceableStorageHUD.xml path)")
            return
        end
    else
        print("[PlaceableStorage] Cannot open HUD: PlaceableStorageHUD.register is missing")
        return
    end

    if PlaceableStorageHUD.INSTANCE ~= nil then
        PlaceableStorageHUD.INSTANCE:setPlaceable(self)
        g_gui:showGui("PlaceableStorageHUD")
    end
end

function PlaceableStorage:psClose()
    if not self.isClient then return end
    if g_gui.currentGuiName == "PlaceableStorageHUD" then
        g_gui:showGui("")
    end
end

function PlaceableStorage:psToggle()
    if self:psIsOpen() then self:psClose() else self:psOpen() end
end

function PlaceableStorage:psIsOpen()
    if not self.isClient then return false end
    return g_gui and g_gui.currentGuiName == "PlaceableStorageHUD"
end

-- Перемещение предметов
function PlaceableStorage:psPutFromInventory(item, leftSlotIndex)
    local spec = self.spec_placeableStorage
    if not item then return false end
    local free = _firstFreeSlot(spec.items, spec.maxSlots)
    if not free then return false end

    spec.items[free] = {
        uid = _genId(),
        id  = item.id,
        name = item.name or "",
        desc = item.desc or "",
        icon = item.icon or "",
        kind = item.kind or "food",
        hungerGain = tonumber(item.hungerGain or 0) or 0,
        vigorGain  = tonumber(item.vigorGain  or 0) or 0,
        effectId   = item.effectId or "",
        effectsFile= item.effectsFile or ""
    }
    self:psMarkDirty()

    if Inventory and Inventory.API and Inventory.API.removeAt then
        Inventory.API.removeAt(leftSlotIndex)
    end

    self:psRefreshBothHUDs()
    return true
end

function PlaceableStorage:psTakeToInventory(rightSlotIndex)
    local spec = self.spec_placeableStorage
    local it = spec.items[rightSlotIndex]
    if not it then return false end

    local ok = false
    if Inventory and Inventory.API and Inventory.API.addItemInstance then
        ok = Inventory.API.addItemInstance({
            id = it.id, name = it.name, desc = it.desc, icon = it.icon, kind = it.kind,
            hungerGain = it.hungerGain, vigorGain = it.vigorGain,
            effectId = it.effectId, effectsFile = it.effectsFile
        })
    end
    if not ok then return false end

    spec.items[rightSlotIndex] = nil
    self:psMarkDirty()
    self:psRefreshBothHUDs()
    return true
end

function PlaceableStorage:psRefreshBothHUDs()
    if Inventory and Inventory.Hud and Inventory.Hud.refresh then
        Inventory.Hud:refresh()
    end
    if PlaceableStorageHUD and PlaceableStorageHUD.INSTANCE and g_gui and g_gui.currentGuiName == "PlaceableStorageHUD" then
        PlaceableStorageHUD.INSTANCE:refreshData()
    end
end

-- Save/Load
function PlaceableStorage:psSaveToXML(xmlFile, key)
    local spec = self.spec_placeableStorage
    if spec == nil then return end

    local suffix = "." .. PlaceableStorage.MOD_NAME .. ".placeableStorage"
    local myKey
    if type(key) == "string" and #key >= #suffix and key:sub(-#suffix) == suffix then
        myKey = key
    else
        myKey = key .. suffix
    end

    if spec.itemSlot and spec.itemSlot > 0 then
        xmlFile:setValue(myKey .. "#itemSlot", spec.itemSlot)
    else
        xmlFile:setValue(myKey .. "#cols", spec.gridCols or 7)
        xmlFile:setValue(myKey .. "#rows", spec.gridRows or 5)
    end

    local idx = 0
    for i = 1, spec.maxSlots or 0 do
        local it = spec.items[i]
        if it then
            local sKey = string.format("%s.slots.slot(%d)", myKey, idx)
            xmlFile:setValue(sKey.."#index",       i)
            xmlFile:setValue(sKey.."#id",          tostring(it.id or it.uid or ""))
            xmlFile:setValue(sKey.."#name",        tostring(it.name or ""))
            xmlFile:setValue(sKey.."#desc",        tostring(it.desc or ""))
            xmlFile:setValue(sKey.."#icon",        tostring(it.icon or ""))
            xmlFile:setValue(sKey.."#kind",        tostring(it.kind or "food"))
            xmlFile:setValue(sKey.."#hungerGain",  tonumber(it.hungerGain or 0))
            xmlFile:setValue(sKey.."#vigorGain",   tonumber(it.vigorGain  or 0))
            xmlFile:setValue(sKey.."#effectId",    tostring(it.effectId or ""))
            xmlFile:setValue(sKey.."#effectsFile", tostring(it.effectsFile or ""))
            idx = idx + 1
        end
    end
end

function PlaceableStorage:psLoadFromXML(xmlFile, key)
    local spec = self.spec_placeableStorage
    if spec == nil then return end

    local suffix = "." .. PlaceableStorage.MOD_NAME .. ".placeableStorage"
    local myKey = key
    if not xmlFile:hasProperty(myKey) then
        local altKey = key .. suffix
        if xmlFile:hasProperty(altKey) then
            myKey = altKey
        end
    end
    if not xmlFile:hasProperty(myKey) then
        return
    end

    local fromItemSlot = xmlFile:getValue(myKey .. "#itemSlot", 0)
    if fromItemSlot and fromItemSlot > 0 then
        spec.itemSlot = fromItemSlot
        spec.maxSlots = fromItemSlot
    else
        spec.gridCols = xmlFile:getValue(myKey .. "#cols", spec.gridCols or 7)
        spec.gridRows = xmlFile:getValue(myKey .. "#rows", spec.gridRows or 5)
        spec.maxSlots = (spec.gridCols or 7) * (spec.gridRows or 5)
    end

    spec.items = {}
    for i = 1, spec.maxSlots do spec.items[i] = nil end

    local i = 0
    while true do
        local sKey = string.format("%s.slots.slot(%d)", myKey, i)
        if not xmlFile:hasProperty(sKey) then break end
        local slotIndex  = xmlFile:getValue(sKey .. "#index")
        local id         = xmlFile:getValue(sKey .. "#id") or ""
        local name       = xmlFile:getValue(sKey .. "#name") or ""
        local desc       = xmlFile:getValue(sKey .. "#desc") or ""
        local icon       = xmlFile:getValue(sKey .. "#icon") or ""
        local kind       = xmlFile:getValue(sKey .. "#kind") or "food"
        local hunger     = xmlFile:getValue(sKey .. "#hungerGain") or 0
        local vigor      = xmlFile:getValue(sKey .. "#vigorGain")  or 0
        local effectId   = xmlFile:getValue(sKey .. "#effectId") or ""
        local effectsFile= xmlFile:getValue(sKey .. "#effectsFile") or ""
        if slotIndex and slotIndex >= 1 and slotIndex <= (spec.maxSlots or 0) then
            spec.items[slotIndex] = {
                uid=("pstore_"..tostring(getTimeSec()).."_"..tostring(i)),
                id=id, name=name, desc=desc, icon=icon, kind=kind,
                hungerGain=hunger, vigorGain=vigor, effectId=effectId, effectsFile=effectsFile
            }
        end
        i = i + 1
    end

    if PlaceableStorageHUD and PlaceableStorageHUD.INSTANCE and g_gui and g_gui.currentGuiName == "PlaceableStorageHUD" then
        PlaceableStorageHUD.INSTANCE:refreshData()
    end
end

-- net sync (SP/MP)
function PlaceableStorage:onWriteStream(streamId, connection)
    local spec = self.spec_placeableStorage
    local max = math.min(255, tonumber(spec.maxSlots or 0) or 0)
    streamWriteUInt8(streamId, max)
    for i=1, max do
        local it = spec.items[i]
        streamWriteBool(streamId, it ~= nil)
        if it then
            streamWriteString(streamId, tostring(it.id or ""))
            streamWriteString(streamId, tostring(it.name or ""))
            streamWriteString(streamId, tostring(it.desc or ""))
            streamWriteString(streamId, tostring(it.icon or ""))
            streamWriteString(streamId, tostring(it.kind or "food"))
            streamWriteInt16(streamId, tonumber(it.hungerGain or 0))
            streamWriteInt16(streamId, tonumber(it.vigorGain  or 0))
            streamWriteString(streamId, tostring(it.effectId or ""))
            streamWriteString(streamId, tostring(it.effectsFile or ""))
        end
    end
end

function PlaceableStorage:onReadStream(streamId, connection)
    local spec = self.spec_placeableStorage
    local max = streamReadUInt8(streamId)
    spec.maxSlots = max
    spec.items = {}
    for i=1, max do spec.items[i] = nil end

    for i=1, max do
        local has = streamReadBool(streamId)
        if has then
            local it = {
                uid=_genId(),
                id         = streamReadString(streamId),
                name       = streamReadString(streamId),
                desc       = streamReadString(streamId),
                icon       = streamReadString(streamId),
                kind       = streamReadString(streamId),
                hungerGain = streamReadInt16(streamId) or 0,
                vigorGain  = streamReadInt16(streamId) or 0,
                effectId   = streamReadString(streamId),
                effectsFile= streamReadString(streamId),
            }
            spec.items[i] = it
        end
    end

    if PlaceableStorageHUD and PlaceableStorageHUD.INSTANCE and g_gui and g_gui.currentGuiName == "PlaceableStorageHUD" then
        PlaceableStorageHUD.INSTANCE:refreshData()
    end
end

function PlaceableStorage:psMarkDirty()
    if self.setDirty ~= nil then self:setDirty() end
end
