-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 12.11.2025

PlaceableMusicCenter = {}
PlaceableMusicCenter.modName      = g_currentModName or "livefarmer"
PlaceableMusicCenter.modDirectory = g_currentModDirectory or ""

local LOG = "[MusicCenter] "

-- ====== режимы ======
PlaceableMusicCenter.MODE_SINGLE_ONCE = 1  -- 1) один выбранный и стоп
PlaceableMusicCenter.MODE_SINGLE_LOOP = 2  -- 2) один выбранный по кругу
PlaceableMusicCenter.MODE_ALL_ONCE    = 3  -- 3) все по очереди и стоп
PlaceableMusicCenter.MODE_ALL_LOOP    = 4  -- 4) все по очереди и с начала

-- ====== надёжные помощники ======
local function _clamp(v, lo, hi)
    v  = tonumber(v)  or 0
    lo = tonumber(lo) or v
    hi = tonumber(hi) or v
    if v < lo then return lo end
    if v > hi then return hi end
    return v
end

local function _l10n(s)
    if not s or s=="" then return "" end
    if s:sub(1,1) ~= "$" then return s end
    local k = s:sub(2)
    if g_i18n and g_i18n:hasText(k) then return g_i18n:getText(k) end
    if k:sub(1,5)=="l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then
        return g_i18n:getText(k:sub(6))
    end
    return s
end

-- ================== Activatable (R) ==================
MC_Activatable = {}
local MC_Activatable_mt = Class(MC_Activatable)

function MC_Activatable.new(placeable)
    local self = setmetatable({}, MC_Activatable_mt)
    self.placeable    = placeable
    self.activateText = (g_i18n and g_i18n:getText("ui_mc_openHint")) or "Открыть музыкальный центр (R)"
    return self
end

function MC_Activatable:getIsActivatable()
    if g_localPlayer == nil then return false end
    if g_localPlayer.getIsInVehicle and g_localPlayer:getIsInVehicle() then return false end
    return self.placeable ~= nil
end

function MC_Activatable:run()
    if not g_gui then return end
    if MusicCenterHUD and MusicCenterHUD.register then
        MusicCenterHUD.register()
    else
        print(LOG .. "HUD module not found"); return
    end
    local guiDef = g_gui.guis and g_gui.guis["MusicCenterHUD"]
    if guiDef and guiDef.target then
        guiDef.target.placeable = self.placeable
    end
    g_gui:showDialog("MusicCenterHUD")
end

-- ================== Регистрация в Placeable ==================
function PlaceableMusicCenter.prerequisitesPresent(_)
    return true
end

function PlaceableMusicCenter.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onLoad",               PlaceableMusicCenter)
    SpecializationUtil.registerEventListener(placeableType, "onDelete",             PlaceableMusicCenter)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement",  PlaceableMusicCenter)
    SpecializationUtil.registerEventListener(placeableType, "onUpdate",             PlaceableMusicCenter)
    SpecializationUtil.registerEventListener(placeableType, "saveToXMLFile",        PlaceableMusicCenter)
    SpecializationUtil.registerEventListener(placeableType, "readStream",           PlaceableMusicCenter)
    SpecializationUtil.registerEventListener(placeableType, "writeStream",          PlaceableMusicCenter)
end

function PlaceableMusicCenter.registerFunctions(placeableType)
    SpecializationUtil.registerFunction(placeableType, "mcTriggerCallback",  PlaceableMusicCenter.mcTriggerCallback)
    SpecializationUtil.registerFunction(placeableType, "mcServerPlay",       PlaceableMusicCenter.mcServerPlay)
    SpecializationUtil.registerFunction(placeableType, "mcServerStop",       PlaceableMusicCenter.mcServerStop)
    SpecializationUtil.registerFunction(placeableType, "mcEnsureBuiltFor",   PlaceableMusicCenter.mcEnsureBuiltFor)
    SpecializationUtil.registerFunction(placeableType, "mcPlayLocal",        PlaceableMusicCenter.mcPlayLocal)
    SpecializationUtil.registerFunction(placeableType, "mcStopLocal",        PlaceableMusicCenter.mcStopLocal)

    -- режимы
    SpecializationUtil.registerFunction(placeableType, "mcServerSetMode",    PlaceableMusicCenter.mcServerSetMode)
    SpecializationUtil.registerFunction(placeableType, "mcSetModeLocal",     PlaceableMusicCenter.mcSetModeLocal)
end

function PlaceableMusicCenter.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.STRING,     basePath..".musicCenter#title",       "Title")
    schema:register(XMLValueType.NODE_INDEX, basePath..".musicCenter#triggerNode", "Trigger node (R to open GUI)")

    local tk = basePath..".musicCenter.tracks.track(?)"
    schema:register(XMLValueType.STRING,     tk.."#name",        "Display name (l10n ok)")
    schema:register(XMLValueType.STRING,     tk.."#file",        "Sound file path (ogg/wav)")
    schema:register(XMLValueType.BOOL,       tk.."#is2D",        "Force 2D", false)
    schema:register(XMLValueType.NODE_INDEX, tk.."#linkNode",    "Link node (3D)")
    schema:register(XMLValueType.FLOAT,      tk.."#innerRadius", "Inner radius (3D)", 4)
    schema:register(XMLValueType.FLOAT,      tk.."#outerRadius", "Outer radius (3D)", 60)
    schema:register(XMLValueType.FLOAT,      tk.."#volume",      "Volume 0..1", 1.0)
    schema:register(XMLValueType.FLOAT,      tk.."#pitch",       "Pitch (speed)", 1.0)
    schema:register(XMLValueType.BOOL,       tk.."#loop",        "Loop", true)
end

function PlaceableMusicCenter.initSpecialization()
    local save = Placeable.xmlSchemaSavegame
    local ns   = (PlaceableMusicCenter.modName or "livefarmer") .. ".musicCenter"

    save:register(XMLValueType.BOOL, "placeables.placeable(?)."..ns.."#playing",      "Сейчас играет", false)
    save:register(XMLValueType.INT,  "placeables.placeable(?)."..ns.."#currentIndex", "Индекс трека (1..N; 0=нет)", 0)
    save:register(XMLValueType.INT,  "placeables.placeable(?)."..ns.."#mode",         "Режим воспроизведения 1..4", PlaceableMusicCenter.MODE_SINGLE_ONCE)
    save:register(XMLValueType.BOOL, "placeables.placeable(?)."..ns.."#resumeSeq",    "Продолжить после загрузки (для 3/4)", false)
end

-- ================== onLoad ==================
function PlaceableMusicCenter:onLoad(savegame)
    self.spec_musicCenter = self.spec_musicCenter or {}
    local spec = self.spec_musicCenter
    local xml  = self.xmlFile

    spec.mode = PlaceableMusicCenter.MODE_SINGLE_ONCE
    spec.endTimeMs   = nil
    spec.playStartMs = nil
    spec.playLenMs   = nil

    -- Заголовок
    local ttl = xml:getValue("placeable.musicCenter#title") or "$ui_mc_title"
    spec.title = _l10n(ttl)

    -- Триггер
    spec.triggerNode = xml:getValue("placeable.musicCenter#triggerNode", nil, self.components, self.i3dMappings)
    if spec.triggerNode ~= nil then
        addTrigger(spec.triggerNode, "mcTriggerCallback", self)
    else
        Logging.xmlWarning(self.xmlFile, LOG.."Missing trigger node 'placeable.musicCenter#triggerNode'")
    end

    -- Треки
    spec.tracks = {}
    local i=0
    while true do
        local key = ("placeable.musicCenter.tracks.track(%d)"):format(i)
        if not xml:hasProperty(key) then break end

        local name     = _l10n(xml:getValue(key.."#name") or ("Track "..(i+1)))
        local fileRel  = xml:getValue(key.."#file")
        local file     = fileRel and Utils.getFilename(fileRel, PlaceableMusicCenter.modDirectory) or nil
        local is2D     = xml:getValue(key.."#is2D", false)
        local linkNode = xml:getValue(key.."#linkNode", nil, self.components, self.i3dMappings)
        local inner    = xml:getValue(key.."#innerRadius", 4)
        local outer    = xml:getValue(key.."#outerRadius", 60)
        local volume   = xml:getValue(key.."#volume", 1.0)
        local pitch    = xml:getValue(key.."#pitch", 1.0)
        local loop     = xml:getValue(key.."#loop", true)

        table.insert(spec.tracks, {
            index = #spec.tracks+1,
            name  = name,
            file  = file,
            is2D  = is2D == true,
            linkNode = linkNode,
            innerRadius = math.max(0.1, inner or 4),
            outerRadius = math.max(0.1, outer or 60),
            volume = math.max(0, math.min(1, volume or 1)),
            pitch  = math.max(0.1, pitch or 1),
            loop   = (loop ~= false),

            node    = nil,
            sample  = nil,
            playing = false,
            lengthSec = nil, -- кэш длительности
        })
        i=i+1
    end

    spec.currentIndex = 0
    spec.playing      = false
    spec.resumeSeq    = false
    spec._inTrigger   = {}
    spec._activatable = spec._activatable or MC_Activatable.new(self)

    if g_gui and (not (g_gui.guis and g_gui.guis["MusicCenterHUD"])) and MusicCenterHUD and MusicCenterHUD.register then
        MusicCenterHUD.register()
    end

    if savegame and savegame.xmlFile then
        local ns = (PlaceableMusicCenter.modName or "livefarmer") .. ".musicCenter"
        local keyBase = savegame.key .. "." .. ns
        spec.playing      = savegame.xmlFile:getValue(keyBase.."#playing", false)
        spec.currentIndex = math.max(0, savegame.xmlFile:getValue(keyBase.."#currentIndex", 0))
        spec.mode         = _clamp(savegame.xmlFile:getValue(keyBase.."#mode", PlaceableMusicCenter.MODE_SINGLE_ONCE), 1, 4)
        spec.resumeSeq    = (savegame.xmlFile:getValue(keyBase.."#resumeSeq", false) == true)

        -- Если sequential-режим и надо продолжить — включим автозапуск
        if (spec.mode == PlaceableMusicCenter.MODE_ALL_ONCE or spec.mode == PlaceableMusicCenter.MODE_ALL_LOOP) then
            if spec.playing or spec.resumeSeq then
                spec._resumeOnFinalize = true
                if spec.currentIndex < 1 then spec.currentIndex = 1 end
                if spec.currentIndex > #spec.tracks then spec.currentIndex = #spec.tracks end
            end
        elseif spec.playing then
            spec._resumeOnFinalize = true
        end
    end
end

function PlaceableMusicCenter:onFinalizePlacement()
    local spec = self.spec_musicCenter
    if not spec then return end
    if g_server == nil then return end

    if spec._resumeOnFinalize and (spec.currentIndex or 0) > 0 then
        self:mcServerPlay(spec.currentIndex)
        spec._resumeOnFinalize = nil
    elseif spec.currentIndex > 0 and spec.playing then
        self:mcServerPlay(spec.currentIndex)
    end
end

function PlaceableMusicCenter:onDelete()
    local spec = self.spec_musicCenter; if not spec then return end
    self:mcStopLocal(true)
    for _, tr in ipairs(spec.tracks or {}) do
        if tr.node ~= nil then
            pcall(function() delete(tr.node) end)
            tr.node = nil
        end
        if tr.sample ~= nil then
            pcall(function() delete(tr.sample) end)
            tr.sample = nil
        end
    end
end

-- ================== Trigger ==================
function PlaceableMusicCenter:mcTriggerCallback(triggerId, otherId, onEnter, onLeave, _onStay, shapeId)
    local spec = self.spec_musicCenter; if not spec then return end
    if g_currentMission == nil or g_currentMission.activatableObjectsSystem == nil then return end

    local isPlayer = false
    if g_localPlayer ~= nil then
        if otherId == g_localPlayer.rootNode or otherId == g_localPlayer.playerNode then
            isPlayer = true
        else
            local parent = getParent(otherId)
            while parent ~= 0 do
                if parent == g_localPlayer.rootNode then isPlayer=true; break end
                parent = getParent(parent)
            end
        end
    end
    if not isPlayer then return end

    local rid = (shapeId or otherId or 0)

    if onEnter then
        spec._inTrigger[rid] = (spec._inTrigger[rid] or 0) + 1
        if spec._activatable then
            spec._activatable.activateText = string.format("%s: %s",
                _l10n("$ui_mc_openHint"), spec.title or "")
            g_currentMission.activatableObjectsSystem:addActivatable(spec._activatable)
        end
    end

    if onLeave then
        spec._inTrigger[rid] = math.max(0, (spec._inTrigger[rid] or 1) - 1)
        local any=false; for _,cnt in pairs(spec._inTrigger) do if (cnt or 0)>0 then any=true; break end end
        if not any and spec._activatable then
            g_currentMission.activatableObjectsSystem:removeActivatable(spec._activatable)
        end
    end
end

-- ================== Network ==================
function PlaceableMusicCenter:writeStream(streamId, _connection)
    local spec = self.spec_musicCenter or {}
    streamWriteBool(streamId, spec.playing == true)
    streamWriteUIntN(streamId, math.max(0, spec.currentIndex or 0), 16)
    streamWriteUIntN(streamId, _clamp((spec.mode or 1), 1, 4), 3)
    streamWriteBool(streamId, spec.resumeSeq == true)
end

function PlaceableMusicCenter:readStream(streamId, _connection)
    local spec = self.spec_musicCenter or {}
    spec.playing      = streamReadBool(streamId)
    spec.currentIndex = streamReadUIntN(streamId, 16)
    spec.mode         = streamReadUIntN(streamId, 3)
    spec.resumeSeq    = streamReadBool(streamId)

    if spec.currentIndex > 0 and (spec.playing or spec.resumeSeq) then
        self:mcPlayLocal(spec.currentIndex)
    else
        self:mcStopLocal(false)
    end
end

-- ================== Save ==================
function PlaceableMusicCenter:saveToXMLFile(xmlFile, key)
    local spec = self.spec_musicCenter
    if not spec then return end
    xmlFile:setValue(key.."#playing",      spec.playing == true)
    xmlFile:setValue(key.."#currentIndex", math.max(0, spec.currentIndex or 0))
    xmlFile:setValue(key.."#mode",         _clamp((spec.mode or 1), 1, 4))
    local isSeq = (spec.mode == PlaceableMusicCenter.MODE_ALL_ONCE or spec.mode == PlaceableMusicCenter.MODE_ALL_LOOP)
    xmlFile:setValue(key.."#resumeSeq",    (isSeq and (spec.playing or spec.resumeSeq)) == true)
end

-- ================== Update ==================
function PlaceableMusicCenter:onUpdate(dt)
    local spec = self.spec_musicCenter
    if not spec then return end
    if g_server == nil then return end

    local playFn = self.mcServerPlay
    local stopFn = self.mcServerStop
    if type(playFn) ~= "function" or type(stopFn) ~= "function" then
        return
    end

    local now = g_time or 0
    if spec.playing and spec.endTimeMs ~= nil and now >= spec.endTimeMs then
        local mode  = spec.mode or PlaceableMusicCenter.MODE_SINGLE_ONCE
        local cur   = spec.currentIndex or 0
        local total = #spec.tracks

        local function nextIndexWrap()
            if total <= 0 then return 0 end
            if cur <= 0 then return 1 end
            if cur < total then return cur + 1 end
            return 1
        end

        if mode == PlaceableMusicCenter.MODE_SINGLE_ONCE then
            stopFn(self)

        elseif mode == PlaceableMusicCenter.MODE_SINGLE_LOOP then
            playFn(self, cur)

        elseif mode == PlaceableMusicCenter.MODE_ALL_ONCE then
            if cur < total then
                playFn(self, cur + 1)
            else
                spec.resumeSeq = false
                stopFn(self)
            end

        elseif mode == PlaceableMusicCenter.MODE_ALL_LOOP then
            playFn(self, nextIndexWrap())
        end
    end
end

-- ================== Звук ==================
local function _calcDurationMs(sample, pitch)
    if sample == nil or sample == 0 then return nil end
    if getSampleDuration == nil then return nil end
    local durMs = getSampleDuration(sample)
    if durMs == nil or durMs <= 0 then return nil end
    local p = math.max(0.01, pitch or 1.0)
    return math.floor(durMs / p)
end

function PlaceableMusicCenter:mcEnsureBuiltFor(track)
    if not track or not track.file or track.file=="" then return false end
    if track.sample ~= nil then return true end

    -- 3D вариант
    if not track.is2D then
        local linkNode = track.linkNode or (self.rootNode or 0)
        if linkNode ~= nil and linkNode ~= 0 then
            local nodeName = string.format("MC_LOOP_%s_%d", "track", track.index or 0)
            local sndNode = createAudioSource(nodeName, track.file, track.outerRadius or 60, track.innerRadius or 4, 1, 1)
            if sndNode ~= 0 and sndNode ~= nil then
                link(linkNode, sndNode); setTranslation(sndNode, 0,0,0)
                local s = getAudioSourceSample(sndNode)
                if s ~= nil and s ~= 0 then
                    track.sample = s
                    track.node   = sndNode
                    if setSamplePitch ~= nil then setSamplePitch(track.sample, track.pitch or 1.0) end
                    return true
                else
                    delete(sndNode)
                end
            end
        end
        track.is2D = true
    end

    -- 2D вариант
    local s = createSample("MC_2D_"..tostring(track.index or 0))
    if s == 0 or not loadSample(s, track.file, true) then
        if s ~= 0 then delete(s) end
        return false
    end
    track.sample = s
    if setSamplePitch ~= nil then setSamplePitch(track.sample, track.pitch or 1.0) end
    return true
end

local function _applyLoopingForMode(sample, mode)
    if setSampleLoopCount == nil or sample == nil then return end
    local wantInfinite =
        (mode == PlaceableMusicCenter.MODE_SINGLE_LOOP) or
        (mode == PlaceableMusicCenter.MODE_ALL_LOOP)
    if wantInfinite then
        setSampleLoopCount(sample, 0) -- бесконечно
    else
        setSampleLoopCount(sample, 1) -- один раз (важно для последовательных режимов)
    end
end

function PlaceableMusicCenter:mcPlayLocal(index)
    local spec = self.spec_musicCenter; if not spec then return end
    index = math.max(1, math.min(index or 1, #spec.tracks))
    local tr = spec.tracks[index]; if not tr then return end
    self:mcStopLocal(false)
    if not self:mcEnsureBuiltFor(tr) then
        print(LOG.."Failed to build sample for track #"..tostring(index))
        return
    end
    _applyLoopingForMode(tr.sample, spec.mode or PlaceableMusicCenter.MODE_SINGLE_ONCE)
    playSample(tr.sample, 0, math.max(0, math.min(1, tr.volume or 1.0)), 0, 0, 0)
    tr.playing = true
    spec.currentIndex = index
    spec.playing      = true
    spec.playStartMs = (g_time or 0)
    local durMs = _calcDurationMs(tr.sample, tr.pitch)
    if durMs == nil or durMs <= 0 then
        durMs = 180000
    end
    spec.playLenMs = durMs
    local mode = spec.mode or PlaceableMusicCenter.MODE_SINGLE_ONCE
    if mode == PlaceableMusicCenter.MODE_SINGLE_LOOP or mode == PlaceableMusicCenter.MODE_ALL_LOOP then
        spec.endTimeMs = nil
    else
        spec.endTimeMs = spec.playStartMs + durMs
    end
    if mode == PlaceableMusicCenter.MODE_ALL_ONCE or mode == PlaceableMusicCenter.MODE_ALL_LOOP then
        spec.resumeSeq = true
    end

    if MusicCenterHUD and MusicCenterHUD.INSTANCE and MusicCenterHUD.INSTANCE.placeable == self then
        MusicCenterHUD.INSTANCE:pullData()
        MusicCenterHUD.INSTANCE:updateButtons()
        MusicCenterHUD.INSTANCE:updateStatus()
    end
end

function PlaceableMusicCenter:mcStopLocal(deleteSamples)
    local spec = self.spec_musicCenter; if not spec then return end

    for _, tr in ipairs(spec.tracks or {}) do
        if tr.sample ~= nil then
            pcall(function() stopSample(tr.sample, 0, 0) end)
        end
        tr.playing = false
        if tr.node ~= nil then
            if entityExists == nil or entityExists(tr.node) then
                delete(tr.node)
            end
            tr.node   = nil
            tr.sample = nil
        elseif deleteSamples and tr.sample ~= nil then
            delete(tr.sample)
            tr.sample = nil
        end
    end
    spec.playing    = false
    spec.endTimeMs  = nil
    spec.playStartMs= nil
    spec.playLenMs  = nil
end

-- ================== Серверные действия ==================
function PlaceableMusicCenter:mcServerPlay(index)
    local spec = self.spec_musicCenter; if not spec then return end
    index = math.max(1, math.min(index or 1, #spec.tracks))
    if g_server ~= nil then
        if MC_SelectTrackEvent and MC_SelectTrackEvent.send then
            MC_SelectTrackEvent.send(self, index)
        end
        self:mcPlayLocal(index)
    end
end

function PlaceableMusicCenter:mcServerStop()
    local spec = self.spec_musicCenter; if not spec then return end
    if g_server ~= nil then
        if MC_StopEvent and MC_StopEvent.send then
            MC_StopEvent.send(self)
        end
        self:mcStopLocal(false)
        spec.resumeSeq = false
    end
end

-- ====== режимы: сетевые сеттеры ======
function PlaceableMusicCenter:mcServerSetMode(mode)
    local spec = self.spec_musicCenter; if not spec then return end
    mode = _clamp(mode or 1, 1, 4)
    if g_server ~= nil then
        if MC_ChangeModeEvent and MC_ChangeModeEvent.send then
            MC_ChangeModeEvent.send(self, mode)
        end
        self:mcSetModeLocal(mode)
    end
end

function PlaceableMusicCenter:mcSetModeLocal(mode)
    local spec = self.spec_musicCenter; if not spec then return end
    spec.mode = _clamp(mode or 1, 1, 4)

    if not (spec.mode == PlaceableMusicCenter.MODE_ALL_ONCE or spec.mode == PlaceableMusicCenter.MODE_ALL_LOOP) then
        spec.resumeSeq = false
    end

    if spec.playing and spec.currentIndex > 0 then
        local tr = spec.tracks[spec.currentIndex]
        if tr and tr.sample then
            self:mcPlayLocal(spec.currentIndex)
        end
    end

    if MusicCenterHUD and MusicCenterHUD.INSTANCE and MusicCenterHUD.INSTANCE.placeable==self then
        MusicCenterHUD.INSTANCE:updateStatus()
        MusicCenterHUD.INSTANCE:updateButtons()
    end
end
