-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 12.11.2025

MusicCenterHUD = {}
local MusicCenterHUD_mt = Class(MusicCenterHUD, MessageDialog)

MusicCenterHUD.modDirectory      = g_currentModDirectory or ""
MusicCenterHUD.consoleRegistered = false
MusicCenterHUD.INSTANCE = nil

local function _l10n(s)
    if not s or s=="" then return "" end
    if s:sub(1,1) ~= "$" then return s end
    local k = s:sub(2)
    if g_i18n and g_i18n:hasText(k) then return g_i18n:getText(k) end
    if k:sub(1,5)=="l10n_" and g_i18n and g_i18n:hasText(k:sub(6)) then return g_i18n:getText(k:sub(6)) end
    return s
end

local function _modeName(mode)
    if mode == 2 then return _l10n("$ui_mc_mode_singleLoop") end
    if mode == 3 then return _l10n("$ui_mc_mode_allOnce") end
    if mode == 4 then return _l10n("$ui_mc_mode_allLoop") end
    return _l10n("$ui_mc_mode_singleOnce")
end

local function _fmt_mmss(sec)
    sec = math.max(0, math.floor(sec or 0))
    local m = math.floor(sec / 60)
    local s = sec % 60
    return string.format("%d:%02d", m, s)
end

local function _getLiveFarmerBaseDir_MC()
    local candidates = {}

    if PlaceableMusicCenter and PlaceableMusicCenter.modDirectory and PlaceableMusicCenter.modDirectory ~= "" then
        table.insert(candidates, PlaceableMusicCenter.modDirectory)
    end
    if MusicCenterHUD.modDirectory and MusicCenterHUD.modDirectory ~= "" then
        table.insert(candidates, MusicCenterHUD.modDirectory)
    end

    if g_modManager ~= nil then
        if g_modManager.getModByName ~= nil then
            local m = g_modManager:getModByName("FS25_liveFarmer")
            if m ~= nil then
                local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                if dir and dir ~= "" then
                    table.insert(candidates, dir)
                end
            end
        end
        if g_modManager.mods ~= nil then
            for _, m in pairs(g_modManager.mods) do
                local name     = m.modName or m.name or ""
                local fileName = m.modFileName or ""
                if name == "FS25_liveFarmer" or fileName:lower():find("fs25_livefarmer") ~= nil then
                    local dir = m.modDir or m.modDirectory or m.absolutePath or m.path or m.directory
                    if dir and dir ~= "" then
                        table.insert(candidates, dir)
                    end
                end
            end
        end
    end

    if debug and debug.getinfo then
        local info = debug.getinfo(_getLiveFarmerBaseDir_MC, "S")
        if info and info.source then
            local src = info.source
            if src:sub(1,1) == "@" then src = src:sub(2) end
            src = src:gsub("\\", "/")
            local idx = src:lower():find("/fs25_livefarmer/")
            if idx then
                local dir = src:sub(1, idx + string.len("/FS25_liveFarmer/") - 1)
                table.insert(candidates, dir)
            end
        end
    end

    if g_modsDirectory ~= nil then
        table.insert(candidates, g_modsDirectory .. "/FS25_liveFarmer/")
    end

    local tried = {}
    for _, baseDir in ipairs(candidates) do
        if type(baseDir) == "string" and baseDir ~= "" then
            baseDir = baseDir:gsub("\\", "/")
            if baseDir:sub(-1) ~= "/" then
                baseDir = baseDir .. "/"
            end
            if not tried[baseDir] then
                tried[baseDir] = true
                local testPath = Utils.getFilename("gui/MusicCenterHUD.xml", baseDir)
                if testPath and fileExists(testPath) then
                    return baseDir
                end
            end
        end
    end

    return MusicCenterHUD.modDirectory or (g_currentModDirectory or "")
end

function MusicCenterHUD.register()
    if not g_gui then return false end

    MusicCenterHUD.registerConsole()

    if g_gui.guis and g_gui.guis["MusicCenterHUD"] then
        MusicCenterHUD.INSTANCE = g_gui.guis["MusicCenterHUD"].target
        return true
    end

    local inst = MusicCenterHUD.new()

    local baseDir = _getLiveFarmerBaseDir_MC() or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/MusicCenterHUD.xml", baseDir)
    local ok = false

    if xmlPath and xmlPath ~= "" then
        ok = pcall(function()
            g_gui:loadGui(xmlPath, "MusicCenterHUD", inst)
        end)
    end

    if (not ok) or not (g_gui.guis and g_gui.guis["MusicCenterHUD"]) then
        local path1 = baseDir .. "gui/MusicCenterHUD.xml"
        local path2 = baseDir .. "MusicCenterHUD.xml"
        for _, p in ipairs({path1, path2}) do
            if p and p ~= "" then
                local loaded = pcall(function()
                    g_gui:loadGui(p, "MusicCenterHUD", inst)
                end)
                if loaded and g_gui.guis and g_gui.guis["MusicCenterHUD"] then
                    ok = true
                    xmlPath = p
                    break
                end
            end
        end
    end

    if not ok or not (g_gui.guis and g_gui.guis["MusicCenterHUD"]) then
        pcall(function() inst:delete() end)
        print("[MusicCenterHUD] GUI XML not found (baseDir=" .. tostring(baseDir) .. ", xmlPath=" .. tostring(xmlPath) .. ")")
        return false
    end

    MusicCenterHUD.INSTANCE = g_gui.guis["MusicCenterHUD"].target
    return true
end

function MusicCenterHUD.registerConsole()
    if MusicCenterHUD.consoleRegistered then return end
    if addConsoleCommand ~= nil then
        addConsoleCommand("gsReloadMusicCenterHUD","Reload MusicCenter HUD xml","reloadHUD",MusicCenterHUD)
        MusicCenterHUD.consoleRegistered = true
        print("[MusicCenterHUD] Console command: gsReloadMusicCenterHUD")
    end
end

function MusicCenterHUD.reloadHUD()
    if not g_gui then
        print("[MusicCenterHUD] g_gui is not ready")
        return
    end

    local baseDir = _getLiveFarmerBaseDir_MC and _getLiveFarmerBaseDir_MC() or (MusicCenterHUD.modDirectory or g_currentModDirectory or "")
    baseDir = baseDir or ""
    baseDir = baseDir:gsub("\\", "/")
    if baseDir ~= "" and baseDir:sub(-1) ~= "/" then
        baseDir = baseDir .. "/"
    end

    local xmlPath = Utils.getFilename("gui/MusicCenterHUD.xml", baseDir)
    if not (xmlPath and fileExists(xmlPath)) then
        print("[MusicCenterHUD] HUD XML not found to reload (baseDir=" .. tostring(baseDir) .. ")")
        return
    end

    local keepPlaceable, wasOpen = nil, false
    if MusicCenterHUD.INSTANCE then
        keepPlaceable = MusicCenterHUD.INSTANCE.placeable
    end

    if g_gui.closeDialogByName then
        if g_gui.guis and g_gui.guis["MusicCenterHUD"] then
            wasOpen = true
        end
        pcall(function()
            g_gui:closeDialogByName("MusicCenterHUD")
        end)
    elseif g_gui.currentGuiName == "MusicCenterHUD" then
        wasOpen = true
        pcall(function() g_gui:showGui("") end)
    end

    if g_gui.guis and g_gui.guis["MusicCenterHUD"] then
        pcall(function() g_gui:unloadGui("MusicCenterHUD") end)
        g_gui.guis["MusicCenterHUD"] = nil
    end
    if MusicCenterHUD.INSTANCE then
        pcall(function() MusicCenterHUD.INSTANCE:delete() end)
        MusicCenterHUD.INSTANCE = nil
    end

    local ok = MusicCenterHUD.register()
    print("[MusicCenterHUD] Reload result: " .. tostring(ok))

    if ok and wasOpen and keepPlaceable and MusicCenterHUD.INSTANCE then
        MusicCenterHUD.INSTANCE.placeable = keepPlaceable
        if g_gui.showDialog then
            g_gui:showDialog("MusicCenterHUD")
        else
            g_gui:showGui("MusicCenterHUD")
        end
    end
end

function MusicCenterHUD.new(target, custom_mt)
    local self = MessageDialog.new(target, custom_mt or MusicCenterHUD_mt)
    self.placeable     = nil
    self.tracks        = {}
    self.selectedIndex = 0

    self.list   = nil
    self.slider = nil
    self.btnPlay  = nil
    self.btnStop  = nil
    self.btnMode  = nil
    self.btnClose = nil
    self.titleEl  = nil
    self.statusEl = nil
    self.modeEl   = nil
    self.timeEl   = nil

    self._timeAccMs = 0
    self._lastShownSec = -1
    return self
end

-- ===== lifecycle =====
function MusicCenterHUD:onOpen()
    local sc = self:superClass(); if sc and sc.onOpen then sc.onOpen(self) end
    if g_inputBinding and g_inputBinding.setShowMouseCursor then
        g_inputBinding:setShowMouseCursor(true)
    end

    self.list     = self:getDescendantById("trackList")
    self.slider   = self:getDescendantById("trackListSlider")
    self.btnPlay  = self:getDescendantById("playButton")
    self.btnStop  = self:getDescendantById("stopButton")
    self.btnMode  = self:getDescendantById("modeButton")
    self.btnClose = self:getDescendantById("closeButton")
    self.titleEl  = self:getDescendantById("dialogTitleElement")
    self.statusEl = self:getDescendantById("statusText")
    self.modeEl   = self:getDescendantById("modeText")
    self.timeEl   = self:getDescendantById("timeText")

    FocusManager:setFocus(self.list)

    self:pullData()
    if self.selectedIndex == 0 and #self.tracks > 0 then self.selectedIndex=1 end
    if self.list then
        self.list:reloadData(true)
        if self.selectedIndex>0 then self.list:setSelectedIndex(self.selectedIndex, true, true) end
    end
    self:updateButtons()
    self:updateStatus()
    self:updateModeText()
    self._timeAccMs = 0
    self._lastShownSec = -1
end

function MusicCenterHUD:update(dt)
    local sc = self:superClass(); if sc and sc.update then sc.update(self, dt) end
    self._timeAccMs = (self._timeAccMs or 0) + (dt or 0)
    if (self._timeAccMs or 0) >= 200 then
        self._timeAccMs = 0
        self:updateTimeText()
    end
end

function MusicCenterHUD:onClose()
    if g_inputBinding and g_inputBinding.setShowMouseCursor then
        g_inputBinding:setShowMouseCursor(false)
    end
    local sc = self:superClass(); if sc and sc.onClose then sc.onClose(self) end
end

-- ===== data =====
function MusicCenterHUD:pullData()
    self.tracks = {}
    local spec = self.placeable and self.placeable.spec_musicCenter
    if spec then
        if self.titleEl then
            self.titleEl:setText(spec.title or _l10n("$ui_mc_title"))
        end

        for i, tr in ipairs(spec.tracks or {}) do
            local isPlaying = (spec.currentIndex == i and spec.playing == true)
            local mark = isPlaying and "[>] " or ""
            table.insert(self.tracks, {
                index      = i,
                name       = tr.name or ("Track "..i),
                playing    = isPlaying,
                displayTxt = string.format("%s%s", mark, tr.name or ("Track "..i))
            })
        end
    end

    if self.list then
        self.list:reloadData(true)
    end
end

function MusicCenterHUD:updateButtons()
    local spec = self.placeable and self.placeable.spec_musicCenter
    local sel  = self.tracks[self.selectedIndex]
    local canPlay = (sel ~= nil)
    local canStop = (spec and spec.playing) or false

    if self.btnPlay then self.btnPlay:setDisabled(not canPlay) end
    if self.btnStop then self.btnStop:setDisabled(not canStop) end
    if self.btnMode then self.btnMode:setDisabled(false) end
end

function MusicCenterHUD:updateStatus()
    local spec = self.placeable and self.placeable.spec_musicCenter
    if not self.statusEl then return end
    if spec and spec.playing and spec.currentIndex>0 then
        local tr = spec.tracks[spec.currentIndex]
        self.statusEl:setText(string.format(_l10n("$ui_mc_status_playing")..": %s", tr and tr.name or ("#"..tostring(spec.currentIndex))))
    else
        self.statusEl:setText(_l10n("$ui_mc_status_stopped"))
    end
    self:updateTimeText()
end

function MusicCenterHUD:updateTimeText()
    if not self.timeEl then return end
    local spec = self.placeable and self.placeable.spec_musicCenter
    if not (spec and spec.playing and spec.currentIndex>0 and spec.playStartMs) then
        self.timeEl:setText("")
        self._lastShownSec = -1
        return
    end
    local now = g_time or 0
    local elapsed = math.max(0, math.floor((now - (spec.playStartMs or now)) / 1000))
    if elapsed == self._lastShownSec then return end
    self._lastShownSec = elapsed

    local totalSec = nil
    if spec.playLenMs and spec.playLenMs > 0 then
        totalSec = math.floor(spec.playLenMs / 1000)
    end
    if totalSec then
        self.timeEl:setText(string.format("%s / %s", _fmt_mmss(elapsed), _fmt_mmss(totalSec)))
    else
        self.timeEl:setText(_fmt_mmss(elapsed))
    end
end

function MusicCenterHUD:updateModeText()
    if not self.modeEl then return end
    local spec = self.placeable and self.placeable.spec_musicCenter
    local mode = (spec and spec.mode) or 1
    self.modeEl:setText(string.format("%s: %s", _l10n("$ui_mc_mode_caption"), _modeName(mode)))
    if self.btnMode then
        self.btnMode:setText(_modeName(mode))
    end
end

-- ===== inputs =====
function MusicCenterHUD:onClickPlay()
    local sel = self.tracks[self.selectedIndex]; if not sel then return end
    if g_server == nil then
        if self.placeable then
            g_client:getServerConnection():sendEvent(MC_SelectTrackEvent.new(self.placeable, sel.index))
        end
    else
        if self.placeable and self.placeable.mcServerPlay then
            self.placeable:mcServerPlay(sel.index)
        end
    end
end

function MusicCenterHUD:onClickStop()
    if g_server == nil then
        if self.placeable then
            g_client:getServerConnection():sendEvent(MC_StopEvent.new(self.placeable))
        end
    else
        if self.placeable and self.placeable.mcServerStop then
            self.placeable:mcServerStop()
        end
    end
end

function MusicCenterHUD:onClickMode()
    local spec = self.placeable and self.placeable.spec_musicCenter
    if not spec then return end
    local nextMode = ((spec.mode or 1) % 4) + 1
    if g_server == nil then
        if self.placeable then
            g_client:getServerConnection():sendEvent(MC_ChangeModeEvent.new(self.placeable, nextMode))
        end
    else
        if self.placeable and self.placeable.mcServerSetMode then
            self.placeable:mcServerSetMode(nextMode)
        end
    end
    self:updateModeText()
end

-- ===== list =====
function MusicCenterHUD:getNumberOfItemsInSection(_, _)
    return #self.tracks
end

function MusicCenterHUD:populateCellForItemInSection(a, b, c, d)
    local list, section, index, cell
    if d ~= nil then
        list, section, index, cell = a, b, c, d
    else
        list, section, index, cell = self.list, a, b, c
    end

    local rowText = nil
    if type(cell) == "table" then
        if type(cell.getChildByName) == "function" then
            rowText = cell:getChildByName("rowText")
        end
        if rowText == nil and type(cell.getDescendantById) == "function" then
            rowText = cell:getDescendantById("rowText")
        end
        if rowText == nil and type(cell.elements) == "table" then
            for _, el in ipairs(cell.elements) do
                if (el.name == "rowText") or (el.id == "rowText") then
                    rowText = el
                    break
                end
            end
        end
    end

    local txt = (self.tracks[index] and self.tracks[index].displayTxt) or ""
    if rowText and rowText.setText then
        rowText:setText(txt)
    end
end

function MusicCenterHUD:onListSelectionChanged(a, b, c)
    local section, index
    if c ~= nil then
        section, index = b, c
    else
        section, index = a, b
    end
    self.selectedIndex = index or 0
    self:updateButtons()
end
