-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

-- Консольные команды:
--   gsSetItemEffects  <effectId> [durationSec] [effectsXmlRelPath] [-r|--reload]
--   gsStopItemEffects
--   gsListItemEffects [effectsXmlRelPath]
--   gsReloadItemEffects [effectsXmlRelPath]
--
-- Пример:
--   gsSetItemEffects beer 60 -r
--   gsListItemEffects
--   gsReloadItemEffects
--   gsStopItemEffects

local ConsoleIE = {}
ConsoleIE.modDirectory = g_currentModDirectory or ""
ConsoleIE._registered  = false

local function nowMs() return g_time or 0 end

local function resolveXmlPath(pathRel)
    local rel = pathRel
    if rel == nil or rel == "" then rel = "itemEffects.xml" end
    return Utils.getFilename(rel, ConsoleIE.modDirectory)
end

local function ensureItemEffects()
    if ItemEffects == nil or ItemEffects.startEffect == nil then
        print("[ItemEffectsConsole] ERROR: ItemEffects.lua не загружен — команды недоступны.")
        return false
    end
    return true
end

local function doReload(xmlAbs)
    if not ensureItemEffects() then return end
    if ItemEffects.reloadEffects then
        ItemEffects:reloadEffects(xmlAbs)
        return
    end
    ItemEffects._cache = ItemEffects._cache or {}
    ItemEffects._cache[xmlAbs] = nil
    if ItemEffects.getEffectDef then
        ItemEffects:getEffectDef("", xmlAbs)
    end
    print(string.format("[ItemEffects] Reloaded (fallback) %s", tostring(xmlAbs)))
end

function ConsoleIE:cmdSet(effectId, durationSecStr, xmlRel, maybeReload)
    if not ensureItemEffects() then return end
    if effectId == nil or effectId == "" then
        print("USAGE: gsSetItemEffects <effectId> [durationSec] [effectsXmlRelPath] [-r|--reload]")
        print("Example: gsSetItemEffects beer 60 -r")
        return
    end

    local dur = tonumber(durationSecStr or "")
    local xmlArg, reloadFlag = xmlRel, (maybeReload == "-r" or maybeReload == "--reload")

    if durationSecStr and durationSecStr ~= "" and dur == nil then
        xmlArg = durationSecStr
        reloadFlag = (xmlRel == "-r" or xmlRel == "--reload") or reloadFlag
        dur = tonumber(maybeReload or "")
    end

    local xmlAbs = resolveXmlPath(xmlArg)

    if reloadFlag then
        doReload(xmlAbs)
    end

    local ok = ItemEffects:startEffect(HungerSystem or ItemEffects, effectId, xmlAbs)
    if not ok then
        print(string.format("[ItemEffectsConsole] FAILED: '%s' из %s", tostring(effectId), tostring(xmlAbs)))
        return
    end

    if dur and dur > 0 and ItemEffects.active and #ItemEffects.active > 0 then
        local ent = ItemEffects.active[#ItemEffects.active]
        ent.untilMs = nowMs() + math.floor(dur * 1000)
    end

    print(string.format("[ItemEffectsConsole] Started '%s' (%s)%s",
        tostring(effectId), xmlAbs, reloadFlag and " [reloaded]" or ""))
end

function ConsoleIE:cmdStop()
    if not ensureItemEffects() then return end
    ItemEffects:clearAll()
    print("[ItemEffectsConsole] All item effects stopped.")
end

function ConsoleIE:cmdList(xmlRel)
    if not ensureItemEffects() then return end
    local xmlAbs = resolveXmlPath(xmlRel)

    if not (ItemEffects._cache and ItemEffects._cache[xmlAbs]) then
        if ItemEffects.getEffectDef then ItemEffects:getEffectDef("", xmlAbs) end
    end

    local pack = ItemEffects._cache and ItemEffects._cache[xmlAbs]
    local map  = pack and pack.effectsById or {}
    print(string.format("[ItemEffectsConsole] Effects in %s:", tostring(xmlAbs)))
    local n = 0
    for id, def in pairs(map) do
        n = n + 1
        print(string.format("  - %s  (title=\"%s\", duration=%ss, tick=%ss, Δhunger=%d, Δvigor=%d)",
            id, tostring(def.title or id), tostring(def.durationSec or "?"), tostring(def.tickSec or "?"),
            tonumber(def.hungerPerTick or 0), tonumber(def.vigorPerTick or 0)))
    end
    if n == 0 then print("  (empty)") end
end

function ConsoleIE:cmdReload(xmlRel)
    if not ensureItemEffects() then return end
    local xmlAbs = resolveXmlPath(xmlRel)
    doReload(xmlAbs)
end

-- ===================== РЕГИСТРАЦИЯ КОМАНД =======================

local Registrar = { _lastTryMs = 0 }

local function registerWithFunctionAPI()
    if type(addConsoleCommand) ~= "function" then return false end
    addConsoleCommand("gsSetItemEffects",
        "Start item effect: gsSetItemEffects <effectId> [durationSec] [effectsXmlRelPath] [-r|--reload]",
        "cmdSet",  ConsoleIE)
    addConsoleCommand("gsStopItemEffects",
        "Stop all item effects",
        "cmdStop", ConsoleIE)
    addConsoleCommand("gsListItemEffects",
        "List effect ids from xml: gsListItemEffects [effectsXmlRelPath]",
        "cmdList", ConsoleIE)
    addConsoleCommand("gsReloadItemEffects",
        "Reload itemEffects xml cache: gsReloadItemEffects [effectsXmlRelPath]",
        "cmdReload", ConsoleIE)
    print("[ItemEffectsConsole] Registered via addConsoleCommand")
    return true
end

local function registerWithManagerAPI()
    local mgr = rawget(_G, "g_consoleCommandManager")
           or  (g_currentMission and g_currentMission.consoleCommandManager)
           or  (g_currentMission and g_currentMission.console and g_currentMission.console.commandManager)
    if not (mgr and mgr.addCommand) then return false end
    mgr:addCommand("gsSetItemEffects",
        "Start item effect: gsSetItemEffects <effectId> [durationSec] [effectsXmlRelPath] [-r|--reload]",
        "cmdSet",  ConsoleIE)
    mgr:addCommand("gsStopItemEffects",
        "Stop all item effects",
        "cmdStop", ConsoleIE)
    mgr:addCommand("gsListItemEffects",
        "List effect ids from xml: gsListItemEffects [effectsXmlRelPath]",
        "cmdList", ConsoleIE)
    mgr:addCommand("gsReloadItemEffects",
        "Reload itemEffects xml cache: gsReloadItemEffects [effectsXmlRelPath]",
        "cmdReload", ConsoleIE)
    print("[ItemEffectsConsole] Registered via g_consoleCommandManager")
    return true
end

local function actuallyRegister()
    if ConsoleIE._registered then return true end
    if registerWithFunctionAPI() or registerWithManagerAPI() then
        ConsoleIE._registered = true
        return true
    end
    return false
end

actuallyRegister()

function Registrar:loadMap()          actuallyRegister() end
function Registrar:loadMapFinished()  actuallyRegister() end

function Registrar:update(dt)
    if ConsoleIE._registered then return end
    self._lastTryMs = (self._lastTryMs or 0) + dt
    if self._lastTryMs > 1000 then
        self._lastTryMs = 0
        actuallyRegister()
    end
end

addModEventListener(Registrar)
