-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

ItemEffects = {}
ItemEffects.modDirectory = g_currentModDirectory or ""
ItemEffects._cache  = ItemEffects._cache or {}
ItemEffects.active  = ItemEffects.active  or {}

local function nowMs() return g_time or 0 end
local function absPath(rel, base) return (rel and rel ~= "" and Utils.getFilename(rel, base or ItemEffects.modDirectory)) or nil end
local function playerLocal()
    return (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
        or (g_currentMission and g_currentMission.player) or nil
end

function ItemEffects:getNowMs(dt)
    if type(g_time) == "number" and g_time > 0 then
        self._nowMs = g_time
        return self._nowMs
    end
    local add = tonumber(dt) or 0
    if add <= 0 then add = 16 end
    self._nowMs = (self._nowMs or 0) + add
    return self._nowMs
end

local function ensureOverlay(pathAbs)
    if not pathAbs or not fileExists(pathAbs) then return nil end
    local ov = Overlay.new(pathAbs, 0, 0, 1, 1)
    if ov then ov:setColor(1,1,1,0) end
    return ov
end

local function ensureVehicleGlobal()
    if g_DrunkVehicleEffects ~= nil then return g_DrunkVehicleEffects end
    g_DrunkVehicleEffects = {
        enabled = false,
        cam   = {
            ampX=0.006, ampY=0.004, ampZ=0.0, rollAmp=0.006, period=2.8,
            phaseX=0.0, phaseY=1.2, phaseZ=0.0, phaseRoll=0.7,
            noiseAmp=0.0, noisePeriod=2.4, fpScale=1.0, tpScale=0.6
        },
        fov   = { amplitudePct=0.0, period=3.0, fpScale=1.0, tpScale=0.6, fpOnly=false, tpOnly=false },
        steer = { amplitude=0.20, period=2.6, deadzoneHold=0.15, onlyWhenMoving=true, speedThreshold=0.5 }
    }
    print("[ItemEffects] ensureVehicleGlobal(): created default g_DrunkVehicleEffects")
    return g_DrunkVehicleEffects
end

local function loadFile(pathAbs)
    ----------------------------------------------------------------------
    -- 1) Если путь не передали или он пустой — используем дефолтный
    --    itemEffects.xml из папки мода (ItemEffects.modDirectory)
    ----------------------------------------------------------------------
    if pathAbs == nil or pathAbs == "" then
        pathAbs = absPath("itemEffects.xml")  -- => "<modDir>/itemEffects.xml"
    end

    ----------------------------------------------------------------------
    -- 2) Инициализируем кэш, чтобы не было nil
    ----------------------------------------------------------------------
    ItemEffects._cache = ItemEffects._cache or {}

    ----------------------------------------------------------------------
    -- 3) Если уже есть в кэше — возвращаем
    ----------------------------------------------------------------------
    local cached = ItemEffects._cache[pathAbs]
    if cached then
        return cached
    end

    ----------------------------------------------------------------------
    -- 4) Файл не найден — не падаем, а создаём пустой пакет
    ----------------------------------------------------------------------
    if not pathAbs or not fileExists(pathAbs) then
        print("[ItemEffects] XML not found: "..tostring(pathAbs))
        local pack = { effectsById = {}, pathAbs = pathAbs }
        ItemEffects._cache[pathAbs] = pack
        return pack
    end

    ----------------------------------------------------------------------
    -- 5) Пытаемся открыть XML
    ----------------------------------------------------------------------
    local xml = loadXMLFile("itemEffectsXml", pathAbs)
    if xml == 0 then
        print("[ItemEffects] Failed to open: "..tostring(pathAbs))
        local pack = { effectsById = {}, pathAbs = pathAbs }
        ItemEffects._cache[pathAbs] = pack
        return pack
    end

    ----------------------------------------------------------------------
    -- 6) Читаем все <itemEffects><effect .../></itemEffects>
    ----------------------------------------------------------------------
    local map, i = {}, 0
    while true do
        local key = string.format("itemEffects.effect(%d)", i)
        if not hasXMLProperty(xml, key) then break end

        local id   = getXMLString(xml, key.."#id") or ("effect"..tostring(i+1))
        local ttl  = getXMLString(xml, key.."#title") or id
        local dur  = getXMLFloat (xml, key.."#durationSec") or 60
        local tsec = getXMLFloat (xml, key.."#tickSec") or 5
        local hpt  = math.floor(getXMLFloat(xml, key.."#hungerPerTick") or 0)
        local vpt  = math.floor(getXMLFloat(xml, key.."#vigorPerTick")  or 0)

        ------------------------------------------------------------------
        -- VISUAL
        ------------------------------------------------------------------
        local vis, vk = nil, key..".visual"
        if hasXMLProperty(xml, vk) then
            vis = {
                loop         = getXMLBool(xml, vk.."#loop") or false,
                crossFadeSec = math.max(0.0, getXMLFloat(xml, vk.."#crossFadeSec") or 0.35),
                fadeInSec    = math.max(0.0, getXMLFloat(xml, vk.."#fadeInSec")    or 0.8),
                fadeOutSec   = math.max(0.0, getXMLFloat(xml, vk.."#fadeOutSec")   or 1.0),
                overlays     = {}
            }

            local j=0
            while true do
                local ok = string.format("%s.overlay(%d)", vk, j)
                if not hasXMLProperty(xml, ok) then break end
                local rel = getXMLString(xml, ok.."#filename")
                local abs = absPath(rel)
                if abs and fileExists(abs) then
                    local d = getXMLFloat(xml, ok.."#durationSec")
                    if d == nil then d = getXMLFloat(xml, ok.."#duration") end
                    table.insert(vis.overlays, {
                        filenameAbs = abs,
                        alphaMax    = getXMLFloat(xml, ok.."#alphaMax") or 0.45,
                        durationSec = math.max(0.05, d or 2.0)
                    })
                else
                    print(string.format("[ItemEffects] Visual overlay missing: %s", tostring(abs or rel)))
                end
                j = j + 1
            end

            if #vis.overlays == 0 and hasXMLProperty(xml, vk..".overlay#filename") then
                local rel = getXMLString(xml, vk..".overlay#filename")
                local abs = absPath(rel)
                if abs and fileExists(abs) then
                    local d = getXMLFloat(xml, vk..".overlay#durationSec")
                    if d == nil then d = getXMLFloat(xml, vk..".overlay#duration") end
                    table.insert(vis.overlays, {
                        filenameAbs = abs,
                        alphaMax    = getXMLFloat(xml, vk..".overlay#alphaMax") or 0.45,
                        durationSec = math.max(0.05, d or 999999.0)
                    })
                end
            end

            if #vis.overlays == 0 then vis = nil end
        end

        ------------------------------------------------------------------
        -- CAMERA (player)
        ------------------------------------------------------------------
        local cam, ck = nil, key..".camera"
        if hasXMLProperty(xml, ck) then
            cam = {}
            if hasXMLProperty(xml, ck..".sway") then
                cam.sway = {
                    ampX  = getXMLFloat(xml, ck..".sway#ampX") or 0.012,
                    ampY  = getXMLFloat(xml, ck..".sway#ampY") or 0.006,
                    ampZ  = getXMLFloat(xml, ck..".sway#ampZ") or 0.0,
                    rollAmp = getXMLFloat(xml, ck..".sway#rollAmp") or 0.01,
                    period  = math.max(0.2, getXMLFloat(xml, ck..".sway#period") or 2.8),
                    noiseAmp    = getXMLFloat(xml, ck..".sway#noiseAmp") or 0.0,
                    noisePeriod = math.max(0.2, getXMLFloat(xml, ck..".sway#noisePeriod") or 2.4),
                    fpScale = getXMLFloat(xml, ck..".sway#fpScale") or 1.0,
                    tpScale = getXMLFloat(xml, ck..".sway#tpScale") or 0.45,
                    fpOnly  = getXMLBool (xml, ck..".sway#fpOnly") or false,
                    tpOnly  = getXMLBool (xml, ck..".sway#tpOnly") or false,
                    phaseX  = getXMLFloat(xml, ck..".sway#phaseX") or 0.0,
                    phaseY  = getXMLFloat(xml, ck..".sway#phaseY") or (math.pi*0.5),
                    phaseZ  = getXMLFloat(xml, ck..".sway#phaseZ") or 0.0,
                    phaseRoll = getXMLFloat(xml, ck..".sway#phaseRoll") or (math.pi*0.33)
                }
            end
            if hasXMLProperty(xml, ck..".fov") then
                cam.fov = {
                    amplitudePct = getXMLFloat(xml, ck..".fov#amplitudePct") or (getXMLFloat(xml, ck..".fov#amplitude") or 3.0),
                    period       = math.max(0.2, getXMLFloat(xml, ck..".fov#period") or 3.0),
                    fpScale      = getXMLFloat(xml, ck..".fov#fpScale") or 1.0,
                    tpScale      = getXMLFloat(xml, ck..".fov#tpScale") or 0.5,
                    fpOnly       = getXMLBool (xml, ck..".fov#fpOnly") or false,
                    tpOnly       = getXMLBool (xml, ck..".fov#tpOnly") or false
                }
            end
        end

        ------------------------------------------------------------------
        -- WOBBLE (player)
        ------------------------------------------------------------------
        local wb
        if hasXMLProperty(xml, key..".wobble") then
            wb = {
                ampStrafe = getXMLFloat(xml, key..".wobble#ampStrafe") or 0.012,
                ampForward= getXMLFloat(xml, key..".wobble#ampForward") or 0.0,
                period    = math.max(0.2, getXMLFloat(xml, key..".wobble#period") or 2.8),
                smoothMs  = math.max(16.0, getXMLFloat(xml, key..".wobble#smoothMs") or 120.0),
                groundOnly= getXMLBool (xml, key..".wobble#groundOnly") ~= false,
                fpOnly    = getXMLBool (xml, key..".wobble#fpOnly") or true,
                tpOnly    = getXMLBool (xml, key..".wobble#tpOnly") or false
            }
        end

        ------------------------------------------------------------------
        -- VEHICLE
        ------------------------------------------------------------------
        local veh
        local vk2 = key..".vehicle"
        if hasXMLProperty(xml, vk2) then
            local swayPath = vk2..".sway"
            local fovPath  = vk2..".fov"
            local steerPath= vk2..".steer"

            if hasXMLProperty(xml, swayPath) or hasXMLProperty(xml, fovPath) or hasXMLProperty(xml, steerPath) then
                veh = {}
                if hasXMLProperty(xml, swayPath) then
                    veh.sway = {
                        ampX  = getXMLFloat(xml, swayPath.."#ampX") or 0.006,
                        ampY  = getXMLFloat(xml, swayPath.."#ampY") or 0.004,
                        ampZ  = getXMLFloat(xml, swayPath.."#ampZ") or 0.0,
                        rollAmp = getXMLFloat(xml, swayPath.."#rollAmp") or 0.006,
                        period  = math.max(0.2, getXMLFloat(xml, swayPath.."#period") or 2.8),
                        phaseX  = getXMLFloat(xml, swayPath.."#phaseX") or 0.0,
                        phaseY  = getXMLFloat(xml, swayPath.."#phaseY") or 1.2,
                        phaseZ  = getXMLFloat(xml, swayPath.."#phaseZ") or 0.0,
                        phaseRoll = getXMLFloat(xml, swayPath.."#phaseRoll") or 0.7,
                        noiseAmp    = getXMLFloat(xml, swayPath.."#noiseAmp") or 0.0,
                        noisePeriod = math.max(0.2, getXMLFloat(xml, swayPath.."#noisePeriod") or 2.4),
                        fpScale   = getXMLFloat(xml, swayPath.."#fpScale") or 1.0,
                        tpScale   = getXMLFloat(xml, swayPath.."#tpScale") or 0.6
                    }
                end
                if hasXMLProperty(xml, fovPath) then
                    veh.fov = {
                        amplitudePct = getXMLFloat(xml, fovPath.."#amplitudePct") or 0.0,
                        period       = math.max(0.2, getXMLFloat(xml, fovPath.."#period") or 3.0)
                    }
                end
                if hasXMLProperty(xml, steerPath) then
                    veh.steer = {
                        amplitude   = getXMLFloat(xml, steerPath.."#amplitude") or 0.20,
                        period      = math.max(0.3, getXMLFloat(xml, steerPath.."#period") or 2.6),
                        deadzoneHold= getXMLFloat(xml, steerPath.."#deadzoneHold") or 0.15
                    }
                end
            end
        end

        ------------------------------------------------------------------
        -- AUDIO (single)
        ------------------------------------------------------------------
        local ak, au = key..".audio", nil
        if hasXMLProperty(xml, ak.."#filename") then
            au = {
                filenameRel = getXMLString(xml, ak.."#filename"),
                filenameAbs = absPath(getXMLString(xml, ak.."#filename")),
                volume      = getXMLFloat(xml, ak.."#volume") or 0.8,
                pitch       = getXMLFloat(xml, ak.."#pitch")  or 1.0,
                loop        = getXMLBool (xml, ak.."#loop")   or false,
                durationSec = nil
            }
        end

        ------------------------------------------------------------------
        -- AUDIOS (playlist)
        ------------------------------------------------------------------
        local audios, akey = nil, key..".audios"
        if hasXMLProperty(xml, akey) then
            audios = {
                loop = getXMLBool(xml, akey.."#loop") or false,
                items = {}
            }
            local j=0
            while true do
                local p = string.format("%s.audio(%d)", akey, j)
                if not hasXMLProperty(xml, p) then break end
                local rel = getXMLString(xml, p.."#filename")
                local abs = absPath(rel)
                if abs and fileExists(abs) then
                    local d  = getXMLFloat(xml, p.."#durationSec")
                    if d == nil then d = getXMLFloat(xml, p.."#duration") end
                    local dl = getXMLFloat(xml, p.."#delaySec")
                    if dl == nil then dl = getXMLFloat(xml, p.."#delay") end
                    table.insert(audios.items, {
                        filenameAbs = abs,
                        volume      = getXMLFloat(xml, p.."#volume") or 0.8,
                        pitch       = getXMLFloat(xml, p.."#pitch")  or 1.0,
                        durationSec = math.max(0.05, d or 1.0),
                        delaySec    = math.max(0.0,  dl or 0.0),
                        loop        = getXMLBool (xml, p.."#loop")   or false,
                        always      = getXMLBool (xml, p.."#always") or false
                    })
                else
                    print(string.format("[ItemEffects] Audio file missing: %s", tostring(abs or rel)))
                end
                j = j + 1
            end
            if #audios.items == 0 then audios = nil end
        end

        map[id] = {
            id = id, title = ttl,
            durationSec = dur, tickSec = tsec,
            hungerPerTick = hpt, vigorPerTick = vpt,
            maxStacks = math.max(1, math.floor(getXMLFloat(xml, key.."#maxStacks") or 1)),
            visual = vis, camera = cam, wobble = wb, vehicle = veh,
            audio = au, audios = audios
        }

        i = i + 1
    end

    delete(xml)

    local pack = { effectsById = map, pathAbs = pathAbs }
    ItemEffects._cache[pathAbs] = pack
    print(string.format("[ItemEffects] Loaded %d effects from %s", i, tostring(pathAbs)))
    return pack
end

function ItemEffects:getEffectDef(effectId, effectsFileAbs)
    -- effectsFileAbs может быть nil, loadFile сам подставит itemEffects.xml
    local pack = loadFile(effectsFileAbs)
    local map  = pack and pack.effectsById or {}
    return map[tostring(effectId or "")]
end


-- ========= сохран =========

function ItemEffects:serializeActive()
    local now = g_time or 0
    local out = {}
    if ItemEffects.active then
        for _, ent in ipairs(ItemEffects.active) do
            local remainSec = math.max(0, math.floor(((ent.untilMs or now) - now) / 1000))
            if remainSec > 0 then
                table.insert(out, {
                    id             = ent.id,
                    stacks         = ent.stacks or 1,
                    remainSec      = remainSec,
                    effectsFileAbs = ent.effectsFileAbs or ""
                })
            end
        end
    end
    return out
end

function ItemEffects:getActiveEntry(id)
    if ItemEffects.active then
        for _, ent in ipairs(ItemEffects.active) do
            if tostring(ent.id) == tostring(id) then
                return ent
            end
        end
    end
    return nil
end

function ItemEffects:resumeSerialized(core, list)
    ItemEffects.active = ItemEffects.active or {}
	local now = ItemEffects:getNowMs(0)

    local revived = 0
    local firstAliveId = nil

    local function isAlive(ent, _now)
        _now = _now or (g_time or 0)
        if _now < (ent.untilMs or 0) then return true end
        if ent.visual and (ent.visual.master or 0) > 0.01 then return true end
        return false
    end

    for _, rec in ipairs(list or {}) do
        local def = ItemEffects:getEffectDef(rec.id, rec.effectsFileAbs)
        if not def then
            def = {
                id = rec.id,
                title = tostring(rec.id),
                durationSec = math.max(0, tonumber(rec.remainSec or 0)),
                maxStacks = rec.stacks or 1
            }
            print(string.format("[ItemEffects] resumeSerialized: missing def for '%s', using stub", tostring(rec.id)))
        end

        local ent = {
            id            = def.id,
            title         = def.title or def.id,
            base          = def,

            startedAtMs   = now,
            untilMs       = now + math.max(0, math.floor((rec.remainSec or 0) * 1000)),
            nextTickMs    = now + math.floor(((def.tickSec or 1) * 1000)),
            tickSec       = def.tickSec or 1,
            hungerPerTick = def.hungerPerTick or 0,
            vigorPerTick  = def.vigorPerTick  or 0,

            stacks        = math.max(1, tonumber(rec.stacks or 1) or 1),
            maxStacks     = math.max(1, tonumber(def.maxStacks or rec.maxStacks or rec.stacks or 1) or 1),

            visual        = nil,
            camera        = (def.camera and { sway = def.camera.sway }) or nil,
            wobble        = def.wobble,
            vehicle       = def.vehicle,

            audio         = nil,
            audios        = nil,

            effectsFileAbs= rec.effectsFileAbs
        }

        if def.visual and def.visual.overlays and #def.visual.overlays > 0 then
            local v = {
                loop         = def.visual.loop == true,
                crossFadeSec = def.visual.crossFadeSec or 0.35,
                fadeInSec    = def.visual.fadeInSec or 0.8,
                fadeOutSec   = def.visual.fadeOutSec or 1.0,
                overlays     = {},
                master       = 1.0,
                idx          = 1,
                timeLeft     = (def.visual.overlays[1] and def.visual.overlays[1].durationSec) or 2.0
            }
            for _, o in ipairs(def.visual.overlays) do
                local ov = ensureOverlay(o.filenameAbs)
                if ov then table.insert(v.overlays, { def=o, ov=ov, a=0.0 }) end
            end
            if #v.overlays > 0 then
                v.overlays[1].a = 1.0
                ent.visual = v
            end
        end

        if def.audios then
            local playItems = {}
            local srcItems  = def.audios.items or {}
            for _, it in ipairs(srcItems) do
                if it and not it.always and it.filenameAbs and fileExists(it.filenameAbs) then
                    table.insert(playItems, it)
                end
            end
			ent.audios = { loop = def.audios.loop == true, items = srcItems, playItems = playItems }
			if ItemEffects.audioAlwaysStart then pcall(ItemEffects.audioAlwaysStart, ItemEffects, ent) end
			if ItemEffects._playlistHasPlayable and ItemEffects.audioPlaylistStart
			   and ItemEffects._playlistHasPlayable(ent.audios) then
				pcall(ItemEffects.audioPlaylistStart, ItemEffects, ent)
			end

        elseif def.audio and def.audio.filenameAbs and fileExists(def.audio.filenameAbs) then
            local name = "itemEffect_"..tostring(ent.id).."_"..tostring(math.random(100000))
            ent.audio = {
                filenameAbs = def.audio.filenameAbs,
                volume      = def.audio.volume or 0.8,
                pitch       = def.audio.pitch  or 1.0,
                loop        = def.audio.loop   or false
            }
            ent.audio.sampleId = createSample and createSample(name) or nil
            if ent.audio.sampleId then
                pcall(function() loadSample(ent.audio.sampleId, ent.audio.filenameAbs, ent.audio.loop) end)
                pcall(function() playSample(ent.audio.sampleId, 0, 1, ent.audio.volume, ent.audio.pitch, 0) end)
            end
        end

        table.insert(ItemEffects.active, ent)

        if applyCameraFrom          then pcall(applyCameraFrom,          ent, true) end
        if applyWobbleFrom          then pcall(applyWobbleFrom,          ent, true) end
        if enableVehicleEffectsFrom then pcall(enableVehicleEffectsFrom, ent) end

        ent._reapplyAtMs = now + 200

        revived = revived + 1
        if not firstAliveId and isAlive(ent, now) then
            firstAliveId = ent.id
        end
    end

    local needPick = false
    if not ItemEffects._hudPrimaryId or ItemEffects._hudPrimaryId == "" then
        needPick = true
    else
        local alive = false
        for _, ent in ipairs(ItemEffects.active) do
            if tostring(ent.id) == tostring(ItemEffects._hudPrimaryId) and isAlive(ent, now) then
                alive = true
                break
            end
        end
        if not alive then needPick = true end
    end
    if needPick and firstAliveId then
        ItemEffects._hudPrimaryId = firstAliveId
    end

    if core then core.saveDirty = true end
    print(string.format("[ItemEffects] resumeSerialized: revived=%d, hudPrimary=%s",
        revived, tostring(ItemEffects._hudPrimaryId or "")))

    return revived
end

function ItemEffects:saveActiveToFile(path)
    local effects = self:serializeActive() or {}
    local xml = createXMLFile("HungerSystemEffects", path, "HungerSystemEffects")
    if xml == 0 then
        print("[ItemEffects] saveActiveToFile(): cannot create XML at " .. tostring(path))
        return false
    end

    local i = 0
    while hasXMLProperty(xml, string.format("HungerSystemEffects.e(%d)", i)) do
        removeXMLProperty(xml, string.format("HungerSystemEffects.e(%d)", i))
        i = i + 1
    end

    for idx, e in ipairs(effects) do
        local key = string.format("HungerSystemEffects.e(%d)", idx - 1)
        setXMLString(xml, key .. "#id",             tostring(e.id))
        setXMLInt  (xml, key .. "#stacks",          tonumber(e.stacks or 1))
        setXMLInt  (xml, key .. "#remainSec",       tonumber(e.remainSec or 0))
        setXMLString(xml, key .. "#effectsFileAbs", tostring(e.effectsFileAbs or ""))
    end

    saveXMLFile(xml)
    delete(xml)
    print(("[ItemEffects] Saved %d active effects to: %s"):format(#effects, tostring(path)))
    return true
end

function ItemEffects:loadFromFile(core, path)
    if not fileExists(path) then
        print(("[ItemEffects] Effects file not found (skip). Path: %s"):format(tostring(path)))
        return false
    end

    local xml = loadXMLFile("HungerSystemEffects", path)
    if xml == 0 then
        print(("[ItemEffects] Cannot open effects XML, skip. Path: %s"):format(tostring(path)))
        return false
    end

    local function readInt(xml, key)
        local v = nil
        pcall(function() v = getXMLInt(xml, key) end)
        if v == nil then
            local s = nil
            pcall(function() s = getXMLString(xml, key) end)
            if s ~= nil then v = tonumber(s) end
        end
        return v
    end

    local function collect(prefix)
        local list, i = {}, 0
        while true do
            local key = string.format("%s.e(%d)", prefix, i)
            if not hasXMLProperty(xml, key) then break end
            local id  = getXMLString(xml, key .. "#id")
            local st  = readInt(xml, key .. "#stacks")    or 1
            local rem = readInt(xml, key .. "#remainSec") or 0
            local ef  = getXMLString(xml, key .. "#effectsFileAbs") or ""
            if id and id ~= "" and rem > 0 then
                table.insert(list, { id = id, stacks = st, remainSec = rem, effectsFileAbs = ef })
            end
            i = i + 1
        end
        return list
    end

    local list = collect("HungerSystemEffects")
    if #list == 0 then
        list = collect("HungerSystem.effects")
    end

    delete(xml)

    if #list > 0 then
        self:resumeSerialized(core, list)
        print(("[ItemEffects] Loaded %d active effects from: %s"):format(#list, tostring(path)))
        return true
    else
        print(("[ItemEffects] No active effects in: %s"):format(tostring(path)))
        return true
    end
end

function ItemEffects:reloadEffects(effectsFileAbs)
    self._cache = self._cache or {}

    -- тот же трюк: если пусто — берём дефолтный itemEffects.xml
    local pathAbs = effectsFileAbs
    if pathAbs == nil or pathAbs == "" then
        pathAbs = absPath("itemEffects.xml")
    end

    -- чистим из кэша и форсируем перезагрузку
    self._cache[pathAbs] = nil
    self:getEffectDef("", pathAbs)

    print(string.format("[ItemEffects] Reloaded effects from %s", tostring(pathAbs)))
end


-- ========= CAMERA / WOBBLE (player) =========
local function _isAlive(ent, now)
    now = now or (g_time or 0)
    if now < (ent.untilMs or 0) then return true end
    if ent.visual and (ent.visual.master or 0) > 0.01 then return true end
    return false
end

local function anyOther(thisEnt, kind)
    local list = ItemEffects.active or {}
    local now  = g_time or 0
    for _, e in ipairs(list) do
        if e ~= thisEnt and _isAlive(e, now) then
            if kind == "wobble"  and e.wobble  then return true end
            if kind == "camera"  and e.camera  then return true end
            if kind == "vehicle" and e.vehicle then return true end
        end
    end
    return false
end

-- ====== КАМЕРА И «КАЧКА» ИГРОКА =============================================
local function applyCameraFrom(ent, enable)
    local pl = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
            or (g_currentMission and g_currentMission.player)
    if not pl or not pl.camera or type(pl.camera.setDrunkSwayEx) ~= "function" then
        return
    end

    if not enable then
        if anyOther and anyOther(ent, "camera") then return end
        pcall(function() pl.camera:setDrunkSwayEx(false) end)
        return
    end

    local sway = ent.camera and ent.camera.sway
    if not sway then return end

    local mx = math.max(1, ent.maxStacks or 1)
    local st = math.max(1, ent.stacks    or 1)
    local k  = st / mx

    local cfg = {
        ampX       = (sway.ampX or 0.0) * k,
        ampY       = (sway.ampY or 0.0) * k,
        ampZ       = (sway.ampZ or 0.0) * k,
        rollAmp    = (sway.rollAmp or 0.0) * k,
        period     = sway.period or 2.8,

        phaseX     = sway.phaseX or 0.0,
        phaseY     = sway.phaseY or (math.pi * 0.5),
        phaseZ     = sway.phaseZ or 0.0,
        phaseRoll  = sway.phaseRoll or (math.pi * 0.33),

        noiseAmp   = (sway.noiseAmp or 0.0) * k,
        noisePeriod= sway.noisePeriod or 2.4,

        fpScale    = sway.fpScale or 1.0,
        tpScale    = sway.tpScale or 0.45,
        fpOnly     = (sway.fpOnly == true) or (sway.fpOnly == "true"),
        tpOnly     = (sway.tpOnly == true) or (sway.tpOnly == "true"),

        fadeInSec  = tonumber(sway.fadeInSec  or 0) or 0,
        fadeOutSec = tonumber(sway.fadeOutSec or 0) or 0,

        startAtMs  = ent.startedAtMs or (g_time or 0),
        untilMs    = ent.untilMs    or (g_time or 0)
    }

    pcall(function() pl.camera:setDrunkSwayEx(true, cfg) end)
end

local function applyWobbleFrom(ent, enable)
    if not g_DrunkPlayerWobble then return end
    if enable and ent.wobble then
        local mx = math.max(1, ent.maxStacks or 1)
        local st = math.max(1, ent.stacks    or 1)
        local k  = st / mx

        g_DrunkPlayerWobble.enabled    = true
        g_DrunkPlayerWobble.ampStrafe  = (ent.wobble.ampStrafe  or 0.012) * k
        g_DrunkPlayerWobble.ampForward = (ent.wobble.ampForward or 0.0)   * k
        g_DrunkPlayerWobble.period     = ent.wobble.period    or 2.8
        g_DrunkPlayerWobble.smoothMs   = ent.wobble.smoothMs  or 120.0
        g_DrunkPlayerWobble.groundOnly = ent.wobble.groundOnly ~= false
        g_DrunkPlayerWobble.fpOnly     = ent.wobble.fpOnly     ~= false
        g_DrunkPlayerWobble.tpOnly     = ent.wobble.tpOnly     or false
    else
        if not anyOther or not anyOther(ent, "wobble") then
            g_DrunkPlayerWobble.enabled = false
        end
    end
end

local function enableVehicleEffectsFrom(ent)

    local vehGlobal = ensureVehicleGlobal()
	if not ent or not ent.vehicle then return end
    if not vehGlobal then return end

    local v = ent.vehicle
    if not v and ent.camera and ent.camera.sway then
        v = {
            sway = {
                ampX = (ent.camera.sway.ampX or 0) * 0.6,
                ampY = (ent.camera.sway.ampY or 0) * 0.6,
                ampZ = (ent.camera.sway.ampZ or 0) * 0.6,
                rollAmp = (ent.camera.sway.rollAmp or 0) * 0.6,
                period = ent.camera.sway.period or 2.8,
                phaseX = ent.camera.sway.phaseX, phaseY = ent.camera.sway.phaseY,
                phaseZ = ent.camera.sway.phaseZ, phaseRoll = ent.camera.sway.phaseRoll,
                noiseAmp = ent.camera.sway.noiseAmp, noisePeriod = ent.camera.sway.noisePeriod,
                fpScale = ent.camera.sway.fpScale or 1.0, tpScale = ent.camera.sway.tpScale or 0.6
            },
            fov = { amplitudePct = 0, period = 3.0 },
            steer = { amplitude = 0.15, period = 2.6, deadzoneHold = 0.15 }
        }
    end
    if not v then return end

    vehGlobal.enabled = true

    if v.sway then
        vehGlobal.cam.ampX      = v.sway.ampX or vehGlobal.cam.ampX
        vehGlobal.cam.ampY      = v.sway.ampY or vehGlobal.cam.ampY
        vehGlobal.cam.ampZ      = v.sway.ampZ or vehGlobal.cam.ampZ
        vehGlobal.cam.rollAmp   = v.sway.rollAmp or vehGlobal.cam.rollAmp
        vehGlobal.cam.period    = v.sway.period or vehGlobal.cam.period
        vehGlobal.cam.phaseX    = v.sway.phaseX or vehGlobal.cam.phaseX
        vehGlobal.cam.phaseY    = v.sway.phaseY or vehGlobal.cam.phaseY
        vehGlobal.cam.phaseZ    = v.sway.phaseZ or vehGlobal.cam.phaseZ
        vehGlobal.cam.phaseRoll = v.sway.phaseRoll or vehGlobal.cam.phaseRoll
        vehGlobal.cam.noiseAmp  = v.sway.noiseAmp or vehGlobal.cam.noiseAmp
        vehGlobal.cam.noisePeriod = v.sway.noisePeriod or vehGlobal.cam.noisePeriod
        vehGlobal.cam.fpScale   = v.sway.fpScale or vehGlobal.cam.fpScale
        vehGlobal.cam.tpScale   = v.sway.tpScale or vehGlobal.cam.tpScale
    end
    if v.fov then
        vehGlobal.fov.amplitudePct = v.fov.amplitudePct or 0
        vehGlobal.fov.period       = v.fov.period or 3.0
    else
        vehGlobal.fov.amplitudePct = 0
    end
    if v.steer then
        vehGlobal.steer.amplitude    = v.steer.amplitude or vehGlobal.steer.amplitude
        vehGlobal.steer.period       = v.steer.period or vehGlobal.steer.period
        vehGlobal.steer.deadzoneHold = v.steer.deadzoneHold or vehGlobal.steer.deadzoneHold
    end
end

local function disableVehicleEffectsIfNoneLeft(ent)
    if not g_DrunkVehicleEffects then return end
    for _, e in ipairs(ItemEffects.active or {}) do
        if e ~= ent and (e.vehicle or e.camera) then
            return
        end
    end
    g_DrunkVehicleEffects.enabled = false
    if g_DrunkVehicleEffects.fov then
        g_DrunkVehicleEffects.fov.amplitudePct = 0
    end
end

-- ========= AUDIO HELPERS =========

function ItemEffects._playlistHasPlayable(audios)
    return (audios and audios.playItems and #audios.playItems > 0) or false
end

function ItemEffects:audioAlwaysStart(ent)
    if not ent or not ent.audios or not ent.audios.items then return end
    if ent.__audioAlways then return end
    ent.__audioAlways = {}
    for _, it in ipairs(ent.audios.items) do
        if it and it.always and it.filenameAbs and fileExists(it.filenameAbs) then
            local sid = createSample and createSample(("ie_alw_%s_%d"):format(tostring(ent.id), math.random(1, 999999))) or nil
            if sid then
                local loop  = (it.loop == true)
                local vol   = tonumber(it.volume) or 0.8
                local pitch = tonumber(it.pitch)  or 1.0
                pcall(loadSample, sid, it.filenameAbs, loop)
                pcall(playSample, sid, 0, 1, vol, pitch, 0)
                table.insert(ent.__audioAlways, { sampleId = sid })
            end
        end
    end
end

function ItemEffects:audioAlwaysStop(ent)
    if not ent or not ent.__audioAlways then return end
    for _, rec in ipairs(ent.__audioAlways) do
        if rec.sampleId then
            pcall(stopSample,   rec.sampleId, 0, 0)
            pcall(deleteSample, rec.sampleId)
        end
    end
    ent.__audioAlways = nil
end

function ItemEffects:audioStopRuntime(ent)
    if not ent then return end
    local rt = ent._audioRuntime
    if rt and rt.sampleId then
        pcall(stopSample,   rt.sampleId, 0, 0)
        pcall(deleteSample, rt.sampleId)
    end
    ent._audioRuntime = nil
end

function ItemEffects:audioPlaylistStart(ent)
    if not ent or not ent.audios then return end
    local a = ent.audios
    if not a.playItems or #a.playItems == 0 then return end
    if ent.__audioPL and ent.__audioPL.active then return end

    ent.__audioPL = {
        active = true,
        idx    = 1,
        state  = "starting",
        sid    = nil,
        waitMs = 0,
        leftMs = 0,
    }
end

function ItemEffects:audioUpdate(ent, dt)
    if not ent or not ent.audios or not ent.__audioPL then return end
    local a  = ent.audios
    local pl = ent.__audioPL
    local list = a.playItems or {}
    if #list == 0 then return end

    local dtMs = tonumber(dt) or 0
    if dtMs <= 0 then dtMs = 16 end

    local function nDelaySec(it)
        return tonumber(it.delaySec or it.delay or 0) or 0
    end
    local function nDurSec(it)
        return tonumber(it.durationSec or it.duration or 1) or 1
    end

    if pl.state == "waiting" then
        pl.waitMs = math.max(0, (pl.waitMs or 0) - dtMs)
        if pl.waitMs <= 0 then pl.state = "starting" end
        return
    end

    if pl.state == "starting" then
        local it = list[pl.idx]
        if not it or not it.filenameAbs or not fileExists(it.filenameAbs) then
            pl.state = "advance"
            return
        end

        local sid = createSample and createSample(("ie_pl_%s_%d"):format(tostring(ent.id), math.random(1000000))) or nil
        if not sid then
            pl.state = "advance"
            return
        end

        local loop  = (it.loop == true)
        local vol   = tonumber(it.volume) or 0.8
        local pitch = tonumber(it.pitch)  or 1.0

        pcall(loadSample, sid, it.filenameAbs, loop)
        pcall(playSample, sid, 0, 1, vol, pitch, 0)

        pl.sid    = sid
        pl.leftMs = math.max(50, math.floor(nDurSec(it) * 1000))
        pl.state  = "playing"
        return
    end

    if pl.state == "playing" then
        pl.leftMs = math.max(0, (pl.leftMs or 0) - dtMs)
        if pl.leftMs > 0 then return end
        if pl.sid then
            pcall(stopSample,   pl.sid, 0, 0)
            pcall(deleteSample, pl.sid)
            pl.sid = nil
        end
        pl.state = "advance"
        return
    end

    if pl.state == "advance" then
        pl.idx = (pl.idx or 1) + 1
        if pl.idx > #list then
            if a.loop then
                pl.idx = 1
            else
                self:audioStopPlaylist(ent)
                return
            end
        end
        local itNext = list[pl.idx]
        pl.waitMs = math.floor(nDelaySec(itNext) * 1000)
        pl.state  = (pl.waitMs > 0) and "waiting" or "starting"
        return
    end

    pl.state  = "starting"
    pl.waitMs = 0
end

function ItemEffects:audioStopPlaylist(ent)
    if not ent then return end
    if ent._playlistSampleId then
        pcall(stopSample,   ent._playlistSampleId, 0, 0)
        pcall(deleteSample, ent._playlistSampleId)
        ent._playlistSampleId = nil
    end
    if ent.__audioPL and ent.__audioPL.sid then
        pcall(stopSample,   ent.__audioPL.sid, 0, 0)
        pcall(deleteSample, ent.__audioPL.sid)
    end
    ent.__audioPL = nil
end

function runLater(fn, delayMs)
    if not fn then return end
    local t = { fn = fn, timeLeft = delayMs or 0 }
    function t:update(dt)
        self.timeLeft = self.timeLeft - dt
        if self.timeLeft <= 0 then
            g_updateables:remove(self)
            pcall(self.fn)
        end
    end
    g_updateables:add(t)
end

local function _fileOk(path) return path and path ~= "" and fileExists(path) end

local function _makeSampleName(prefix, entId)
    return string.format("%s_%s_%d", prefix or "ie", tostring(entId or "x"), math.random(1000000))
end

local function _stopSampleSafe(sid) if sid then pcall(stopSample, sid, 0, 0) end end
local function _deleteSampleSafe(sid) if sid then pcall(deleteSample, sid) end end

function ItemEffects:canConsume(effectId, effectsFileAbs)
    local def = ItemEffects.getEffectDef and ItemEffects:getEffectDef(effectId, effectsFileAbs) or nil
    if not def then
        return false, "notFound"
    end

    local list = ItemEffects.active or {}
    local now = g_time or 0

    for _, ent in ipairs(list) do
        local alive = (now < (ent.untilMs or 0)) or (ent.visual and (ent.visual.master or 0) > 0.01)
        if alive then
            if tostring(ent.id) ~= tostring(def.id) then
                return false, "busy", { activeId = ent.id, activeTitle = ent.title }
            else
                local st = ent.stacks or 1
                local mx = ent.maxStacks or def.maxStacks or 1
                if st >= mx then
                    return false, "atCap", { stacks = st, maxStacks = mx }
                end
            end
        end
    end

    return true, nil, { stacks = 0, maxStacks = def.maxStacks or 1 }
end

function ItemEffects:getActiveEffectInfo()
    local list = ItemEffects.active or {}
    local now  = g_time or 0

    local primaryId = ItemEffects._hudPrimaryId
    if primaryId and primaryId ~= "" then
        for _, ent in ipairs(list) do
            if tostring(ent.id) == tostring(primaryId) then
                local alive = (now < (ent.untilMs or 0)) or (ent.visual and (ent.visual.master or 0) > 0.01)
                if alive then
                    local left = math.max(0, (ent.untilMs or 0) - now)
                    return {
                        id          = ent.id,
                        title       = ent.title or ent.id,
                        stacks      = ent.stacks or 1,
                        maxStacks   = ent.maxStacks or 1,
                        timeLeftSec = math.ceil(left / 1000),
                    }
                end
                break
            end
        end
    end

    for _, ent in ipairs(list) do
        local alive = (now < (ent.untilMs or 0)) or (ent.visual and (ent.visual.master or 0) > 0.01)
        if alive then
            local left = math.max(0, (ent.untilMs or 0) - now)
            return {
                id          = ent.id,
                title       = ent.title or ent.id,
                stacks      = ent.stacks or 1,
                maxStacks   = ent.maxStacks or 1,
                timeLeftSec = math.ceil(left / 1000),
            }
        end
    end

    return nil
end

function ItemEffects:startEffect(core, effectId, effectsFileAbs)
    local def = ItemEffects:getEffectDef(effectId, effectsFileAbs)
    if not def then
        print("[ItemEffects] Effect not found: "..tostring(effectId))
        return false, "notFound"
    end

    local ok, reason = self:canConsume(effectId, effectsFileAbs)
    if not ok then return false, reason end

    ItemEffects.active = ItemEffects.active or {}
	local now = ItemEffects:getNowMs(0)

    for _, ent in ipairs(ItemEffects.active) do
        if tostring(ent.id) == tostring(def.id) and now < (ent.untilMs or 0) then
            ent.stacks  = math.min((ent.stacks or 1) + 1, ent.maxStacks or def.maxStacks or 1)
            ent.untilMs = (ent.untilMs or now) + math.floor((def.durationSec or 0) * 1000)

            if applyCameraFrom          then pcall(applyCameraFrom,          ent, true) end
            if applyWobbleFrom          then pcall(applyWobbleFrom,          ent, true) end
            if enableVehicleEffectsFrom then pcall(enableVehicleEffectsFrom, ent) end
            ent._reapplyAtMs = now + 200

            ItemEffects._hudPrimaryId = ent.id

            if core then core.saveDirty = true end
            print(string.format("[ItemEffects] Stacked '%s' -> %d stacks, +%ds",
                ent.id, ent.stacks or 1, math.floor(def.durationSec or 0)))
            return true
        end
    end

    local ent = {
        id            = def.id,
        title         = def.title or def.id,
        base          = def,
        startedAtMs   = now,
        untilMs       = now + math.floor((def.durationSec or 0) * 1000),
        nextTickMs    = now + math.floor((def.tickSec or 1) * 1000),
        tickSec       = def.tickSec or 1,
        hungerPerTick = def.hungerPerTick or 0,
        vigorPerTick  = def.vigorPerTick  or 0,
        stacks        = 1,
        maxStacks     = def.maxStacks or 1,
        visual        = nil,
        camera        = (def.camera and { sway = def.camera.sway }) or nil,
        wobble        = def.wobble,
        vehicle       = def.vehicle,
        audio         = nil,
        audios        = nil,
        effectsFileAbs= effectsFileAbs
    }

    if def.visual and def.visual.overlays and #def.visual.overlays > 0 then
        local v = {
            loop         = def.visual.loop == true,
            crossFadeSec = def.visual.crossFadeSec or 0.35,
            fadeInSec    = def.visual.fadeInSec or 0.8,
            fadeOutSec   = def.visual.fadeOutSec or 1.0,
            overlays     = {},
            master       = 1.0,
            idx          = 1,
            timeLeft     = (def.visual.overlays[1] and def.visual.overlays[1].durationSec) or 2.0
        }
        for _, o in ipairs(def.visual.overlays) do
            local ov = ensureOverlay(o.filenameAbs)
            if ov then table.insert(v.overlays, { def=o, ov=ov, a=0.0 }) end
        end
        if #v.overlays > 0 then
            v.overlays[1].a = 1.0
            ent.visual = v
        end
    end

    if def.audios then
        local playItems = {}
        local srcItems  = def.audios.items or {}
        for _, it in ipairs(srcItems) do
            if it and not it.always and it.filenameAbs and fileExists(it.filenameAbs) then
                table.insert(playItems, it)
            end
        end
        ent.audios = { loop = def.audios.loop == true, items = srcItems, playItems = playItems }
		if ItemEffects.audioAlwaysStart then pcall(ItemEffects.audioAlwaysStart, ItemEffects, ent) end
		if ItemEffects._playlistHasPlayable and ItemEffects.audioPlaylistStart
		   and ItemEffects._playlistHasPlayable(ent.audios) then
			pcall(ItemEffects.audioPlaylistStart, ItemEffects, ent)
		end

    elseif def.audio and def.audio.filenameAbs and fileExists(def.audio.filenameAbs) then
        local name = "itemEffect_"..tostring(ent.id).."_"..tostring(math.random(100000))
        ent.audio = {
            filenameAbs = def.audio.filenameAbs,
            volume      = def.audio.volume or 0.8,
            pitch       = def.audio.pitch  or 1.0,
            loop        = def.audio.loop   or false
        }
        ent.audio.sampleId = createSample and createSample(name) or nil
        if ent.audio.sampleId then
            pcall(function() loadSample(ent.audio.sampleId, ent.audio.filenameAbs, ent.audio.loop) end)
            pcall(function() playSample(ent.audio.sampleId, 0, 1, ent.audio.volume, ent.audio.pitch, 0) end)
        end
    end

    table.insert(ItemEffects.active, ent)

    if applyCameraFrom          then pcall(applyCameraFrom,          ent, true) end
    if applyWobbleFrom          then pcall(applyWobbleFrom,          ent, true) end
    if enableVehicleEffectsFrom then pcall(enableVehicleEffectsFrom, ent) end
    ent._reapplyAtMs = now + 200

    ItemEffects._hudPrimaryId = ent.id

    if core and core.updateHungerSpeeds and not (core.isEnergyActive and core:isEnergyActive()) then
        local pl = (g_playerSystem and g_playerSystem:getLocalPlayer()) or (g_currentMission and g_currentMission.player)
        if pl then core:updateHungerSpeeds(pl) end
    end

    print(string.format("[ItemEffects] Started effect '%s' for %ds", ent.id, math.floor((ent.untilMs-now)/1000)))
    if core then core.saveDirty = true end
    return true
end


local function applyTick(core, ent)
    local pl = playerLocal(); if not pl then return end

    local mx = math.max(1, ent.maxStacks or 1)
    local st = math.max(1, ent.stacks or 1)
    local k  = st / mx

    local function scale(v)
        local r = (v or 0) * k
        if r > 0 then r = math.max(1, math.floor(r + 1e-6))
        elseif r < 0 then r = math.min(-1, math.ceil(r - 1e-6))
        else r = 0 end
        return r
    end

    local hpt = scale(ent.hungerPerTick or 0)
    local vpt = scale(ent.vigorPerTick  or 0)

    if hpt ~= 0 then
        pl.hunger = math.max(0, math.min(100, (pl.hunger or core.hunger or 100) + hpt))
        core.hunger = pl.hunger
    end
    if vpt ~= 0 then
        pl.vigor  = math.max(0, math.min(100, (pl.vigor or core.vigor or 100) + vpt))
        core.vigor = pl.vigor
    end
    core.saveDirty = true
end

local function updateVisual(ent, dt)
    local v = ent.visual; if not v or not v.overlays or #v.overlays == 0 then return end

    local now = nowMs()
    local timeLeftEff = math.max(0, ent.untilMs - now)

    local fadeInK  = math.min(1.0, (v.master or 0) + (dt/300))          -- ~0.3с вход
    local fadeOutK = (timeLeftEff < 400) and math.max(0, 1 - (400 - timeLeftEff)/400) or 1.0 -- ~0.4с выход
    v.master = math.max(0.0, math.min(1.0, fadeInK * fadeOutK))

    local cross = math.max(0.0, v.crossFadeSec or 0.35)

    if v._xf_t and v._xf_t > 0 then
        v._xf_t = v._xf_t - dt/1000.0
        local t = 1.0 - math.max(0.0, v._xf_t)/math.max(0.0001, cross)
        local from = v._xf_from or 1
        local to   = v._xf_to   or 1
        local tt = 0.5 - 0.5*math.cos(t*math.pi) -- синус-ин-аут
        if v.overlays[from] then v.overlays[from].a = 1.0 - tt end
        if v.overlays[to]   then v.overlays[to].a   = tt end

        if v._xf_t <= 0 then
            if v.overlays[from] then v.overlays[from].a = 0.0 end
            if v.overlays[to]   then v.overlays[to].a   = 1.0 end
            v._xf_t, v._xf_from, v._xf_to = nil, nil, nil
            v.timeLeft = v.nextDuration or 2.0
            v.nextDuration = nil
        end
        return
    end

    v.timeLeft = (v.timeLeft or 0) - dt/1000.0
    if v.timeLeft > 0 then return end
    if #v.overlays <= 1 then return end

    local cur = v.idx or 1
    local nxt = cur + 1
    if nxt > #v.overlays then
        nxt = v.loop and 1 or #v.overlays
    end

    v._xf_t    = cross
    v._xf_from = cur
    v._xf_to   = nxt
    v.idx      = nxt
    v.nextDuration = v.overlays[nxt].def.durationSec or 2.0
    v.timeLeft = cross

    if v.overlays[nxt] then v.overlays[nxt].a = 0.0 end
    if v.overlays[cur] then v.overlays[cur].a = 1.0 end
end

local function finishEntry(ent)
    if ItemEffects.audioStopPlaylist then ItemEffects:audioStopPlaylist(ent) end
    if ItemEffects.audioStopRuntime  then ItemEffects:audioStopRuntime(ent)  end
    if ItemEffects.audioAlwaysStop   then ItemEffects:audioAlwaysStop(ent)   end

    if ent.audio and ent.audio.sampleId then
        pcall(function() stopSample(ent.audio.sampleId, 0, 0) end)
        pcall(function() deleteSample(ent.audio.sampleId) end)
        ent.audio.sampleId = nil
    end

    if ent.visual and ent.visual.overlays then
        for _, o in ipairs(ent.visual.overlays) do
            if o.ov then pcall(function() o.ov:delete() end) end
        end
    end
    ent.visual = nil

    if applyCameraFrom then pcall(applyCameraFrom, ent, false) end
    if applyWobbleFrom then pcall(applyWobbleFrom, ent, false) end
    if disableVehicleEffectsIfNoneLeft then pcall(disableVehicleEffectsIfNoneLeft, ent) end
end

local function _updateVisualVignette(ent, nowMs, dtMs)
    local v = ent.visual
    if not v then return end

    local dtSec = math.max(0.0, (tonumber(dtMs) or 0) / 1000.0)
    local cross = math.max(0.0001, tonumber(v.crossFadeSec or 0.2))
    local fin   = math.max(0.0,   tonumber(v.fadeInSec    or 0.8))
    local fout  = math.max(0.0,   tonumber(v.fadeOutSec   or 1.0))

    v.master = v.master or 0.0
    v.idx    = v.idx    or 1

    local mx = math.max(1, tonumber(ent.maxStacks or 1) or 1)
    local st = math.max(1, tonumber(ent.stacks    or 1) or 1)
    v.stackK = math.max(0.0, math.min(1.0, st / mx))

    local target = 1.0

    local tSinceStart = math.max(0, (nowMs - (ent.startedAtMs or nowMs))) / 1000.0
    if fin > 0 then
        target = math.min(target, math.min(1.0, tSinceStart / fin))
    end

    if v.loop ~= true and fout > 0 then
        local remainSec = math.max(0, (ent.untilMs or nowMs) - nowMs) / 1000.0
        target = math.min(target, math.min(1.0, remainSec / fout))
    end

    local step = dtSec / cross
    if target > v.master then
        v.master = math.min(1.0, v.master + step)
    else
        v.master = math.max(0.0, v.master - step)
    end

    local list = v.overlays or {}
    if #list > 0 then
        local cur = list[v.idx]
        if cur then
            v.timeLeft = (v.timeLeft ~= nil) and v.timeLeft or (cur.def.durationSec or cur.def.duration or 2.0)
            v.timeLeft = v.timeLeft - dtSec
            if v.timeLeft <= 0 then
                if v.idx < #list then
                    v.idx = v.idx + 1
                elseif v.loop == true then
                    v.idx = 1
                end
                cur = list[v.idx]
                v.timeLeft = (cur and (cur.def.durationSec or cur.def.duration or 2.0)) or 0
            end
        end

        for i, o in ipairs(list) do
            o.a = (i == v.idx) and 1.0 or 0.0
        end
    end
end

function ItemEffects:update(core, dt)
    ItemEffects.active = ItemEffects.active or {}

    local dtMs = tonumber(dt) or 0
    if dtMs <= 0 then dtMs = 16 end
    local now  = ItemEffects:getNowMs(dtMs)

    _G.g_ItemEffectsNowMs = now

    local hs = core
    if type(hs) ~= "table" or (hs.addHungerDelta == nil and hs.addVigorDelta == nil) then
        hs = (ItemEffects.core)
          or (HungerSystem and HungerSystem.instance)
          or (g_currentMission and (g_currentMission.hungerSystem or g_currentMission.hungerSystemCore))
    end

    local function clamp01_100(v) v = tonumber(v) or 0; if v < 0 then return 0 elseif v > 100 then return 100 end; return v end
    local function _applyHungerDelta(delta)
        delta = tonumber(delta) or 0
        if hs and type(hs.addHungerDelta) == "function" then
            return pcall(hs.addHungerDelta, hs, delta)
        end
        local pl = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)
        local cur = (pl and pl.hunger) or (hs and hs.hunger) or 100
        local newv = clamp01_100(cur + delta)
        if pl then pl.hunger = newv end
        if hs then hs.hunger = newv end
        if hs and hs.updateHungerSpeeds and pl then pcall(hs.updateHungerSpeeds, hs, pl) end
        if hs then hs.saveDirty = true end
    end
    local function _applyVigorDelta(delta)
        delta = tonumber(delta) or 0
        if hs and type(hs.addVigorDelta) == "function" then
            return pcall(hs.addVigorDelta, hs, delta)
        end
        local pl = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)
        local cur = (pl and pl.vigor) or (hs and hs.vigor) or 100
        local newv = clamp01_100(cur + delta)
        if pl then pl.vigor = newv end
        if hs then hs.vigor = newv end
        if hs and hs.updateHungerSpeeds and pl then pcall(hs.updateHungerSpeeds, hs, pl) end
        if hs then hs.saveDirty = true end
    end

    for _, ent in ipairs(ItemEffects.active) do
        if ent._reapplyAtMs and now >= ent._reapplyAtMs then
            ent._reapplyAtMs = nil
            if ent.camera  and applyCameraFrom          then pcall(applyCameraFrom,          ent, true) end
            if ent.wobble  and applyWobbleFrom          then pcall(applyWobbleFrom,          ent, true) end
            if ent.vehicle and enableVehicleEffectsFrom then pcall(enableVehicleEffectsFrom, ent) end
        end
    end

    for _, ent in ipairs(ItemEffects.active) do
        local isPostFade = (ent._postFade == true)

        if not isPostFade and ent.tickSec and ent.tickSec > 0 then
            if not ent.nextTickMs then
                ent.nextTickMs = now + math.floor((ent.tickSec or 1) * 1000)
            end
            if now >= (ent.nextTickMs or 0) then
                ent.nextTickMs = now + math.floor((ent.tickSec or 1) * 1000)
                if ent.hungerPerTick and ent.hungerPerTick ~= 0 then _applyHungerDelta(ent.hungerPerTick) end
                if ent.vigorPerTick  and ent.vigorPerTick  ~= 0 then _applyVigorDelta (ent.vigorPerTick)  end
            end
        end

        if ent.__audioPL and ItemEffects.audioUpdate then
            ItemEffects:audioUpdate(ent, dtMs)
        end

        if ent.visual and _updateVisualVignette then
            _updateVisualVignette(ent, now, dtMs)
        end
    end

    local i = 1
    while i <= #ItemEffects.active do
        local ent = ItemEffects.active[i]

        local expired = (ent.untilMs or 0) <= now
        local removeNow = false

        if not expired then
            i = i + 1

        else
            if ent._postFade then
                if now >= (ent._removeAtMs or now) then
                    removeNow = true
                else
                    i = i + 1
                end

            else
                local sway = ent.camera and ent.camera.sway or nil
                local fadeOutSec = (sway and tonumber(sway.fadeOutSec or 0) or 0)
                if fadeOutSec > 0 then
                    ent._postFade = true
                    ent._removeAtMs = now + math.floor(fadeOutSec * 1000)
                    ent.untilMs = ent._removeAtMs

                    if ItemEffects.audioStopPlaylist then ItemEffects:audioStopPlaylist(ent) end
                    if ItemEffects.audioStopRuntime  then ItemEffects:audioStopRuntime(ent)  end
                    if ItemEffects.audioAlwaysStop   then ItemEffects:audioAlwaysStop(ent)   end
                    if ent.audio and ent.audio.sampleId then
                        pcall(function() stopSample(ent.audio.sampleId, 0, 0) end)
                        pcall(function() deleteSample(ent.audio.sampleId) end)
                        ent.audio.sampleId = nil
                    end
                    i = i + 1
                else
                    removeNow = true
                end
            end
        end

        if removeNow then

            if ent.camera  and applyCameraFrom                then pcall(applyCameraFrom,  ent, false) end
            if ent.wobble  and applyWobbleFrom                then pcall(applyWobbleFrom,  ent, false) end
            if disableVehicleEffectsIfNoneLeft                then pcall(disableVehicleEffectsIfNoneLeft, ent) end

            if ItemEffects.audioStopPlaylist then ItemEffects:audioStopPlaylist(ent) end
            if ItemEffects.audioStopRuntime  then ItemEffects:audioStopRuntime(ent)  end
            if ItemEffects.audioAlwaysStop   then ItemEffects:audioAlwaysStop(ent)   end
            if ent.audio and ent.audio.sampleId then
                pcall(function() stopSample(ent.audio.sampleId, 0, 0) end)
                pcall(function() deleteSample(ent.audio.sampleId) end)
                ent.audio.sampleId = nil
            end

            if ent.visual and ent.visual.overlays then
                for _, o in ipairs(ent.visual.overlays) do
                    if o.ov then pcall(function() o.ov:delete() end) end
                end
            end
            ent.visual = nil

            if ItemEffects._hudPrimaryId == ent.id then ItemEffects._hudPrimaryId = nil end
            table.remove(ItemEffects.active, i)
            if hs then hs.saveDirty = true end
        end
    end
end

function ItemEffects:draw(core)
    if not ItemEffects.active or #ItemEffects.active == 0 then return end
    for _, ent in ipairs(ItemEffects.active) do
        local v = ent.visual
        if v and v.overlays and #v.overlays > 0 and (v.master or 0) > 0 then
            local master = v.master or 0
            local stackK = math.max(0.0, math.min(1.0, v.stackK or ((ent.stacks or 1) / math.max(1, ent.maxStacks or 1))))
            for _, o in ipairs(v.overlays) do
                if o.ov and (o.a or 0) > 0.001 then
                    local aMax = (o.def and o.def.alphaMax) or 0.45
                    o.ov:setColor(1,1,1, (master * aMax * (o.a or 0) * stackK))
                    o.ov:render()
                end
            end
        end
    end
end

function ItemEffects:clearAll()
    for _, ent in ipairs(ItemEffects.active) do finishEntry(ent) end
    ItemEffects.active = {}
end

