-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

Inventory = {
  Hud = {},
  Dialogs = {},
  I18n = {},
  Const = {},
  Storage = {},
  API = {},
}

Inventory.modName = g_currentModName
Inventory.modDir  = g_currentModDirectory

source(Inventory.modDir .. "scripts/Inventory/InventoryHUD.lua")
source(Inventory.modDir .. "scripts/Inventory/InventoryDialogs.lua")

addModEventListener(Inventory)

function Inventory.getNamespaces()
  return Inventory, Inventory.Hud, Inventory.Dialogs, Inventory.I18n
end

-- ===== CONST / STYLE =====
function Inventory:setupConst()
  self.Const = {
    COLORS = {
      WHITE = {1,1,1,1},
      BLACK = {0,0,0,1},
      BG = {0.02,0.02,0.02,0.96},
      BG_DIM = {0,0,0,0.65},
      ACCENT = {0.238, 0.462, 0, 1},
      ACCENT_HOVER = {0.08, 0.150, 0, 1},
      BTN = {0.16, 0.16, 0.16, 1},
      BTN_HOVER = {0.22, 0.22, 0.22, 1},
    }
  }
  -- сетка
  self.gridCols = 7
  self.gridRows = 5
  self.maxSlots = self.gridCols * self.gridRows

  self._lmbCooldown = 0
  self._rmbCooldown = 0
end

function Inventory.I18n.init(QS, Hud, Dialogs, I18n)
  local texts = g_i18n.modEnvironments[Inventory.modName].texts or {}
  function I18n:t(id) return texts[id] and texts[id] or id end
end

local function _genId()
  Inventory.__uid = (Inventory.__uid or 0) + 1
  return string.format("itm_%d_%d", g_time or 0, Inventory.__uid)
end

local function _ensureDir(path)
  if createFolder ~= nil then createFolder(path) end
end

function Inventory:getSettingsDir()
  local base = getUserProfileAppPath() .. "modSettings/"
  local modFolder = base .. (Inventory.modName or "Inventory") .. "/"
  _ensureDir(base); _ensureDir(modFolder)
  return modFolder
end

function Inventory:getSavePath()
  return self:getSettingsDir() .. "inventoryHUD.xml"
end

function Inventory:getLegacySavePath()
  if g_currentMission and g_currentMission.missionInfo and g_currentMission.missionInfo.savegameDirectory then
    return g_currentMission.missionInfo.savegameDirectory .. "/inventoryHUD.xml"
  end
  return nil
end

function Inventory.Storage:load(inv)
  inv.items = {}
  for i=1, inv.maxSlots do inv.items[i]=nil end

  local path = Inventory:getSavePath()
  local loadedFromLegacy = false
  if not fileExists(path) then
    local legacy = Inventory:getLegacySavePath()
    if legacy and fileExists(legacy) then path = legacy; loadedFromLegacy = true else return end
  end

  local xml = loadXMLFile("InventoryXML", path)
  local i = 0
  while true do
    local key = string.format("inventory.slots.slot(%d)", i)
    if not hasXMLProperty(xml, key) then break end
    local slotIndex  = getXMLInt(xml, key .. "#index")
    local id         = getXMLString(xml, key .. "#id") or _genId()
    local name       = getXMLString(xml, key .. "#name") or ""
    local desc       = getXMLString(xml, key .. "#desc") or ""
    local icon       = getXMLString(xml, key .. "#icon") or ""
    local kind       = getXMLString(xml, key .. "#kind") or "food"
    local hunger     = getXMLInt(xml, key .. "#hungerGain") or 0
    local vigor      = getXMLInt(xml, key .. "#vigorGain")  or 0
    local effectId   = getXMLString(xml, key .. "#effectId") or ""
    local effectsFile= getXMLString(xml, key .. "#effectsFile") or ""
    if slotIndex and slotIndex >= 1 and slotIndex <= inv.maxSlots then
      inv.items[slotIndex] = {
        uid=_genId(), id=id,
        name=name, desc=desc, icon=icon,
        kind=kind, hungerGain=hunger, vigorGain=vigor,
        effectId=effectId, effectsFile=effectsFile
      }
    end
    i = i + 1
  end
  delete(xml)

  if loadedFromLegacy then inv.needsSave = true; inv._forceImmediateSave = true end
end

function Inventory.Storage:save(inv)
  local dir = Inventory:getSettingsDir()
  local path = dir .. "inventoryHUD.xml"
  local xml = createXMLFile("InventoryXML", path, "inventory")
  local idx = 0
  for slot=1, inv.maxSlots do
    local it = inv.items[slot]
    if it then
      local key = string.format("inventory.slots.slot(%d)", idx)
      setXMLInt   (xml, key.."#index",       slot)
      setXMLString(xml, key.."#id",          tostring(it.id or it.uid))
      setXMLString(xml, key.."#name",        tostring(it.name or ""))
      setXMLString(xml, key.."#desc",        tostring(it.desc or ""))
      setXMLString(xml, key.."#icon",        tostring(it.icon or ""))
      setXMLString(xml, key.."#kind",        tostring(it.kind or "food"))
      setXMLInt   (xml, key.."#hungerGain",  tonumber(it.hungerGain or 0))
      setXMLInt   (xml, key.."#vigorGain",   tonumber(it.vigorGain  or 0))
      setXMLString(xml, key.."#effectId",    tostring(it.effectId or ""))
      setXMLString(xml, key.."#effectsFile", tostring(it.effectsFile or ""))
      idx = idx + 1
    end
  end
  saveXMLFile(xml); delete(xml)
end

-- ===== API =====
function Inventory.API.firstFreeSlot()
  for i=1, Inventory.maxSlots do
    if Inventory.items and Inventory.items[i] == nil then return i end
  end
  return nil
end

function Inventory.API.addItemInstance(item)
  local inv = Inventory
  local free = Inventory.API.firstFreeSlot()
  if not free then return false, "noSpace" end

  local inst = {
    uid=_genId(),
    id  = item.id or _genId(),
    name = item.name or "",
    desc = item.desc or item.description or "",
    icon = item.icon or "",
    kind = (item.kind or "food"),
    hungerGain = tonumber(item.hungerGain or 0) or 0,
    vigorGain  = tonumber(item.vigorGain  or 0) or 0,
    effectId   = item.effectId or "",
    effectsFile= item.effectsFile or ""
  }
  inv.items[free] = inst
  inv.needsSave = true
  if inv.Hud.isVisible then inv.Hud:refresh() end
  return true, free
end

function Inventory.API.removeAt(slotIndex)
  local inv = Inventory
  if slotIndex < 1 or slotIndex > inv.maxSlots then return false end
  inv.items[slotIndex] = nil
  inv.needsSave = true
  if inv.Hud.isVisible then inv.Hud:refresh() end
  return true
end

local function _clamp01(x) return math.max(0, math.min(100, x)) end

local function _resolveEffectsFileForInventory(rawPath)
  -- Если в слоте уже лежит абсолютный путь и файл существует — используем его
  if rawPath ~= nil and rawPath ~= "" then
    if fileExists(rawPath) then
      return rawPath
    end

    -- Пытаемся собрать путь как относительный
    local baseDir = (HungerSystem and HungerSystem.modDirectory)
                 or (ItemEffects   and ItemEffects.modDirectory)
                 or Inventory.modDir
                 or g_currentModDirectory
                 or ""

    if baseDir ~= "" then
      local candidate = Utils.getFilename(rawPath, baseDir)
      if candidate and fileExists(candidate) then
        return candidate
      end
    end
  end

  -- Если ничего не указано или не нашли — берём дефолтный itemEffects.xml из LiveFarmer / ItemEffects
  local baseDir = (HungerSystem and HungerSystem.modDirectory)
               or (ItemEffects   and ItemEffects.modDirectory)
               or Inventory.modDir
               or g_currentModDirectory
               or ""

  if baseDir ~= "" then
    local candidate = Utils.getFilename("itemEffects.xml", baseDir)
    if candidate and fileExists(candidate) then
      return candidate
    end
  end

  -- В крайнем случае вернём nil — HungerSystem сам решит, что делать
  return nil
end


function Inventory.API.consume(item)
  if not item then return false end

  local hs   = _G.HungerSystem
  local ok   = false
  local kind = tostring(item.kind or "food")
  local hG   = tonumber(item.hungerGain or 0) or 0
  local vG   = tonumber(item.vigorGain  or 0) or 0

  ----------------------------------------------------------------
  -- 1) ПРЕДМЕТ С ЭФФЕКТОМ (kind = "effect")
  ----------------------------------------------------------------
  if kind == "effect" and hs and hs.applyItemEffect then
    local effectId = tostring(item.effectId or "")

    -- если id вообще не указан — нет смысла дергать систему эффектов
    if effectId == "" then
      return false
    end

    -- аккуратно резолвим путь к XML:
    --  - если в слоте лежит абсолютный/относительный путь — пробуем его
    --  - если пусто — падаем на дефолтный itemEffects.xml
    local effectsFileAbs = _resolveEffectsFileForInventory(item.effectsFile)

    ok = hs:applyItemEffect(effectId, effectsFileAbs) and true or false

  ----------------------------------------------------------------
  -- 2) Обычная еда / всё остальное
  ----------------------------------------------------------------
  else
    ok = true
    if hs then
      if hs.addHungerDelta then
        hs:addHungerDelta(hG)
      elseif hs.hunger ~= nil then
        hs.hunger = _clamp01((hs.hunger or 0) + hG)
      end

      if hs.addVigorDelta then
        hs:addVigorDelta(vG)
      elseif hs.vigor ~= nil then
        hs.vigor = _clamp01((hs.vigor or 0) + vG)
      end
    end
  end

  ----------------------------------------------------------------
  -- 3) Обновляем HUD, если что-то применилось
  ----------------------------------------------------------------
  if ok then
    if _G.HungerHUD and HungerHUD.refreshImmediate then
      pcall(function() HungerHUD:refreshImmediate() end)
    end
    if hs and hs.requestHudRefresh then
      pcall(function() hs:requestHudRefresh() end)
    end
  end

  return ok
end


--------------------------------------------------------------------------------
-- ДОП API ДЛЯ КВЕСТОВ (по item.id)
--------------------------------------------------------------------------------

--- Посчитать сколько в инвентаре предметов с заданным item.id
function Inventory.API.getTotalCount(id)
  local inv = Inventory
  if not (inv and inv.items) or not id or id == "" then return 0 end
  local n = 0
  for slot = 1, inv.maxSlots do
    local it = inv.items[slot]
    if it and it.id == id then
      n = n + 1
    end
  end
  return n
end

--- Удалить count предметов с заданным item.id
function Inventory.API.removeItems(id, count)
  local inv = Inventory
  if not (inv and inv.items) or not id or id == "" then return false end
  local toRemove = tonumber(count or 1) or 1
  if toRemove <= 0 then return true end

  for slot = 1, inv.maxSlots do
    local it = inv.items[slot]
    if it and it.id == id then
      inv.items[slot] = nil
      toRemove = toRemove - 1
      if toRemove <= 0 then
        inv.needsSave = true
        if inv.Hud.isVisible then inv.Hud:refresh() end
        return true
      end
    end
  end

  inv.needsSave = true
  if inv.Hud.isVisible then inv.Hud:refresh() end
  return toRemove <= 0
end

--- Добавить несколько предметов с заданным id (для награды квеста)
function Inventory.API.addItemsById(id, count, name, desc, icon, kind, hungerGain, vigorGain, effectId, effectsFile)
  if not id or id == "" then return false end
  local total = tonumber(count or 1) or 1
  if total <= 0 then return false end

  local added = 0
  for i = 1, total do
    local ok = Inventory.API.addItemInstance({
      id          = id,
      name        = name or "",
      desc        = desc or "",
      icon        = icon or "",
      kind        = kind or "food",
      hungerGain  = hungerGain or 0,
      vigorGain   = vigorGain or 0,
      effectId    = effectId or "",
      effectsFile = effectsFile or ""
    })
    if ok then added = added + 1 end
  end
  return added > 0
end

-- ===== Console =====
function Inventory:addConsoleCommands()
  addConsoleCommand("invAddFood",  "Add FOOD: <hunger> <vigor> [icon]", "ccAddFood", self)
  addConsoleCommand("invClear",    "Clear inventory",                    "ccClear",    self)
end
function Inventory:ccAddFood(hunger, vigor, icon)
  local h = tonumber(hunger) or 100
  local v = tonumber(vigor)  or 100
  Inventory.API.addItemInstance({hungerGain=h, vigorGain=v, kind="food", icon=icon or "", name="$l10n_food_test", desc="$l10n_food_desc_test"})
  print(("[Inventory] +1 (H%+d V%+d)"):format(h, v))
end
function Inventory:ccClear()
  for i=1,self.maxSlots do self.items[i]=nil end
  self.needsSave = true
  if self.Hud.isVisible then self.Hud:refresh() end
  print("[Inventory] cleared")
end

function Inventory:injectInputL10n()
  local env = g_i18n and g_i18n.modEnvironments and g_i18n.modEnvironments[self.modName]
  if not (env and env.texts) then return end
  local function inject(key)
    local val = env.texts[key]
    if val and val ~= "" then g_i18n.texts[key] = val end
  end
  inject("input_INVENTORY_SHOW_HUD")
  inject("input_INVENTORY_HIDE_HUD")
  inject("action_INVENTORY_SHOW_HUD")
  inject("action_INVENTORY_HIDE_HUD")
end

-- ===== Input / lifecycle =====
function Inventory:registerActionEvents()
  if not self._aeShow then
    _, self._aeShow = g_inputBinding:registerActionEvent(InputAction.INVENTORY_SHOW_HUD, self, self.onPressShow, false, true, false, true)
  end
  if not self._aeHide then
    _, self._aeHide = g_inputBinding:registerActionEvent(InputAction.INVENTORY_HIDE_HUD, self, self.onPressHide, false, true, false, true)
  end
  self:updateActionEvents()
end

function Inventory:updateActionEvents()
  local showActive = not self.Hud.isVisible
  local hideActive = self.Hud.isVisible
  if self._aeShow then
    g_inputBinding:setActionEventActive(self._aeShow, showActive)
    local tShow = g_i18n:getText("input_INVENTORY_SHOW_HUD") or g_i18n:getText("action_INVENTORY_SHOW_HUD") or "Открыть инвентарь"
    g_inputBinding:setActionEventText(self._aeShow, tShow)
  end
  if self._aeHide then
    g_inputBinding:setActionEventActive(self._aeHide, hideActive)
    local tHide = g_i18n:getText("input_INVENTORY_HIDE_HUD") or g_i18n:getText("action_INVENTORY_HIDE_HUD") or "Закрыть инвентарь"
    g_inputBinding:setActionEventText(self._aeHide, tHide)
  end
end

function Inventory:onPressShow() self:showHud() end
function Inventory:onPressHide() self:hideHud() end
function Inventory:toggleHud() if self.Hud.isVisible then self:hideHud() else self:showHud() end end

function Inventory:showHud()
  if not self.Hud.isVisible then
    self.Hud.isVisible = true
    self.Hud:refresh()
    g_inputBinding:setShowMouseCursor(true)
    self:updateActionEvents()
  end
end

function Inventory:hideHud()
  if self.Hud.isVisible then
    self.Hud.isVisible = false
    g_inputBinding:setShowMouseCursor(false)
    self:updateActionEvents()
  end
end

function Inventory:mouseEvent(posX,posY,isDown,isUp,button)
  if not self.Hud.isVisible then return end
  if button == 1 and isDown and (self._lmbCooldown or 0) <= 0 then
    self._lmbCooldown = 120
    if self.Hud and self.Hud.onMouseClick then self.Hud:onMouseClick(posX, posY) end
  end
  if button == 2 and isDown and (self._rmbCooldown or 0) <= 0 then
    self._rmbCooldown = 120
    if self.Hud and self.Hud.onRightClick then self.Hud:onRightClick(posX, posY) end
  end
  if self.Hud and self.Hud.mouseMove then self.Hud:mouseMove(posX,posY,isDown,isUp,button) end
end

function Inventory:draw()
  if self.Hud.isVisible then self.Hud:render() end
end

function Inventory:loadMap()
  self:setupConst()
  self.I18n.init(self.getNamespaces())
  self.Hud.init(self.getNamespaces())
  self.Dialogs.init(self.getNamespaces())

  self:injectInputL10n()
  self.Storage:load(self)
  if g_currentMission:getIsServer() then self:addConsoleCommands() end
end

function Inventory:update(dt)
  self._lmbCooldown = math.max(0, (self._lmbCooldown or 0) - (dt or 0))
  self._rmbCooldown = math.max(0, (self._rmbCooldown or 0) - (dt or 0))

  if self._forceImmediateSave then self._forceImmediateSave = false; self.Storage:save(self) end

  self._saveDebounce = (self._saveDebounce or 0) - dt
  if self.needsSave and self._saveDebounce <= 0 then
    self._saveDebounce = 5000
    self.needsSave = false
    self.Storage:save(self)
  end
end

local function onPlayerLoaded(player)
  if player.isOwner then
    g_inputBinding:beginActionEventsModification(PlayerInputComponent.INPUT_CONTEXT_NAME)
    Inventory:registerActionEvents()
    g_inputBinding:endActionEventsModification()
  end
end
Player.load = Utils.appendedFunction(Player.load, onPlayerLoaded)


local function VS_mouseEvent(px,py,isDown,isUp,button)
  if g_currentMission and g_currentMission.vehicles then
    for _,veh in ipairs(g_currentMission.vehicles) do
      if veh.spec_vehicleStorage and veh.spec_vehicleStorage.hud and veh:vsIsOpen() then
        local hud = veh.spec_vehicleStorage.hud
        if button == 1 and isDown then hud:onMouseClick(px,py) end
        if button == 2 and isDown then hud:onRightClick(px,py) end
        hud:mouseMove(px,py,isDown,isUp,button)
        return
      end
    end
  end
end
g_inputBinding.mouseEvent = Utils.appendedFunction(g_inputBinding.mouseEvent, VS_mouseEvent)

local function VS_draw()
  if g_currentMission and g_currentMission.vehicles then
    for _,veh in ipairs(g_currentMission.vehicles) do
      local spec = veh.spec_vehicleStorage
      if spec and spec.hud and spec.hud.isVisible then
        spec.hud:render()
      end
    end
  end
end
Mission00.draw = Utils.appendedFunction(Mission00.draw, VS_draw)
