-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

-- База (ядро Логики)

HungerSystem = {}
HungerSystem.modDirectory = g_currentModDirectory or ""

function HungerSystem.getFarmIdFromConnection(connection)
    local fallbackFarmId = (FarmManager and FarmManager.SINGLEPLAYER_FARM_ID) or 1

    if g_currentMission ~= nil and connection ~= nil then
        if g_currentMission.userManager ~= nil
           and g_currentMission.userManager.getUserIdByConnection ~= nil
           and g_farmManager ~= nil
           and g_farmManager.getFarmByUserId ~= nil then

            local userId = g_currentMission.userManager:getUserIdByConnection(connection)
            if userId ~= nil then
                local farm = g_farmManager:getFarmByUserId(userId)
                if farm ~= nil and farm.farmId ~= nil and farm.farmId ~= 0 then
                    return farm.farmId
                end
            end
        end
        local user = connection.user
        if user ~= nil and user.getFarmId ~= nil then
            local farmId = user:getFarmId()
            if farmId ~= nil and farmId ~= 0 then
                return farmId
            end
        end

        if connection.farmId ~= nil and connection.farmId ~= 0 then
            return connection.farmId
        end
    end
    if g_currentMission ~= nil then
        if g_currentMission.getFarmId ~= nil then
            local farmId = g_currentMission:getFarmId()
            if farmId ~= nil and farmId ~= 0 then
                return farmId
            end
        end

        if g_currentMission.player ~= nil and g_currentMission.player.farmId ~= nil then
            return g_currentMission.player.farmId
        end
    end
    return fallbackFarmId
end


-- === КОНФИГУРАЦИЯ HUD === 
HungerSystem.hud = {
    offsetXPixels   = 0,
    offsetYPixels   = 0,
    iconFile        = "gui/hunger_icon.dds",
    iconScale       = 1.0,
    iconExtraSpace  = 0.0,
    attachGap       = 0.007,
    edgeGap         = 0.038,
}

-- === Состояние ===
HungerSystem.initialised         = false
HungerSystem.hunger              = 100
HungerSystem.vigor               = 100
HungerSystem.saveLoaded          = false
HungerSystem.saveDirty           = false
HungerSystem.saveDebounce        = 0
HungerSystem.autoSaveIntervalSec = 250.0

-- эффекты/звуки
HungerSystem.effect = {
    overlay = nil,
    audioEntries = {},          
    sampleCache = {},
    sampleStopTimers = {},
    duration = 3,
    alpha = 0,
    soundRepeatInterval = 30,
    nextSoundTimer = 0,
    soundActive = false,
    lastAudioIndex = nil,

    -- техника-виньетка
    vehicleOverlay = nil,
    vehicleActiveUntil = 0,
    vehicleAlpha = 0.0,

    -- звуки для виньетки техники
    vehicleAudioEntries = {},
    vehicleLastAudioIndex = nil,
    vehicleLastPlayedMs = 0,
    vehicleMinIntervalMs = 1800
}

-- === ПОСТОЯННАЯ ВИНЬЕТКА НИЗКОЙ БОДРОСТИ ===
HungerSystem.vigorCont = {
    overlay = nil,
    alpha   = 0
}

-- === БОДРОСТЬ: ПЕРИОДИЧЕСКИЕ ЗВУКИ ПРИ 0% ===
HungerSystem.vigorPeriodic = {
    audioEntries = {},          
    soundRepeatInterval = 35,   
    nextSoundTimer = 0,
    soundActive = false,
    lastAudioIndex = nil
}

-- === УСТАЛОСТЬ (эпизод от бодрости) ===
HungerSystem.fatigue = {
    overlay = nil,            
    overlayPathFull = nil,
    alpha = 0,

    imageList = {},           

    -- логика триггера
    isActive      = false,
    activeUntilMs = 0,
    checkTimerMs  = 0,
    checkPeriodMs = 7200,     -- частота проверки
    durationMs    = 3000,     -- если у картинки нет duration
    thresholdPct  = 10,       -- запускаем эпизоды при бодрости < 10%
    maxChance     = 0.35,     -- макс. шанс при 0% бодрости
}

-- === СОНЛИВОСТЬ ОТ ГОЛОДА (эпизод пешком) ===
HungerSystem.sleep = {
    imageList        = {}, 
    currentEntry     = nil,
    defaultImageRel  = "gui/vignettehungerEffect.dds",

    isActive         = false,
    activeUntilMs    = 0,
    checkTimerMs     = 0,
    checkPeriodMs    = 5500,
    durationMs       = 2500,
    maxChance        = 0.20,

    overlay          = nil,
    overlayPathFull  = nil,
    alpha            = 0
}

-- =========================
-- SETTINGS + MENU
-- =========================
HungerSystem.DIFFICULTY = {
    -- id=1..5: { titleKey, hunger = {min,max}, vigor = {min,max} } (проценты в ЧАС)
    [1] = { key="l10n_hunger_diff_1", hunger={1,1}, vigor={1,1} },       -- Сосунок
    [2] = { key="l10n_hunger_diff_2", hunger={2,2}, vigor={1,1} },       -- Лёгкий
    [3] = { key="l10n_hunger_diff_3", hunger={2,5}, vigor={1,5} },       -- Нормально 
    [4] = { key="l10n_hunger_diff_4", hunger={3,7}, vigor={2,7} },       -- Сложно
    [5] = { key="l10n_hunger_diff_5", hunger={5,9}, vigor={4,9} },       -- Хардкор
}
HungerSystem.difficulty = 3  -- (по умолчанию)

function HungerSystem:_getSettingsFile()
    return self:_getModSettingsDir() .. "/HungerSystemSettings.xml"
end

function HungerSystem:readSettings()
    local path = self:_getSettingsFile()
    if not fileExists(path) then
        self:writeSettings()
        return
    end
    local xml = loadXMLFile("HungerSystemSettings", path)
    if xml ~= 0 then
        local d = getXMLInt(xml, "HungerSystemSettings.difficulty#value")
        if d then
            d = math.floor(tonumber(d) or 3)
            if d < 1 then d = 1 elseif d > 5 then d = 5 end
            self.difficulty = d
        end

        local sp = getXMLInt(xml, "HungerSystemSettings.speedProfile#value")
        if sp then
            sp = math.floor(tonumber(sp) or 1)
            if sp < 1 then sp = 1 elseif sp > 3 then sp = 3 end
            self.speedProfile = sp
        end
        delete(xml)
    end
end

function HungerSystem:writeSettings()
    local path = self:_getSettingsFile()
    local xml = createXMLFile("HungerSystemSettings", path, "HungerSystemSettings")
    if xml == 0 then
        print("[HungerSystem] Failed to create settings XML: "..tostring(path))
        return
    end
    setXMLInt(xml, "HungerSystemSettings.difficulty#value", tonumber(self.difficulty or 3))
    setXMLInt(xml, "HungerSystemSettings.speedProfile#value", tonumber(self.speedProfile or 1))
    saveXMLFile(xml)
    delete(xml)
end


function HungerSystem:consoleCommandHungerDifficulty(newState)
    local n = tonumber(newState)
    if not n then
        return string.format("Difficulty: %d (1..5)", self.difficulty or 3)
    end
    n = math.max(1, math.min(5, math.floor(n)))
    self.difficulty = n
    self:writeSettings()
    return string.format("Difficulty set to %d", n)
end

-- UI (меню настроек)
function HungerSystem:_installMenuControl()
    if not g_gui or not g_gui.screenControllers or not g_gui.screenControllers[InGameMenu] then return end
    local inGameMenu    = g_gui.screenControllers[InGameMenu]
    local settingsPage  = inGameMenu.pageSettings
    local settingsLayout= settingsPage.generalSettingsLayout
    if not settingsLayout then return end

    -- тексты для сложности (1..5)
    local diffTexts = {}
    for i=1,5 do
        local def = self.DIFFICULTY[i]
        local t = (g_i18n and g_i18n:getText(def.key)) or ("Level "..i)
        diffTexts[#diffTexts+1] = t
    end

    -- тексты для профилей скоростей
    local speedTexts = {}
    for i=1,3 do
		local def = self.SPEED_PROFILES[i]
		local t = (g_i18n and g_i18n:getText(def.key)) or ("Profile "..i)
		speedTexts[#speedTexts+1] = t
	end

    local headerTpl
    for _, elem in ipairs(settingsLayout.elements) do
        if elem.name == "sectionHeader" then headerTpl = elem; break end
    end
    local boxTpl = settingsPage.multiVolumeVoiceBox or settingsPage.checkWoodHarvesterAutoCutBox
    if not boxTpl then return end

    local function wireFocus(element)
        if not element then return end
        element.focusId = FocusManager:serveAutoFocusId()
        for _, ch in pairs(element.elements) do wireFocus(ch) end
    end

    -- Заголовок секции
    local sectionTitle
    if headerTpl then
        sectionTitle = headerTpl:clone(settingsLayout)
        if sectionTitle.setText and g_i18n then
            sectionTitle:setText(g_i18n:getText("menu_HungerSystem_TITLE"))
        end
        wireFocus(sectionTitle)
        table.insert(settingsPage.controlsList, sectionTitle)
    end

    -- 1) СЛОЖНОСТЬ
    do
        local box = boxTpl:clone(settingsLayout)
        box.id = "hungerDifficultyBox"

        local combo = box.elements[1]
        combo.id = "hungerDifficulty"
        combo.target = self
        combo:setCallback("onClickCallback", "onMenuOptionChanged_Difficulty")
        combo:setDisabled(false)
        combo:setTexts(diffTexts)
        combo:setState(self.difficulty or 3)

        local toolTip = combo.elements[1]
        if toolTip and toolTip.setText and g_i18n then
            toolTip:setText(g_i18n:getText("tooltip_hunger_difficulty"))
        end

        local labelElem = box.elements[2]
        if labelElem and labelElem.setText and g_i18n then
            labelElem:setText(g_i18n:getText("setting_hunger_difficulty"))
        end

        wireFocus(box)
        table.insert(settingsPage.controlsList, box)
    end

    -- 2) ПРОФИЛЬ СКОРОСТЕЙ
    do
        local box = boxTpl:clone(settingsLayout)
        box.id = "hungerSpeedProfileBox"

        local combo = box.elements[1]
        combo.id = "hungerSpeedProfile"
        combo.target = self
        combo:setCallback("onClickCallback", "onMenuOptionChanged_SpeedProfile")
        combo:setDisabled(false)
        combo:setTexts(speedTexts)
        combo:setState(self.speedProfile or 1)

        local toolTip = combo.elements[1]
        if toolTip and toolTip.setText and g_i18n then
            toolTip:setText(g_i18n:getText("tooltip_hunger_speedProfile"))
        end

        local labelElem = box.elements[2]
        if labelElem and labelElem.setText and g_i18n then
            labelElem:setText(g_i18n:getText("setting_hunger_speedProfile"))
        end

        wireFocus(box)
        table.insert(settingsPage.controlsList, box)
    end

    settingsLayout:invalidateLayout()

    InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(InGameMenuSettingsFrame.onFrameOpen, function()
        local diff = settingsLayout:getDescendantById("hungerDifficulty")
        if diff and diff.setState then diff:setState(self.difficulty or 3) end

        local sp = settingsLayout:getDescendantById("hungerSpeedProfile")
        if sp and sp.setState then sp:setState(self.speedProfile or 1) end
    end)
end

function HungerSystem:onMenuOptionChanged_Difficulty(state, _elem)
    local d = math.floor(tonumber(state) or 3)
    if d < 1 then d = 1 elseif d > 5 then d = 5 end
    self.difficulty = d
    self:writeSettings()
    self:_applySpeedsNow()
end

function HungerSystem:onMenuOptionChanged_SpeedProfile(state, _elem)
    local n = math.floor(tonumber(state) or 1)
    if n < 1 then n = 1 elseif n > 3 then n = 3 end
    self.speedProfile = n
    self:writeSettings()
    self:_applySpeedsNow()
end


function HungerSystem:_applySpeedsNow()
    local player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)
    if player then
        pcall(function() self:updateHungerSpeeds(player) end)
    end
end


function HungerSystem:onMenuOptionChanged_Difficulty(state, _elem)
    local n = math.max(1, math.min(5, tonumber(state) or 3))
    self.difficulty = n
    self:writeSettings()
end

-- =========================
-- SPEED PROFILES
-- =========================
HungerSystem.SPEED_PROFILES = {
    -- 1) НОРМАЛЬНЫЙ (по умолчанию)
    [1] = {
        key   = "l10n_speed_profile_normal", -- для i18n в меню
        steps = {
            {90, 5.0, 6.0, "hud_vigor_status_fresh"},     -- супер-бодрый
            {70, 4.9, 5.9, "hud_vigor_status_ok"},        -- бодрый/норм
            {50, 4.5, 5.5, "hud_vigor_status_tired"},     -- утомлён
            {30, 4.1, 4.9, "hud_vigor_status_sleepy"},    -- сонный
            {15, 3.2, 3.9, "hud_vigor_status_exhausted"}, -- очень сонный
            { 0, 1.8, 2.8, "hud_vigor_status_exhausted"}, -- вымотан
        }
    },
    -- 2) УВЕЛИЧЕННЫЙ
    [2] = {
        key   = "l10n_speed_profile_boosted",
        steps = {
            {90, 6.5, 8.5, "hud_vigor_status_fresh"},
            {70, 6.2, 7.2, "hud_vigor_status_ok"},
            {50, 5.5, 6.5, "hud_vigor_status_tired"},
            {30, 4.8, 5.8, "hud_vigor_status_sleepy"},
            {15, 3.8, 4.8, "hud_vigor_status_exhausted"},
            { 0, 1.9, 2.8, "hud_vigor_status_exhausted"},
        }
    },
	-- 3) НеЗависящий
	[3] = {
        key   = "l10n_speed_profile_NoBoosted",
        steps = {
            {90, 4.6, 5.6, "hud_vigor_status_fresh"},
            {70, 4.6, 5.6, "hud_vigor_status_ok"},
            {50, 4.6, 5.6, "hud_vigor_status_tired"},
            {30, 4.6, 5.6, "hud_vigor_status_sleepy"},
            {15, 4.6, 5.6, "hud_vigor_status_exhausted"},
            { 0, 4.4, 5.6, "hud_vigor_status_exhausted"},
        }
    },
}
HungerSystem.speedProfile = 1 -- по умолчанию «нормальный»

local function _pickSpeedForVigor(vigorPct, profile)
    local def = HungerSystem.SPEED_PROFILES[profile or 1] or HungerSystem.SPEED_PROFILES[1]
    local steps = def.steps or {}
    local v = math.max(0, math.min(100, tonumber(vigorPct) or 100))
    for _, s in ipairs(steps) do
        local thr, w, r, key = s[1], s[2], s[3], s[4]
        if v >= thr then
            return w, r, key
        end
    end
    -- fallback если что-то пойдёт не так
    return 3.8, 5.2, "hud_vigor_status_ok"
end


local function __isPlayerInVehicle()
    if g_currentMission and g_currentMission.controlledVehicle ~= nil then
        return true
    end

    local player = (g_playerSystem and g_playerSystem:getLocalPlayer()) or (g_currentMission and g_currentMission.player)
    if player == nil then
        return true
    end

    if player.isControlled ~= nil then
        return (player.isControlled == false)
    end
    if player.baseInformation ~= nil and player.baseInformation.isControlled ~= nil then
        return (player.baseInformation.isControlled == false)
    end
    if player.controlledMovement ~= nil then
        return (player.controlledMovement == false)
    end

    if player.rootNode and getParent(player.rootNode) == 0 then
        return true
    end

    return false
end

-- i18n keys
HungerSystem.hungerLabelKey  = "hud_hunger_label"
HungerSystem.vigorLabelKey   = "hud_vigor_label"
HungerSystem.hungerStatusKey = "hud_status_normal"

local function logInfo(s) print("[HungerSystem] "..tostring(s)) end
local function logWarn(s) print("[HungerSystem] Warning: "..tostring(s)) end
local function logErr(s)  print("[HungerSystem] Error: "..tostring(s)) end

-- =========================
-- ПУТИ modSettings
-- =========================
local function ensureFolder(path) pcall(function() createFolder(path) end) end

function HungerSystem:getModSettingsRoot()
    local root = getUserProfileAppPath() .. "modSettings/"
    ensureFolder(root)
    return root
end

function HungerSystem:getPerSaveSubdir()
    local base = self:getModSettingsRoot() .. "HungerSystem/"
    ensureFolder(base)

    local idx = nil
    if g_currentMission and g_currentMission.missionInfo then
        idx = g_currentMission.missionInfo.savegameIndex
    end
    if idx == nil then
        return base
    end

    local sub = string.format("%ssavegame%02d/", base, tonumber(idx) or 0)
    ensureFolder(sub)
    return sub
end

function HungerSystem:getSavegameDir_legacy()
    if g_currentMission and g_currentMission.missionInfo and g_currentMission.missionInfo.savegameDirectory then
        return g_currentMission.missionInfo.savegameDirectory
    end
    return nil
end

function HungerSystem:getSaveFilePath_legacy()
    local dir = self:getSavegameDir_legacy()
    if not dir or dir == "" then return nil end
    return dir .. "/HungerSystem.xml"
end

-- =========================
-- SAVE / LOAD
-- =========================
function HungerSystem:getSaveFilePath()
    return self:getSaveFilePath_modSettings()
end

function HungerSystem:loadFromSavegameIfNeeded(player)
    if self.saveLoaded then return end

    local function clamp01_100(x)
        x = tonumber(x or 100) or 100
        if x < 0 then return 0 end
        if x > 100 then return 100 end
        return x
    end

    local statsPath   = self:_getStatsFilePath()
    local effectsPath = self:_getEffectsFilePath()

    local function readFloat(xml, key)
        local v = nil
        pcall(function() v = getXMLFloat(xml, key) end)
        if v == nil then
            local s = nil
            pcall(function() s = getXMLString(xml, key) end)
            if s ~= nil then v = tonumber(tostring(s):gsub(",", ".")) end
        end
        return v
    end
    local function readInt(xml, key)
        local v = nil
        pcall(function() v = getXMLInt(xml, key) end)
        if v == nil then
            local s = nil
            pcall(function() s = getXMLString(xml, key) end)
            if s ~= nil then v = tonumber(s) end
        end
        return v
    end

    local function applyStatsToSelfAndPlayer(h, v)
        self.hunger = clamp01_100(h)
        self.vigor  = clamp01_100(v)

        local pl = player
        if not pl then
            pl = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
               or (g_currentMission and g_currentMission.player)
        end
        if pl then
            pl.hunger = self.hunger
            pl.vigor  = self.vigor
            if self.updateHungerSpeeds then pcall(function() self:updateHungerSpeeds(pl) end) end
            self._pendingPlayerSync = nil
        else
            self._pendingPlayerSync = { hunger = self.hunger, vigor = self.vigor }
        end
    end

    if not fileExists(statsPath) then
        print(("[HungerSystem] Stats file not found, using defaults. Path: %s"):format(statsPath))
        applyStatsToSelfAndPlayer(100, 100)
        self.lastGlobalHungerUpdate = 0
        self.lastGlobalVigorUpdate  = 0
    else
        local xml = loadXMLFile("HungerSystemStats", statsPath)
        if xml == 0 then
            print(("[HungerSystem] Cannot open stats XML, using defaults. Path: %s"):format(statsPath))
            applyStatsToSelfAndPlayer(100, 100)
            self.lastGlobalHungerUpdate = 0
            self.lastGlobalVigorUpdate  = 0
        else
            local h  = readFloat(xml, "HungerSystem.player#hunger")
            local v  = readFloat(xml, "HungerSystem.player#vigor")
            local lH = readInt  (xml, "HungerSystem.player#lastGlobalHungerUpdate")
            local lV = readInt  (xml, "HungerSystem.player#lastGlobalVigorUpdate")

            if h  == nil then h  = readFloat(xml, "HungerSystem#hunger") end
            if v  == nil then v  = readFloat(xml, "HungerSystem#vigor") end
            if lH == nil then lH = readInt  (xml, "HungerSystem#lastGlobalHungerUpdate") end
            if lV == nil then lV = readInt  (xml, "HungerSystem#lastGlobalVigorUpdate") end

            applyStatsToSelfAndPlayer(h or 100, v or 100)
            self.lastGlobalHungerUpdate = tonumber(lH or 0) or 0
            self.lastGlobalVigorUpdate  = tonumber(lV or 0) or 0

            delete(xml)
        end
    end

    print(("[HungerSystem] Loaded STATS h/v = %.3f / %.3f (lastH=%s lastV=%s) from: %s")
        :format(self.hunger or -1, self.vigor or -1,
                tostring(self.lastGlobalHungerUpdate), tostring(self.lastGlobalVigorUpdate), statsPath))

    if ItemEffects and ItemEffects.loadFromFile then
        ItemEffects:loadFromFile(self, effectsPath)
    end

    self._pendingTimeSync = true

    self.saveLoaded = true
end


function HungerSystem:syncPlayerStatsIfNeeded()
    local pend = self._pendingPlayerSync
    if not pend then return end

    local pl = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
            or (g_currentMission and g_currentMission.player)
    if not pl then return end

    pl.hunger = math.max(0, math.min(100, tonumber(pend.hunger or self.hunger or 100) or 100))
    pl.vigor  = math.max(0, math.min(100, tonumber(pend.vigor  or self.vigor  or 100) or 100))

    self._pendingPlayerSync = nil

    if self.updateHungerSpeeds then
        pcall(function() self:updateHungerSpeeds(pl) end)
    end

    self._reapplySpeedTimer = 0.4
end


function HungerSystem:saveToSavegame(force)
    if not force then
        if not self.saveDirty then return end
        if self.saveDebounce and self.saveDebounce > 0 then return end
    end

    local statsPath   = self:_getStatsFilePath()
    local effectsPath = self:_getEffectsFilePath()
    local dir         = self:_getModSettingsDir()
    pcall(function() createFolder(dir) end)

    local xml = createXMLFile("HungerSystemStats", statsPath, "HungerSystem")
    if xml == 0 then
        print("[HungerSystem] saveToSavegame(): cannot create stats XML at " .. tostring(statsPath))
        return
    end

    setXMLFloat(xml, "HungerSystem.player#hunger", tonumber(self.hunger or 100))
    setXMLFloat(xml, "HungerSystem.player#vigor",  tonumber(self.vigor  or 100))
    setXMLInt  (xml, "HungerSystem.player#lastGlobalHungerUpdate", tonumber(self.lastGlobalHungerUpdate or 0))
    setXMLInt  (xml, "HungerSystem.player#lastGlobalVigorUpdate",  tonumber(self.lastGlobalVigorUpdate  or 0))

    saveXMLFile(xml)
    delete(xml)

    print(("[HungerSystem] Saved STATS h/v = %.3f / %.3f to: %s")
        :format(self.hunger or -1, self.vigor or -1, statsPath))

    if ItemEffects and ItemEffects.saveActiveToFile then
        ItemEffects:saveActiveToFile(effectsPath)
    end

    self.saveDirty   = false
    self.saveDebounce = 1.0
end

local function _ensureFolder(path) pcall(function() createFolder(path) end) end

function HungerSystem:_getModSettingsDir()
    local base = getUserProfileAppPath() .. "modSettings/"
    _ensureFolder(base)
    local dir = base .. "FS25_liveFarmer"
    _ensureFolder(dir)
    return dir
end

function HungerSystem:_getStatsFilePath()
    return self:_getModSettingsDir() .. "/HungerSystemStats.xml"
end

function HungerSystem:_getEffectsFilePath()
    return self:_getModSettingsDir() .. "/HungerSystemEffects.xml"
end

function HungerSystem:getSaveFilePath()               return self:_getStatsFilePath() end
function HungerSystem:getSaveFilePath_modSettings()   return self:_getStatsFilePath() end
function HungerSystem:_getSettingsFilePath()          return self:_getStatsFilePath() end

function HungerSystem:_getScopedSettingsDir()
    local function sanitize(s)
        s = tostring(s or ""):gsub("[^%w_%-%.]+", "_")
        if #s == 0 then s = "default" end
        return s
    end

    local base = getUserProfileAppPath() .. "modSettings/"
    local modBase = "HungerSystem"
    if self.modDirectory and self.modDirectory ~= "" then
        local name = self.modDirectory:match("([^/\\]+)[/\\]?$")
        if name and #name > 0 then modBase = sanitize(name) end
    end
    pcall(function() createFolder(base) end)
    local modDir = base .. modBase
    pcall(function() createFolder(modDir) end)

    local mi = g_currentMission and g_currentMission.missionInfo
    local scope = "default"
    if mi then
        local saveDir = mi.savegameDirectory
        if saveDir and saveDir ~= "" then
            scope = sanitize(saveDir:match("([^/\\]+)$") or saveDir)
        elseif mi.savegameIndex then
            scope = sanitize(("savegame%d"):format(tonumber(mi.savegameIndex) or 0))
        end
    end

    local mapId = (mi and (mi.mapId or mi.mapTitle or mi.mapName)) or "map"
    local scoped = string.format("%s/%s_%s", modDir, scope, sanitize(mapId))
    pcall(function() createFolder(scoped) end)
    return scoped
end

function HungerSystem:_subscribeSavegameEvents()
    if g_messageCenter and MessageType then
        if MessageType.SAVEGAME_SAVE_START then
            g_messageCenter:subscribe(MessageType.SAVEGAME_SAVE_START, self.onSavegameStart, self)
        end
        if MessageType.SAVEGAME_SAVE_FINISHED then
            g_messageCenter:subscribe(MessageType.SAVEGAME_SAVE_FINISHED, self.onSavegameFinished, self)
        end
    end
end

function HungerSystem:onSavegameStart()    self:saveToSavegame(true) end
function HungerSystem:onSavegameFinished() self.saveDebounce = 0 end

-- =========================
-- AUDIO 
-- =========================
function HungerSystem:getOrCreateSampleForFile(filename)
    if not filename or filename == "" then return nil end
    local cache = self.effect.sampleCache[filename]
    if cache and cache.sampleId then return cache.sampleId end

    local uniqueName = "hunger_sample_"..tostring(math.random(1000000))
    local sampleId = createSample(uniqueName)
    if not sampleId then logErr("createSample returned nil for "..tostring(filename)); return nil end

    local ok, err = pcall(function() loadSample(sampleId, filename, false) end)
    if not ok then
        logWarn("loadSample failed for "..tostring(filename)..": "..tostring(err))
        self.effect.sampleCache[filename] = { sampleId = sampleId, loaded = false }
        return sampleId
    end
    self.effect.sampleCache[filename] = { sampleId = sampleId, loaded = true }
    return sampleId
end

function HungerSystem:stopAllCachedSamples()
    for _, data in pairs(self.effect.sampleCache) do
        if data and data.sampleId then
            pcall(function() stopSample(data.sampleId, 0.05, 0) end)
            self.effect.sampleStopTimers[data.sampleId] = nil
        end
    end
end

local function chooseRandomIndexAvoidRepeat(n, lastIdx)
    if n == 0 then return nil end
    if n == 1 then return 1 end
    local idx = math.random(1, n)
    if lastIdx and n > 1 then
        local tries = 0
        while idx == lastIdx and tries < 8 do
            idx = math.random(1, n)
            tries = tries + 1
        end
    end
    return idx
end

function HungerSystem:updateSampleStopTimers(dt)
    local timers = self.effect and self.effect.sampleStopTimers
    if not timers then return end

    local toRemove = {}
    for sid, tleft in pairs(timers) do
        local newt = (tleft or 0) - (dt / 1000)
        if newt <= 0 then
            pcall(function() stopSample(sid, 0.05, 0) end)
            table.insert(toRemove, sid)
        else
            timers[sid] = newt
        end
    end

    for _, sid in ipairs(toRemove) do
        timers[sid] = nil
    end
end

function HungerSystem:__applyAttackMovementLock(lock)
    self._attackMoveLocked = (lock == true)

    local pl = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
            or (g_currentMission and g_currentMission.player)
    if not pl then return end

    if lock then
        if type(pl.setMovementEnabled) == "function" then
            local ok = pcall(pl.setMovementEnabled, pl, false)
            if ok then self.__usedMovementEnabled = true end
        end

        if pl.mover and type(pl.mover.setMaxMoveSpeed) == "function" then
            if self.__savedMaxMoveSpeed == nil then
                self.__savedMaxMoveSpeed =
                    pl.mover.maxMoveSpeed or pl.mover.defaultMaxSpeed or 5.0
            end
            pcall(pl.mover.setMaxMoveSpeed, pl.mover, 0.0)
            self.__usedMoverClamp = true
        end
    else
        if self.__usedMovementEnabled and type(pl.setMovementEnabled) == "function" then
            pcall(pl.setMovementEnabled, pl, true)
        end
        self.__usedMovementEnabled = false

        if self.__usedMoverClamp and pl.mover and type(pl.mover.setMaxMoveSpeed) == "function" then
            local base = self.__savedMaxMoveSpeed or pl.mover.maxMoveSpeed or 5.0
            pcall(pl.mover.setMaxMoveSpeed, pl.mover, base)
        end
        self.__usedMoverClamp = false

        self:__reapplyHungerSpeeds(pl)
    end
end

function HungerSystem:__reapplyHungerSpeeds(player)
    if type(self.updateHungerSpeeds) == "function" then
        pcall(self.updateHungerSpeeds, self, player or ((g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)))
    end
end

-- =========================
-- EFFECTS xml
-- =========================
function HungerSystem:loadEffectsFromXML()
    local xmlPath = Utils.getFilename("hungerEffects.xml", self.modDirectory)
    if not xmlPath or not fileExists(xmlPath) then
        logWarn("hungerEffects.xml not found — effects disabled")
        self.sleep.imageList = { { filenameRel = self.sleep.defaultImageRel } }
        return
    end
    local xmlFile = loadXMLFile("hungerEffects", xmlPath)
    if xmlFile == 0 then
        logErr("Failed to open hungerEffects.xml")
        self.sleep.imageList = { { filenameRel = self.sleep.defaultImageRel } }
        return
    end

    -- === ОСНОВНАЯ ВИНЬЕТКА (ГОЛОД, постоянная) ===
    do
        local imgPath = getXMLString(xmlFile, "hungerEffects.effectimage.image#filename")
        if not imgPath then imgPath = getXMLString(xmlFile, "hungerEffects.effect.image#filename") end
        if imgPath and imgPath ~= "" then
            local fullImg = Utils.getFilename(imgPath, self.modDirectory)
            if fullImg and fileExists(fullImg) then
                self.effect.overlay = Overlay.new(fullImg, 0, 0, 1, 1)
                if self.effect.overlay then self.effect.overlay:setColor(1,1,1,0) end
            else
                logWarn("vignette file missing: "..tostring(fullImg))
            end
        end
    end

    -- === ЗВУКИ 0% СЫТОСТИ ===
    self.effect.audioEntries = {}
    do
        local i = 0
        while true do
            local key = ("hungerEffects.effectAudio.audio(%d)"):format(i)
            if not hasXMLProperty(xmlFile, key) then break end
            local fn = getXMLString(xmlFile, key .. "#filename")
            if fn and fn ~= "" then
                local full = Utils.getFilename(fn, self.modDirectory)
                table.insert(self.effect.audioEntries, {
                    filename = full, raw = fn,
                    volume = getXMLFloat(xmlFile, key .. "#volume") or 1.0,
                    pitch  = getXMLFloat(xmlFile, key .. "#pitch") or 1.0,
                    duration = getXMLFloat(xmlFile, key .. "#duration") or self.effect.duration
                })
            end
            i = i + 1
        end
        if #self.effect.audioEntries == 0 then
            local legacy = getXMLString(xmlFile, "hungerEffects.effect.audio#filename")
            if legacy and legacy ~= "" then
                local full = Utils.getFilename(legacy, self.modDirectory)
                table.insert(self.effect.audioEntries, { filename = full, raw = legacy, volume = 1.0, pitch = 1.0, duration = self.effect.duration })
            end
        end
    end

    -- === ВИНЬЕТКА ДЛЯ ТЕХНИКИ + ЗВУКИ ===
    do
        local vehImg = getXMLString(xmlFile, "hungerEffects.effectVehicle.image#filename")
        if vehImg and vehImg ~= "" then
            local fullVeh = Utils.getFilename(vehImg, self.modDirectory)
            if fullVeh and fileExists(fullVeh) then
                self.effect.vehicleOverlay = Overlay.new(fullVeh, 0, 0, 1, 1)
                if self.effect.vehicleOverlay then self.effect.vehicleOverlay:setColor(1,1,1,0) end
            else
                logWarn("vehicle vignette file missing: "..tostring(fullVeh))
            end
        end
    end
    self.effect.vehicleAudioEntries = {}
    do
        local i = 0
        while true do
            local key = ("hungerEffects.effectVehicleAudio.audio(%d)"):format(i)
            if not hasXMLProperty(xmlFile, key) then break end
            local fn = getXMLString(xmlFile, key .. "#filename")
            if fn and fn ~= "" then
                local full = Utils.getFilename(fn, self.modDirectory)
                table.insert(self.effect.vehicleAudioEntries, {
                    filename = full, raw = fn,
                    volume = getXMLFloat(xmlFile, key .. "#volume") or 1.0,
                    pitch  = getXMLFloat(xmlFile, key .. "#pitch") or 1.0,
                    duration = getXMLFloat(xmlFile, key .. "#duration") or 5
                })
            end
            i = i + 1
        end
    end

    -- === СОНЛИВОСТЬ ОТ ГОЛОДА (эпизод пешком): список картинок + опц. звук ===
    self.sleep.imageList = {}
    do
        local i = 0
        while true do
            local key = ("hungerEffects.effectimageSleep.image(%d)"):format(i)
            if not hasXMLProperty(xmlFile, key) then break end
            local fnRel = getXMLString(xmlFile, key .. "#filename")
            if fnRel and fnRel ~= "" then
                local entry = {
                    filenameRel = fnRel,
                    audioRel    = getXMLString(xmlFile, key .. "#audio") or nil,
                    volume      = getXMLFloat (xmlFile, key .. "#volume")   or 1.0,
                    pitch       = getXMLFloat (xmlFile, key .. "#pitch")    or 1.0,
                    duration    = getXMLFloat (xmlFile, key .. "#duration") or 3.0
                }
                table.insert(self.sleep.imageList, entry)
            end
            i = i + 1
        end
        if #self.sleep.imageList == 0 then
            local single = getXMLString(xmlFile, "hungerEffects.effectimageSleep.image#filename")
            if single and single ~= "" then
                table.insert(self.sleep.imageList, { filenameRel = single })
            end
        end
        if #self.sleep.imageList == 0 then
            self.sleep.imageList = { { filenameRel = self.sleep.defaultImageRel } }
        end
    end

    -- === УСТАЛОСТЬ (бодрость, ЭПИЗОД): СПИСОК картинок + ПРИВЯЗАННЫЕ звуки ===
    self.fatigue.imageList = {}
    do
        local i = 0
        while true do
            local key = ("hungerEffects.vigorEffects.effectimage.image(%d)"):format(i)
            if not hasXMLProperty(xmlFile, key) then break end
            local fnRel = getXMLString(xmlFile, key .. "#filename")
            if fnRel and fnRel ~= "" then
                local entry = {
                    filenameRel = fnRel,
                    audioRel    = getXMLString(xmlFile, key .. "#audio") or nil,
                    volume      = getXMLFloat (xmlFile, key .. "#volume")   or 1.0,
                    pitch       = getXMLFloat (xmlFile, key .. "#pitch")    or 1.0,
                    duration    = getXMLFloat (xmlFile, key .. "#duration") or 3.0
                }
                table.insert(self.fatigue.imageList, entry)
            end
            i = i + 1
        end
        if #self.fatigue.imageList == 0 then
            local single = getXMLString(xmlFile, "hungerEffects.vigorEffects.effectimage.image#filename")
            if single and single ~= "" then
                table.insert(self.fatigue.imageList, { filenameRel = single })
            end
        end
        if #self.fatigue.imageList == 0 then
            table.insert(self.fatigue.imageList, { filenameRel = "gui/vignetteSleep.dds" })
        end
    end

    -- === БОДРОСТЬ (ПЕРИОД НА 0%) ===
    self.vigorPeriodic.audioEntries = {}
    do
        local i = 0
        while true do
            local key = ("hungerEffects.vigorEffects.effectAudio.audio(%d)"):format(i)
            if not hasXMLProperty(xmlFile, key) then break end
            local fn = getXMLString(xmlFile, key .. "#filename")
            if fn and fn ~= "" then
                local full = Utils.getFilename(fn, self.modDirectory)
                table.insert(self.vigorPeriodic.audioEntries, {
                    filename = full, raw = fn,
                    volume = getXMLFloat(xmlFile, key .. "#volume") or 1.0,
                    pitch  = getXMLFloat(xmlFile, key .. "#pitch") or 1.0,
                    duration = getXMLFloat(xmlFile, key .. "#duration") or 3.0
                })
            end
            i = i + 1
        end
    end

    -- === БОДРОСТЬ (ПОСТОЯННАЯ ВИНЬЕТКА) ===
    do
        local vigorImg = getXMLString(xmlFile, "hungerEffects.vigorEffects.effectimageVigor.image#filename")
        if (not vigorImg or vigorImg == "") then
            vigorImg = getXMLString(xmlFile, "hungerEffects.effectimage.image#filename")
            if not vigorImg or vigorImg == "" then
                vigorImg = getXMLString(xmlFile, "hungerEffects.effect.image#filename")
            end
        end

        if vigorImg and vigorImg ~= "" then
            local full = Utils.getFilename(vigorImg, self.modDirectory)
            if full and fileExists(full) then
                self.vigorCont.overlay = Overlay.new(full, 0, 0, 1, 1)
                self.vigorCont.overlay:setColor(1,1,1,0)
            else
                logWarn("vigor (continuous) vignette missing: "..tostring(full))
            end
        end
    end

    delete(xmlFile)
end

function HungerSystem:__playEntryAudio(entry)
    if not entry or not entry.audioRel or entry.audioRel == "" then return end
    local full = Utils.getFilename(entry.audioRel, self.modDirectory)
    if not (full and fileExists(full)) then
        logWarn("Entry audio missing: "..tostring(full))
        return
    end

    local sampleId = self:getOrCreateSampleForFile(full)
    if not sampleId then return end

    local volume   = entry.volume or 1.0
    local pitch    = entry.pitch  or 1.0
    local offsetMs = 0

    local ok, err = pcall(function() playSample(sampleId, 0, 1, volume, pitch, offsetMs) end)
    if not ok then
        local ok2 = pcall(function() playSample(sampleId, 1, volume, offsetMs) end)
        if not ok2 then
            logErr("entry playSample failed: "..tostring(err)); return
        end
    end

    local stopAfter = tonumber(entry.duration) or 3.0
    self.effect.sampleStopTimers[sampleId] = stopAfter
end

local function _playRandomFromListGeneric(self, list, lastIndexStore)
    local n = #list; if n == 0 then return end
    local last = self[lastIndexStore]
    local idx = chooseRandomIndexAvoidRepeat(n, last)
    if not idx then return end
    local e = list[idx]
    if not (e and e.filename and fileExists(e.filename)) then return end

    local sid = self:getOrCreateSampleForFile(e.filename); if not sid then return end
    self[lastIndexStore] = idx

    local vol = e.volume or 1.0
    local pit = e.pitch  or 1.0
    local ok = pcall(function() playSample(sid, 0, 1, vol, pit, 0) end)
    if not ok then pcall(function() playSample(sid, 1, vol, 0) end) end

    local stopAfter = tonumber(e.duration) or 3.0
    self.effect.sampleStopTimers[sid] = stopAfter
end

-- звуки при 0% сытости
function HungerSystem:playRandomSoundNow()
    _playRandomFromListGeneric(self, self.effect.audioEntries or {}, "effect.lastAudioIndex")
    local base = self.effect.soundRepeatInterval or 30
    self.effect.nextSoundTimer = math.random(math.max(10, base - 10), base + 15)
end

-- звуки при 0% БОДРОСТИ
function HungerSystem:playRandomVigorSoundNow()
    _playRandomFromListGeneric(self, self.vigorPeriodic.audioEntries or {}, "vigorPeriodic.lastAudioIndex")
    local base = self.vigorPeriodic.soundRepeatInterval or 35
    self.vigorPeriodic.nextSoundTimer = math.random(math.max(12, base - 12), base + 18)
end

function HungerSystem:playRandomVehicleSoundNow()
    local list = self.effect.vehicleAudioEntries or {}
    if #list == 0 then return end
    local idx = chooseRandomIndexAvoidRepeat(#list, self.effect.vehicleLastAudioIndex)
    if not idx then return end
    self.effect.vehicleLastAudioIndex = idx
    local e = list[idx]
    if not (e and e.filename and fileExists(e.filename)) then return end

    local sid = self:getOrCreateSampleForFile(e.filename); if not sid then return end
    local vol = e.volume or 1.0
    local pit = e.pitch  or 1.0
    local now = g_time or 0
    if now - (self.effect.vehicleLastPlayedMs or 0) < (self.effect.vehicleMinIntervalMs or 1600) then return end

    local ok = pcall(function() playSample(sid, 0, 1, vol, pit, 0) end)
    if not ok then pcall(function() playSample(sid, 1, vol, 0) end) end

    self.effect.sampleStopTimers[sid] = tonumber(e.duration) or 3.0
    self.effect.vehicleLastPlayedMs = now
end

-- =========================
-- API (для спецализаций)
-- =========================
function HungerSystem:triggerVehicleEffect(durationMs, severity)
    if not self.effect then return end

    if self.effect.vehicleOverlay then
        local hold = math.max(0, tonumber(durationMs) or 0)
        local sev  = math.max(0, math.min(1, tonumber(severity) or 0))
        self.effect.vehicleActiveUntil = (g_time or 0) + hold
        local targetAlpha = 0.85 + 0.99 * sev
        self.effect.vehicleAlpha = math.max(self.effect.vehicleAlpha or 0, targetAlpha)
    end

    if self.effect.vehicleAudioEntries and #self.effect.vehicleAudioEntries > 0 then
        self:playRandomVehicleSoundNow()
    end
end

-- =========================
-- SPEEDS / STATUS
-- =========================
function HungerSystem:updateHungerSpeeds(player)
    if not player then
        player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
              or (g_currentMission and g_currentMission.player)
        if not player then return end
    end

    local v = tonumber(player.vigor  or self.vigor  or 100) or 100 -- бодрость
    local h = tonumber(player.hunger or self.hunger or 100) or 100 -- сытость

    -- скоростя по активному профилю
    local walkSpeed, runSpeed, statusKey = _pickSpeedForVigor(v, self.speedProfile or 1)
    self.vigorStatusKey = statusKey or "hud_vigor_status_ok"

    -- штраф от НИЗКОЙ сытости
    local hungerPenaltyActive = (h < 15)
    if hungerPenaltyActive then
        walkSpeed = walkSpeed * 0.45
        runSpeed  = runSpeed  * 0.45
    end
    self.hungerPenaltyActive = hungerPenaltyActive

    if PlayerStateWalk ~= nil then
        PlayerStateWalk.MAXIMUM_WALK_SPEED = walkSpeed
        PlayerStateWalk.MAXIMUM_RUN_SPEED  = runSpeed
    end

    -- статус сытости для совместимости (как было)
    if h >= 90 then
        self.hungerStatusKey = "hud_status_sated"
    elseif h >= 70 then
        self.hungerStatusKey = "hud_status_normal"
    elseif h >= 50 then
        self.hungerStatusKey = "hud_status_slightly_hungry"
    elseif h >= 25 then
        self.hungerStatusKey = "hud_status_hungry"
    elseif h >= 15 then
        self.hungerStatusKey = "hud_status_very_hungry"
    else
        self.hungerStatusKey = "hud_status_starving"
    end
end


-- === ITEM EFFECTS API ===
-- === ITEM EFFECTS API ===
function HungerSystem:applyItemEffect(effectId, effectsFilePathAbs)
    -- во время сна ничего не применяем и не показываем ошибок
    if self.sleepGame and self.sleepGame.isSleeping then
        return false, "sleeping"
    end

    local ok, reason, ctx = ItemEffects:canConsume(effectId, effectsFilePathAbs)
    if not ok then
        -- если отказ не из-за сна — показываем причину, как раньше
        if reason ~= "sleeping" then
            local hud = g_currentMission and g_currentMission.hud
            if hud and hud.topNotification and hud.topNotification.setNotification then
                local title = (g_i18n and g_i18n:getText("ui_effect_denied")) or "Недоступно"
                local msg
                if reason == "busy" then
                    msg = string.format((g_i18n and g_i18n:getText("ui_effect_busy")) or "Уже действует: %s",
                        ctx and (ctx.activeTitle or ctx.activeId) or "?")
                elseif reason == "atCap" then
                    msg = string.format((g_i18n and g_i18n:getText("ui_effect_cap")) or "Достигнут лимит x%d",
                        ctx and ctx.maxStacks or 1)
                elseif reason == "notFound" then
                    msg = (g_i18n and g_i18n:getText("ui_effect_notfound")) or "Эффект не найден"
                else
                    msg = (g_i18n and g_i18n:getText("ui_effect_fail")) or "Сейчас нельзя"
                end
                local iconPath = Utils.getFilename(self.hud and self.hud.iconFile or "gui/hunger_icon.dds", self.modDirectory)
                if iconPath == "" then iconPath = nil end
                hud.topNotification:setNotification(title, msg, "", iconPath, nil)
            end
        end
        return false, reason
    end

    local ok2, reason2 = ItemEffects:startEffect(self, effectId, effectsFilePathAbs)
    if ok2 then
        local def  = ItemEffects:getEffectDef(effectId, effectsFilePathAbs)
        local name = (def and def.title) or tostring(effectId)
        local act  = ItemEffects:getActiveEffectInfo()

        local hud = g_currentMission and g_currentMission.hud
        if hud and hud.topNotification and hud.topNotification.setNotification then
            local title   = (g_i18n and g_i18n:getText("ui_item_used")) or "Использовано"
            local details = name
            if act and act.maxStacks and act.maxStacks > 1 then
                details = string.format("%s  (x%d/%d)", name, act.stacks or 1, act.maxStacks)
            end
            local iconPath = Utils.getFilename(self.hud and self.hud.iconFile or "gui/hunger_icon.dds", self.modDirectory)
            if iconPath == "" then iconPath = nil end
            hud.topNotification:setNotification(title, details, "", iconPath, nil)
        end
        return true
    end

    return false, reason2 or "fail"
end


function HungerSystem:canUseItemKind(kind)
    return true, nil
end

function HungerSystem:canStartEffect(effectId)
    if ItemEffects and ItemEffects.isEffectActive and ItemEffects:isEffectActive(tostring(effectId or "")) then
        return false, "alreadyActive"
    end
    return true, nil
end

function HungerSystem:notifyEffectAlreadyActive(itemTitle, iconPath)
    local hud = g_currentMission and g_currentMission.hud
    if hud and hud.topNotification and hud.topNotification.setNotification then
        local ttl = (g_i18n and g_i18n:getText("ui_effect_already_active_title")) or "Уже активен"
        local body= (g_i18n and g_i18n:getText("ui_effect_already_active_body"))  or "Этот эффект уже действует .. "
        hud.topNotification:setNotification(ttl, tostring(itemTitle or ""), body, iconPath or "", nil)
    end
end

-- =========================
-- TopNotification / Status text helpers
-- =========================
local function getVigorStatusText(self, v)
    v = math.max(0, math.min(100, tonumber(v) or 100))
    local key
    if v >= 80 then
        key = "hud_vigor_status_fresh"
    elseif v >= 60 then
        key = "hud_vigor_status_ok"
    elseif v >= 40 then
        key = "hud_vigor_status_tired"
    elseif v >= 20 then
        key = "hud_vigor_status_sleepy"
    else
        key = "hud_vigor_status_exhausted"
    end
    if g_i18n and g_i18n.getText then
        local loc = g_i18n:getText(key)
        if loc and loc ~= key then return loc end
    end

    if v >= 80 then return "Бодрый"
    elseif v >= 60 then return "Нормально"
    elseif v >= 40 then return "Утомлён"
    elseif v >= 20 then return "Сонный"
    else return "Вымотан" end
end

-- ЕДИНОЕ почасовое уведомление (сколько снято сытости/бодрости + их статусы)
function HungerSystem:notifyHourlyChange(hungerLoss, vigorLoss)
    local hud = g_currentMission and g_currentMission.hud
    if not (hud and hud.topNotification and hud.topNotification.setNotification) then return end

    local hungerLabel = (g_i18n and g_i18n.getText) and g_i18n:getText(self.hungerLabelKey) or "Сытость"
    local vigorLabel  = (g_i18n and g_i18n.getText) and g_i18n:getText(self.vigorLabelKey)  or "Бодрость"
    local title = hungerLabel .. " / " .. vigorLabel

    local parts = {}
    if (hungerLoss or 0) > 0 then table.insert(parts, string.format("-%d%% сытости", hungerLoss)) end
    if (vigorLoss  or 0) > 0 then table.insert(parts,  string.format("-%d%% бодрости", vigorLoss)) end
    local line1 = (#parts > 0) and table.concat(parts, ", ") or ""

    local hungerStatusText = (g_i18n and g_i18n.getText) and g_i18n:getText(self.hungerStatusKey or "hud_status_normal") or "Статус сытости"
    local vigorStatusText  = getVigorStatusText(self, self.vigor or 100)
    local line2 = hungerStatusText .. " | " .. vigorStatusText

    local iconPath = Utils.getFilename(self.hud.iconFile or "gui/hunger_icon.dds", self.modDirectory)
    hud.topNotification:setNotification(title, line1, line2, iconPath, nil)
end

function HungerSystem:notifyHungerStatusChange(lossAmount)
    local hud = g_currentMission and g_currentMission.hud
    if not (hud and hud.topNotification and hud.topNotification.setNotification) then
        return
    end

    local title = (g_i18n and g_i18n.getText) and g_i18n:getText(self.hungerLabelKey) or "HUNGER"
    local statusText = (g_i18n and g_i18n.getText) and g_i18n:getText(self.hungerStatusKey) or "STATUS"
    local info = (lossAmount and lossAmount > 0) and ("-"..tostring(lossAmount).."%") or ""
    local iconPath = Utils.getFilename(self.hud.iconFile or "gui/hunger_icon.dds", self.modDirectory)

    hud.topNotification:setNotification(title, statusText, info, iconPath, nil)
end

-- =========================
-- СОНЛИВОСТЬ (логика от ГОЛОДА)
-- =========================
local function sleepChanceForHunger(self, h)
    h = math.max(0, math.min(100, h or 100))
    if h >= 10 then
        return 0
    end
    local maxP = (self and self.sleep and self.sleep.maxChance) or 0.15
    local t = (10 - h) / 10
    return t * maxP
end

function HungerSystem:__ensureSleepOverlayForCurrent()
    local entry = self.sleep.currentEntry
    local rel   = (entry and entry.filenameRel) or self.sleep.defaultImageRel
    local full  = Utils.getFilename(rel, self.modDirectory)

    if full ~= self.sleep.overlayPathFull then
        if self.sleep.overlay then
            pcall(function() self.sleep.overlay:delete() end)
            self.sleep.overlay = nil
        end
        self.sleep.overlayPathFull = full
    end

    if not self.sleep.overlay and full and fileExists(full) then
        self.sleep.overlay = Overlay.new(full, 0, 0, 1, 1)
        self.sleep.overlay:setColor(1, 1, 1, 0)
    end
end

function HungerSystem:__ensureFatigueOverlayForCurrent()
    local entry = self.fatigue.currentEntry
    local rel   = (entry and entry.filenameRel) or "gui/vignetteSleep.dds"
    local full  = Utils.getFilename(rel, self.modDirectory)

    if full ~= self.fatigue.overlayPathFull then
        if self.fatigue.overlay then
            pcall(function() self.fatigue.overlay:delete() end)
            self.fatigue.overlay = nil
        end
        self.fatigue.overlayPathFull = full
    end

    if not self.fatigue.overlay and full and fileExists(full) then
        self.fatigue.overlay = Overlay.new(full, 0, 0, 1, 1)
        self.fatigue.overlay:setColor(1, 1, 1, 0)
    end
end

function HungerSystem:__updateSleepCheck(dt, nowMs)
    if self.sleep.isActive or (__isPlayerInVehicle and __isPlayerInVehicle()) then return end
    self.sleep.checkTimerMs = (self.sleep.checkTimerMs or 0) + (dt or 0)
    if self.sleep.checkTimerMs < (self.sleep.checkPeriodMs or 8000) then return end
    self.sleep.checkTimerMs = 0

    local h = self.hunger or 100
    local maxP = (self and self.sleep and self.sleep.maxChance) or 0.15
    local p = (h >= 10) and 0 or ((10 - h) / 10) * maxP
    if p > 0 and math.random() < p then
        self:__startSleep(nowMs or (g_time or 0))
    end
end

function HungerSystem:__pickRandomSleepEntry()
    local list = self.sleep.imageList or {}
    if #list == 0 then return { filenameRel = self.sleep.defaultImageRel } end
    return list[math.random(1, #list)]
end

-- ================================
--  СТАРТ ЭПИЗОДА "СОН" (от голода)
-- ================================
function HungerSystem:__startSleep(nowMs)
    if __isPlayerInVehicle and __isPlayerInVehicle() then return end
    self.sleep = self.sleep or {}
    if self.sleep.isActive then return end

    local now   = nowMs or (g_time or 0)
    local entry = (self.__pickRandomSleepEntry and self:__pickRandomSleepEntry()) or
                  { filenameRel = (self.sleep.defaultImageRel or "gui/vignetteSleep.dds"),
                    duration = (self.sleep.durationMs or 3000)/1000 }

    local durSec = tonumber(entry.duration) or (self.sleep.durationMs or 3000)/1000

    self.sleep.isActive      = true
    self.sleep.alpha         = 0
    self.sleep.activeUntilMs = now + math.floor(durSec * 1000)
    self.sleep.currentEntry  = entry

    if self.__ensureOverlaySleep then
        self:__ensureOverlaySleep(entry.filenameRel)
    end

    self:__applyAttackMovementLock(true)

    if self.__playEntryAudio then
        self:__playEntryAudio(entry)
    end
end

function HungerSystem:__endSleep()
    local s = self.sleep
    if not s then return end

    s.isActive = false
    s.activeUntilMs = nil
    s.alpha = 0

    if s.currentSampleId then
        pcall(function() stopSample(s.currentSampleId, 0, 0) end)
        pcall(function() deleteSample(s.currentSampleId) end)
        s.currentSampleId = nil
    end
    if s.sampleId then
        pcall(function() stopSample(s.sampleId, 0, 0) end)
        pcall(function() deleteSample(s.sampleId) end)
        s.sampleId = nil
    end

    if s.overlay then pcall(function() s.overlay:delete() end) ; s.overlay = nil end
    if s.overlay2 then pcall(function() s.overlay2:delete() end); s.overlay2 = nil end

    s.currentEntry = nil

    self:__applyAttackMovementLock(false)
end

function HungerSystem:__endFatigue()
    local f = self.fatigue
    if not f then return end

    f.isActive = false
    f.activeUntilMs = nil
    f.alpha = 0

    if f.currentSampleId then
        pcall(function() stopSample(f.currentSampleId, 0, 0) end)
        pcall(function() deleteSample(f.currentSampleId) end)
        f.currentSampleId = nil
    end
    if f.sampleId then
        pcall(function() stopSample(f.sampleId, 0, 0) end)
        pcall(function() deleteSample(f.sampleId) end)
        f.sampleId = nil
    end

    if f.overlay then pcall(function() f.overlay:delete() end) ; f.overlay = nil end
    if f.overlay2 then pcall(function() f.overlay2:delete() end); f.overlay2 = nil end

    f.currentEntry = nil

    self:__applyAttackMovementLock(false)
end

function HungerSystem:isSleepActive() return self.sleep.isActive end
function HungerSystem:getSleepOverlayPath()
    local rel = (self.sleep.currentEntry and self.sleep.currentEntry.filenameRel) or self.sleep.defaultImageRel
    return Utils.getFilename(rel, self.modDirectory)
end

-- =========================
function HungerSystem:consoleCommandSleepForce()
    self:__startSleep(g_time or 0)
    local rel = (self.sleep.currentEntry and self.sleep.currentEntry.filenameRel) or self.sleep.defaultImageRel
    print(string.format("[HungerSystem] Sleep forced for %d ms (imageRel=%s)",
        self.sleep.durationMs or 0, tostring(rel)))
end

function HungerSystem:consoleCommandSleepTune(periodMs, durationMs, maxChance)
    local p = tonumber(periodMs)
    local d = tonumber(durationMs)
    local m = tonumber(maxChance)
    if p then self.sleep.checkPeriodMs = math.max(200, p) end
    if d then self.sleep.durationMs    = math.max(200, d) end
    if m then self.sleep.maxChance     = math.max(0, math.min(1, m)) end
    print(string.format("[HungerSystem] Sleep tuned: period=%dms duration=%dms maxChance=%.2f",
        self.sleep.checkPeriodMs, self.sleep.durationMs, self.sleep.maxChance))
end

-- === Вспомогательные функции усталости (бодрость) ===
local function fatigueChance(self, vigorPct)
    local th = self.fatigue.thresholdPct or 20
    if vigorPct >= th then return 0 end
    local t = (th - vigorPct) / math.max(1, th) -- 0..1
    return t * (self.fatigue.maxChance or 0.25)
end

function HungerSystem:__pickRandomFatigueEntry()
    local list = self.fatigue.imageList or {}
    if #list == 0 then return { filenameRel = "gui/vignetteSleep.dds" } end
    return list[math.random(1, #list)]
end

-- ================================
--  СТАРТ ЭПИЗОДА "УСТАЛОСТЬ" (от бодрости)
-- ================================
function HungerSystem:__startFatigue(nowMs)
    if __isPlayerInVehicle and __isPlayerInVehicle() then return end
    self.fatigue = self.fatigue or {}
    if self.fatigue.isActive then return end

    local now   = nowMs or (g_time or 0)
    local entry = (self.__pickRandomFatigueEntry and self:__pickRandomFatigueEntry()) or
                  { filenameRel = "gui/vignetteSleep.dds", duration = (self.fatigue.durationMs or 3000)/1000 }

    local durSec = tonumber(entry.duration) or (self.fatigue.durationMs or 3000)/1000

    self.fatigue.isActive      = true
    self.fatigue.alpha         = 0
    self.fatigue.activeUntilMs = now + math.floor(durSec * 1000)
    self.fatigue.currentEntry  = entry

    if self.__ensureFatigueOverlayForCurrent then
        self:__ensureFatigueOverlayForCurrent()
    else
        self:__ensureOverlayFatigue(entry.filenameRel)
    end

    self:__applyAttackMovementLock(true)

    if self.__playEntryAudio then
        self:__playEntryAudio(entry)
    end
end

function HungerSystem:__ensureOverlayFatigue(filenameRel)
    self.fatigue = self.fatigue or {}
    local alpha = self.fatigue.alpha or 0

    local filenameAbs = Utils.getFilename(filenameRel or "gui/vignetteSleep.dds", self.modDirectory)
    if not (filenameAbs and fileExists(filenameAbs)) then
        print(string.format("[HungerSystem] __ensureOverlayFatigue: file not found '%s'", tostring(filenameAbs)))
        return
    end

    if self.fatigue.overlay and self.fatigue.overlay.id and self.fatigue.overlay.filename == filenameAbs then
        setOverlayColor(self.fatigue.overlay.id, 1, 1, 1, alpha)
        return
    end

    if self.fatigue.overlay and self.fatigue.overlay.id then
        delete(self.fatigue.overlay.id)
        self.fatigue.overlay = nil
    end

    local ov = createImageOverlay(filenameAbs)
    setOverlayColor(ov, 1, 1, 1, alpha)
    self.fatigue.overlay = { id = ov, filename = filenameAbs }
end

function HungerSystem:__ensureOverlaySleep(filenameRel)
    self.sleep = self.sleep or {}
    local alpha = self.sleep.alpha or 0

    local filenameAbs = Utils.getFilename(filenameRel or "gui/vignetteSleep.dds", self.modDirectory)
    if not (filenameAbs and fileExists(filenameAbs)) then
        print(string.format("[HungerSystem] __ensureOverlaySleep: file not found '%s'", tostring(filenameAbs)))
        return
    end

    if self.sleep.overlay and self.sleep.overlay.id and self.sleep.overlay.filename == filenameAbs then
        setOverlayColor(self.sleep.overlay.id, 1, 1, 1, alpha)
        return
    end

    if self.sleep.overlay and self.sleep.overlay.id then
        delete(self.sleep.overlay.id)
        self.sleep.overlay = nil
    end

    local ov = createImageOverlay(filenameAbs)
    setOverlayColor(ov, 1, 1, 1, alpha)
    self.sleep.overlay = { id = ov, filename = filenameAbs }
end

-- =========================
-- Сон/пробуждение: считаем часы, применяем изменения и показываем сводку
-- =========================
function HungerSystem:onSleepingState(isSleeping)
    local env = g_currentMission and g_currentMission.environment
    if not env then return end

    self.sleepGame = self.sleepGame or { isSleeping = false, startDayTime = nil }

    if isSleeping then
        -- вошли в режим сна
        self.sleepGame.isSleeping   = true
        self.sleepGame.startDayTime = env.dayTime or 0

        -- на время сна гасим активные эпизоды
        if self.sleep and self.sleep.isActive     then self:__endSleep()   end
        if self.fatigue and self.fatigue.isActive then self:__endFatigue() end
        return
    end

    -- просыпаемся
    local endDayTime   = env.dayTime or 0
    local startDayTime = self.sleepGame.startDayTime or endDayTime
    local dayMs        = 24 * 60 * 60 * 1000

    local deltaMs    = (endDayTime - startDayTime) % dayMs
    local sleptHours = math.floor(deltaMs / (60 * 60 * 1000) + 1e-6)

    local player = (g_playerSystem and g_playerSystem:getLocalPlayer()) or (g_currentMission and g_currentMission.player)
    local vigorGain = 0

    if player and sleptHours > 0 then
        -- во сне сытость убывает по 1%/час
        local beforeH = player.hunger or 100
        player.hunger = math.max(0, (beforeH or 100) - sleptHours)
        self.hunger   = player.hunger

        -- бодрость растёт 10..15%/час (суммируем по каждому часу)
        for _=1, sleptHours do
            vigorGain = vigorGain + math.random(10, 15)
        end
        local beforeV = player.vigor or self.vigor or 100
        local afterV  = math.max(0, math.min(100, (beforeV or 100) + vigorGain))
        player.vigor  = afterV
        self.vigor    = afterV

        -- сдвигаем «последние часы апдейта» на текущий час,
        -- чтобы не было доп. списаний сразу после сна
        local curDay  = env.currentDay or 1
        local curHour = math.floor(env.currentHour or 0)
        local globalHourNow = (curDay * 24) + curHour
        player.lastGlobalHungerUpdate = globalHourNow
        player.lastGlobalVigorUpdate  = globalHourNow
        self.lastGlobalHungerUpdate   = globalHourNow
        self.lastGlobalVigorUpdate    = globalHourNow

        self.saveDirty = true

        -- ПЕРЕПРИМЕНИТЬ СКОРОСТИ ПО НОВОЙ БОДРОСТИ/СЫТОСТИ
        if self.updateHungerSpeeds then
            pcall(function() self:updateHungerSpeeds(player) end)
        end
        -- и отложенный реапплай, чтобы попасть в момент, когда стейты игрока уже полностью подняты
        self._reapplySpeedTimer = 0.3

        -- Показать сводку: сколько спали и сколько бодрости получили
        local hud = g_currentMission and g_currentMission.hud
        if hud and hud.topNotification and hud.topNotification.setNotification then
            local titleTxt = (g_i18n and g_i18n:getText("ui_sleep_title")) or "СОН"
            local pattern  = (g_i18n and g_i18n:getText("ui_sleep_summary")) or "Проснулись — спали %d ч., бодрость +%d%%"
            local infoTxt  = string.format(pattern, sleptHours, vigorGain)
            -- иконка: берём файл бодрости, если задан; иначе общий
            local iconPath = Utils.getFilename((self.hud and (self.hud.vigorIconFile or self.hud.iconFile)) or "gui/vigor_icon.dds", self.modDirectory)
            if iconPath == "" then iconPath = nil end
            hud.topNotification:setNotification(titleTxt, infoTxt, "", iconPath, nil)
        end
    end

    -- выходим из режима сна
    self.sleepGame.isSleeping   = false
    self.sleepGame.startDayTime = nil
end

-- =========================
-- LIFECYCLE
-- =========================
function HungerSystem:loadMap(_mapFileName)
    if g_i18n and g_i18n.loadModI18n then
        pcall(function() g_i18n:loadModI18n(self.modDirectory) end)
    end

    if self.loadEffectsFromXML then
        pcall(function() self:loadEffectsFromXML() end)
    end

    -- загрузка состояния (сытость/бодрость + метки часов)
    do
        local player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                    or (g_currentMission and g_currentMission.player)
        pcall(function() self:loadFromSavegameIfNeeded(player) end)
    end

    -- команды
    addConsoleCommand("gsHungerSet",         "Set hunger (0-100)",                                  "consoleCommandSetHunger",          self)
    addConsoleCommand("gsSleepForce",        "Force sleep blackout",                                 "consoleCommandSleepForce",         self)
    addConsoleCommand("gsSleepTune",         "Tune sleep: <periodMs> <durationMs> <maxChance0..1>", "consoleCommandSleepTune",          self)
    -- выбор сложности (1..5)
    if self.readSettings then
        pcall(function() self:readSettings() end)
    end
    addConsoleCommand("gsHungerDifficulty",  "Get/Set hunger difficulty (1..5)",                    "consoleCommandHungerDifficulty",   self)
    if self._installMenuControl then
        pcall(function() self:_installMenuControl() end)
    end

    -- HUD
    if HungerHUD and HungerHUD.init then
        pcall(function() HungerHUD:init(self, self.hud) end)
    else
        print("[HungerSystem] Warning: HungerHUD not found; HUD column will not be drawn")
    end

    -- события сохранения
    if self._subscribeSavegameEvents then
        pcall(function() self:_subscribeSavegameEvents() end)
    end

    -- создание игрока / сон мира
    if g_messageCenter and MessageType and MessageType.PLAYER_CREATED then
        g_messageCenter:subscribe(MessageType.PLAYER_CREATED, self.onPlayerCreated, self)
    end
    self.sleepGame = self.sleepGame or { isSleeping = false, startDayTime = nil }
    if g_messageCenter and MessageType and MessageType.SLEEPING then
        g_messageCenter:subscribe(MessageType.SLEEPING, self.onSleepingState, self)
    end

    -- hook на save (совместимость)
    if g_messageCenter ~= nil and not self._saveHooked then
        g_messageCenter:subscribe(SaveEvent, self.onGameSave, self)
        self._saveHooked = true
    end

    print("[HungerSystem] HungerSystem loaded (core with sleep+vigor; difficulty-ready)")
end


function HungerSystem:onPlayerCreated(player)
    if not self.saveLoaded then
        pcall(function() self:loadFromSavegameIfNeeded(player) end)
    end

    local h = tonumber(self.hunger or 100) or 100
    local v = tonumber(self.vigor  or 100) or 100
    player.hunger = math.max(0, math.min(100, h))
    player.vigor  = math.max(0, math.min(100, v))

    local nowGH = self:_getCurrentGlobalHour()

    local lH = tonumber(self.lastGlobalHungerUpdate or 0) or 0
    if lH <= 0 or lH > nowGH then lH = nowGH end

    local lV = tonumber(self.lastGlobalVigorUpdate  or 0) or 0
    if lV <= 0 or lV > nowGH then lV = nowGH end

    player.lastGlobalHungerUpdate = lH
    player.lastGlobalVigorUpdate  = lV
    self.lastGlobalHungerUpdate   = lH
    self.lastGlobalVigorUpdate    = lV

    if self.updateHungerSpeeds then
        pcall(function() self:updateHungerSpeeds(player) end)
    end

    self._reapplySpeedTimer = 0.4

    self._pendingPlayerSync = nil
    self.initialised = true

    self.saveDirty = true
    self.saveDebounce = 0.2

    print(("[HungerSystem] Applied loaded stats to player: hunger=%.1f, vigor=%.1f (lastH=%d lastV=%d, nowGH=%d)")
        :format(player.hunger, player.vigor, lH, lV, nowGH))
end


function HungerSystem:_getCurrentGlobalHour()
    local env = g_currentMission and g_currentMission.environment
    if not env then return 0 end
    local curDay  = env.currentDay or 1
    local curHour = env.currentHour or 0
    local curMin  = env.currentMinute or 0
    local totalMinutes = (curDay * 24 * 60) + (math.floor(curHour) * 60) + math.floor(curMin)
    return math.floor(totalMinutes / 60)
end

function HungerSystem:consoleCommandSetHunger(value)
    local player = g_playerSystem and g_playerSystem:getLocalPlayer() or (g_currentMission and g_currentMission.player)
    if not player then logWarn("local player not found"); return end
    local v = tonumber(value)
    if not v then logWarn("Usage: gsHungerSet <0-100>"); return end
    local before = player.hunger or 100
    player.hunger = math.max(0, math.min(100, v))
    self.hunger = player.hunger
    self:updateHungerSpeeds(player)
    self.saveDirty = true
    self:saveToSavegame(false)

    local diff = math.max(0, before - self.hunger)
    if diff > 0 then self:notifyHungerStatusChange(diff) end
end

local function __clamp01_100(x)
    x = tonumber(x or 0) or 0
    if x < 0 then return 0 end
    if x > 100 then return 100 end
    return x
end

function HungerSystem:_applyDelta(kind, delta)
    delta = tonumber(delta or 0) or 0
    if delta == 0 then return 0 end

    local player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                or (g_currentMission and g_currentMission.player)

    if kind == "hunger" then
        local before = (player and player.hunger) or (self.hunger or 100)
        local after  = __clamp01_100(before + delta)
        if player then player.hunger = after end
        self.hunger = after
    elseif kind == "vigor" then
        local before = (player and player.vigor) or (self.vigor or 100)
        local after  = __clamp01_100(before + delta)
        if player then player.vigor = after end
        self.vigor = after
    else
        return 0
    end

    if player and self.updateHungerSpeeds then
        pcall(function() self:updateHungerSpeeds(player) end)
    end

    self.saveDirty = true
    return delta
end

function HungerSystem:addHungerDelta(delta)
    return self:_applyDelta("hunger", delta)
end

function HungerSystem:addVigorDelta(delta)
    return self:_applyDelta("vigor",  delta)
end

-- =========================
-- UPDATE
-- =========================
function HungerSystem:update(dt)
    if self.syncPlayerStatsIfNeeded then self:syncPlayerStatsIfNeeded() end
    self:updateSampleStopTimers(dt)
    if self.saveDebounce and self.saveDebounce > 0 then
        self.saveDebounce = math.max(0, self.saveDebounce - (dt / 1000))
    end

    local player = (g_playerSystem and g_playerSystem:getLocalPlayer()) or (g_currentMission and g_currentMission.player)

    if self._reapplySpeedTimer then
        self._reapplySpeedTimer = self._reapplySpeedTimer - (dt / 1000)
        if self._reapplySpeedTimer <= 0 then
            self._reapplySpeedTimer = nil

            local pl = player or (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                        or (g_currentMission and g_currentMission.player)

            if pl and self.updateHungerSpeeds then
                pcall(function() self:updateHungerSpeeds(pl) end)
            end
        end
    end

    if not player then return end

    if not self.saveLoaded then
        self:loadFromSavegameIfNeeded(player)
    end
    if not self.initialised then
        player.hunger = (player.hunger ~= nil) and player.hunger or 100
        player.vigor  = (player.vigor  ~= nil) and player.vigor  or 100
        player.lastGlobalHungerUpdate = player.lastGlobalHungerUpdate or 0
        player.lastGlobalVigorUpdate  = player.lastGlobalVigorUpdate  or 0
        self.hunger = player.hunger
        self.vigor  = player.vigor

        self:updateHungerSpeeds(player)

        self._reapplySpeedTimer = 0.4
        self.initialised = true
    end

    if self._pendingTimeSync then
        local envSync = g_currentMission and g_currentMission.environment
        if envSync then
            local nowGH = self:_getCurrentGlobalHour()
            nowGH = tonumber(nowGH or 0) or 0

            player.lastGlobalHungerUpdate = nowGH
            player.lastGlobalVigorUpdate  = nowGH
            self.lastGlobalHungerUpdate   = nowGH
            self.lastGlobalVigorUpdate    = nowGH

            -- print(("[HungerSystem] Time sync after load: globalHour=%d"):format(nowGH))
        end
        self._pendingTimeSync = nil
    end

    local env = g_currentMission and g_currentMission.environment
    if env and not (self.sleepGame and self.sleepGame.isSleeping) then
        local curDay  = env.currentDay or 1
        local curHour = env.currentHour or 0
        local curMin  = env.currentMinute or 0
        local totalMinutes = (curDay * 24 * 60) + (math.floor(curHour) * 60) + math.floor(curMin)
        local globalHourFromMinutes = math.floor(totalMinutes / 60)

        local hungerLossTotal, vigorLossTotal = 0, 0

        -- === выбор сложности ===
        local diff = (self.DIFFICULTY and self.DIFFICULTY[self.difficulty or 3]) or (self.DIFFICULTY and self.DIFFICULTY[3])
        local function sampleLoss(minMax)
            if not minMax then return 0 end
            local a, b = tonumber(minMax[1]) or 0, tonumber(minMax[2]) or 0
            if a >= b then return a end
            return math.random(a, b)
        end

        -- СЫТОСТЬ
        do
            local lastGlobal = tonumber(player.lastGlobalHungerUpdate or 0) or 0
            if lastGlobal > globalHourFromMinutes then lastGlobal = globalHourFromMinutes end
            if globalHourFromMinutes > lastGlobal then
                local hoursPassed = math.min(12, globalHourFromMinutes - lastGlobal)
                local hungerLoss = 0
                for _=1, hoursPassed do
                    hungerLoss = hungerLoss + sampleLoss(diff and diff.hunger or {3,5})
                end
                hungerLossTotal = hungerLoss

                local beforeH = player.hunger or 100
                player.hunger  = math.max(0, (beforeH or 100) - hungerLoss)
                player.lastGlobalHungerUpdate = globalHourFromMinutes
                self.hunger = player.hunger
                self.saveDirty = true
            end
        end

        -- БОДРОСТЬ
        do
            local lastGlobal = tonumber(player.lastGlobalVigorUpdate or 0) or 0
            if lastGlobal > globalHourFromMinutes then lastGlobal = globalHourFromMinutes end
            if globalHourFromMinutes > lastGlobal then
                local hoursPassed = math.min(12, globalHourFromMinutes - lastGlobal)
                local vigorLoss = 0
                for _=1, hoursPassed do
                    vigorLoss = vigorLoss + sampleLoss(diff and diff.vigor or {1,7})
                end
                vigorLossTotal = vigorLoss

                local beforeV = player.vigor or 100
                player.vigor  = math.max(0, (beforeV or 100) - vigorLoss)
                player.lastGlobalVigorUpdate = globalHourFromMinutes
                self.vigor = player.vigor
                self.saveDirty = true
            end
        end

        if (hungerLossTotal > 0) or (vigorLossTotal > 0) then
            self:updateHungerSpeeds(player)
            self:notifyHourlyChange(hungerLossTotal, vigorLossTotal)
        end
    end

    -- период звуков 0% сытости
    if (self.hunger or 100) <= 0 then
        if not self.effect.soundActive then
            self.effect.soundActive = true
            self.effect.nextSoundTimer = 0
        elseif (self.effect.nextSoundTimer or 0) > 0 then
            self.effect.nextSoundTimer = self.effect.nextSoundTimer - (dt / 1000)
        end
        if (self.effect.nextSoundTimer or 0) <= 0 then
            self:playRandomSoundNow()
        end
    else
        if self.effect.soundActive then
            self.effect.soundActive = false
            self.effect.nextSoundTimer = 0
            self:stopAllCachedSamples()
        end
    end

    -- периодика звуков 0% бодрости
    if (self.vigor or 100) <= 0 then
        if not self.vigorPeriodic.soundActive then
            self.vigorPeriodic.soundActive = true
            self.vigorPeriodic.nextSoundTimer = math.random(4, 10)
        elseif (self.vigorPeriodic.nextSoundTimer or 0) > 0 then
            self.vigorPeriodic.nextSoundTimer = self.vigorPeriodic.nextSoundTimer - (dt / 1000)
        end
        if (self.vigorPeriodic.nextSoundTimer or 0) <= 0 then
            self:playRandomVigorSoundNow()
        end
    else
        if self.vigorPeriodic.soundActive then
            self.vigorPeriodic.soundActive = false
            self.vigorPeriodic.nextSoundTimer = 0
        end
    end

    -- альфы виньеток
    if self.effect and self.effect.overlay then
        local h = self.hunger or 100
        if h <= 15 then
            if h <= 0 then self.effect.alpha = 1.0
            else self.effect.alpha = 0.35 + ((15 - h) / 15) * (1.0 - 0.35) end
        else
            self.effect.alpha = 0.0
        end
    end
    if self.vigorCont and self.vigorCont.overlay then
        local v = self.vigor or 100
        if v <= 15 then
            if v <= 0 then self.vigorCont.alpha = 1.0
            else self.vigorCont.alpha = 0.35 + ((15 - v) / 15) * (1.0 - 0.35) end
        else
            self.vigorCont.alpha = 0.0
        end
    end
    if self.effect and self.effect.vehicleOverlay then
        if (g_time or 0) >= (self.effect.vehicleActiveUntil or 0) then
            if self.effect.vehicleAlpha and self.effect.vehicleAlpha > 0 then
                self.effect.vehicleAlpha = math.max(0, (self.effect.vehicleAlpha or 0) - 0.03)
            end
        end
    end

    -- эпизоды/состояния
    local inVehicle = (g_currentMission and g_currentMission.controlledVehicle ~= nil)
    if inVehicle then
        if self.sleep and self.sleep.isActive     then self:__endSleep()   end
        if self.fatigue and self.fatigue.isActive then self:__endFatigue() end
    else
        self.fatigue.checkTimerMs = (self.fatigue.checkTimerMs or 0) + (dt or 0)
        if (not self.fatigue.isActive) and self.fatigue.checkTimerMs >= (self.fatigue.checkPeriodMs or 8000) then
            self.fatigue.checkTimerMs = 0
            local v = self.vigor or 100
            local function fatigueChanceLocal(selfRef, vPct)
                local t = math.max(0, (selfRef.fatigue.thresholdPct or 20) - vPct)
                if t <= 0 then return 0 end
                local k = math.min(1, (t / (selfRef.fatigue.thresholdPct or 20)))
                return (selfRef.fatigue.maxChance or 0.35) * k
            end
            local p = fatigueChanceLocal(self, v)
            if p > 0 and math.random() < p then
                self:__startFatigue(g_time or 0)
            end
        end
        if not self.fatigue.isActive then
            self:__updateSleepCheck(dt, g_time or 0)
        end
    end

    if self.fatigue.isActive then
        self.fatigue.alpha = math.min(1, (self.fatigue.alpha or 0) + (dt or 0)/250)
        if not inVehicle and PlayerStateWalk ~= nil then
            PlayerStateWalk.MAXIMUM_WALK_SPEED = 0.01
            PlayerStateWalk.MAXIMUM_RUN_SPEED  = 0.01
        end
        if (g_time or 0) >= (self.fatigue.activeUntilMs or 0) then
            self:__endFatigue()
        end
    elseif (self.fatigue.alpha or 0) > 0 then
        self.fatigue.alpha = math.max(0, (self.fatigue.alpha or 0) - (dt or 0)/300)
    end

    if self.sleep.isActive then
        self.sleep.alpha = math.min(1, self.sleep.alpha + (dt or 0) / 250)
        if not inVehicle and PlayerStateWalk ~= nil then
            PlayerStateWalk.MAXIMUM_WALK_SPEED = 0.01
            PlayerStateWalk.MAXIMUM_RUN_SPEED  = 0.01
        end
        if (g_time or 0) >= (self.sleep.activeUntilMs or 0) then
            self:__endSleep()
        end
    else
        if self.sleep.alpha and self.sleep.alpha > 0 then
            self.sleep.alpha = math.max(0, self.sleep.alpha - (dt or 0) / 300)
        end
    end

    -- ВАЖНО: ItemEffects НЕ обновляем, пока мир спит
    if not (self.sleepGame and self.sleepGame.isSleeping) then
        ItemEffects:update(self, dt)
    end
end

function HungerSystem:draw()
    local function renderAnyOverlay(ref, alpha)
        if not ref then return end
        local a = math.max(0, math.min(1, alpha or 1))

        local t = type(ref)
        if t == "number" then
            setOverlayColor(ref, 1, 1, 1, a)
            renderOverlay(ref, 0, 0, 1, 1)
            return
        elseif t == "table" then
            local id = ref.id
            if type(id) == "number" then
                setOverlayColor(id, 1, 1, 1, a)
                renderOverlay(id, 0, 0, 1, 1)
                return
            end
            if ref.setColor and ref.render then
                pcall(function() ref:setColor(1,1,1,a) end)
                pcall(function() ref:render() end)
                return
            end
        elseif t == "userdata" then
            if ref.setColor and ref.render then
                pcall(function() ref:setColor(1,1,1,a) end)
                pcall(function() ref:render() end)
                return
            end
        end
    end

    -- эффекты голода (виньетка)
    if self.effect and self.effect.overlay and (self.effect.alpha or 0) > 0 then
        renderAnyOverlay(self.effect.overlay, self.effect.alpha or 0)
    end

    -- бодрость (конт. виньетка)
    if self.vigorCont and self.vigorCont.overlay and (self.vigorCont.alpha or 0) > 0 then
        renderAnyOverlay(self.vigorCont.overlay, self.vigorCont.alpha or 0)
    end

    -- техничная виньетка (затухает сама в update)
    if self.effect and self.effect.vehicleOverlay and (self.effect.vehicleAlpha or 0) > 0 then
        renderAnyOverlay(self.effect.vehicleOverlay, self.effect.vehicleAlpha or 0)
    end

    -- эпизод «усталость»
    if self.fatigue and self.fatigue.overlay and (self.fatigue.alpha or 0) > 0 then
        renderAnyOverlay(self.fatigue.overlay, self.fatigue.alpha or 0)
    end

    -- эпизод «сон от голода»
    if self.sleep and self.sleep.overlay and (self.sleep.alpha or 0) > 0 then
        renderAnyOverlay(self.sleep.overlay, self.sleep.alpha or 0)
    end
	
	if ItemEffects and ItemEffects.draw then ItemEffects:draw(self) end
end

function HungerSystem:delete()
    pcall(function() self:saveToSavegame(true) end)

    if self._saveHooked and g_messageCenter ~= nil then
        pcall(function() g_messageCenter:unsubscribe(SaveEvent, self) end)
        self._saveHooked = false
    end

    if g_messageCenter and MessageType then
        pcall(function() if MessageType.SAVEGAME_SAVE_START    then g_messageCenter:unsubscribe(MessageType.SAVEGAME_SAVE_START,    self) end end)
        pcall(function() if MessageType.SAVEGAME_SAVE_FINISHED then g_messageCenter:unsubscribe(MessageType.SAVEGAME_SAVE_FINISHED, self) end end)
        pcall(function() if MessageType.SLEEPING               then g_messageCenter:unsubscribe(MessageType.SLEEPING,               self) end end)
    end

    if self.effect and self.effect.sampleCache then
        for _, data in pairs(self.effect.sampleCache) do
            if data and data.sampleId then
                pcall(function() stopSample(data.sampleId, 0.05, 0) end)
                pcall(function() delete(data.sampleId) end)
            end
        end
        self.effect.sampleCache = {}
        self.effect.sampleStopTimers = {}
    end

    if HungerHUD then
        if HungerHUD.INSTANCE and HungerHUD.INSTANCE.delete then
            pcall(function() HungerHUD.INSTANCE:delete() end)
            HungerHUD.INSTANCE = nil
        elseif HungerHUD.delete then
            pcall(function() HungerHUD:delete() end)
        end
    end

    local function _delOverlay(ov)
        if not ov then return end
        if type(ov) == "table" and ov.id then
            pcall(function() delete(ov.id) end)
            return
        end
        if type(ov) == "table" and ov.delete then
            pcall(function() ov:delete() end)
            return
        end
        if type(ov) == "number" then
            pcall(function() delete(ov) end)
            return
        end
    end

    if self.effect then
        _delOverlay(self.effect.overlay);        self.effect.overlay        = nil
        _delOverlay(self.effect.vehicleOverlay); self.effect.vehicleOverlay = nil
    end
    if self.vigorCont then
        _delOverlay(self.vigorCont.overlay);     self.vigorCont.overlay     = nil
    end
    if self.sleep then
        _delOverlay(self.sleep.overlay);         self.sleep.overlay         = nil
    end
    if self.fatigue then
        _delOverlay(self.fatigue.overlay);       self.fatigue.overlay       = nil
    end

    if ItemEffects and ItemEffects.clearAll then
        pcall(function() ItemEffects:clearAll() end)
    end
end

function HungerSystem:onGameSave(_, _)
    self:saveToSavegame(true)
end

function HungerSystem:mouseEvent() end
function HungerSystem:keyEvent() end
function HungerSystem:drawHud() end

addModEventListener(HungerSystem)
