-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

HungerHUD = {
    core = nil, cfg = nil, patched=false, origDraw=nil,
    bgLeft=nil, bgScale=nil, bgRight=nil,
    icHunger=nil, icVigor=nil,
    hungerPath=nil, vigorPath=nil,
}

local UI = {
    scaleHFactor = 1.0,
    padX = 0.001, padY = 0.006,
    labelAlpha = 0.85,
    valueBold = true,
    minPanelW = 0.0, maxPanelW = 0.07,
    colGapMul = 0.35,
    iconSizeMul = 1.18,
}

local function getCurrentHunger(self)
    local pl = (g_playerSystem and g_playerSystem:getLocalPlayer()) or (g_currentMission and g_currentMission.player)
    if pl and pl.hunger ~= nil then return pl.hunger end
    local hs = self.core
    if hs and hs.hunger ~= nil then return hs.hunger end
    return 100
end

local function getCurrentVigor(self)
    local pl = (g_playerSystem and g_playerSystem:getLocalPlayer()) or (g_currentMission and g_currentMission.player)
    if pl and pl.vigor ~= nil then return pl.vigor end
    local hs = self.core
    if hs and hs.vigor ~= nil then return hs.vigor end
    return 100
end

local function _ensureOverlay(texPath)
    if texPath and texPath ~= "" and fileExists(texPath) then
        local ov = Overlay.new(texPath, 0, 0, 0, 0)
        ov:setColor(1,1,1,1)
        return ov
    end
    return nil
end

function HungerHUD:init(coreRef, config)
    self.core = coreRef; self.cfg = config or {}

    local modDir = (self.core and self.core.modDirectory) or (g_currentModDirectory or "")
    self.hungerPath = Utils.getFilename("gui/icon/hunger.dds", modDir)
    self.vigorPath  = Utils.getFilename("gui/icon/vigor.dds",  modDir)

    -- Background
    local c = HUD.COLOR.BACKGROUND
    self.bgLeft  = g_overlayManager:createOverlay("gui.gameInfo_left",   0,0,0,0); self.bgLeft:setColor(c[1],c[2],c[3],c[4])
    self.bgScale = g_overlayManager:createOverlay("gui.gameInfo_middle", 0,0,0,0); self.bgScale:setColor(c[1],c[2],c[3],c[4])
    self.bgRight = g_overlayManager:createOverlay("gui.gameInfo_right",  0,0,0,0); self.bgRight:setColor(c[1],c[2],c[3],c[4])

    -- Icon
    if self.hungerPath and fileExists(self.hungerPath) then
        self.icHunger = _ensureOverlay(self.hungerPath)
    end
    if self.vigorPath and fileExists(self.vigorPath) then
        self.icVigor = _ensureOverlay(self.vigorPath)
    end

    self:patchGameInfoDisplay()
end

function HungerHUD:patchGameInfoDisplay()
    if not (g_currentMission and g_currentMission.hud and g_currentMission.hud.gameInfoDisplay) then return end
    local gid = g_currentMission.hud.gameInfoDisplay; if not gid or type(gid.draw)~="function" then return end
    if self.patched then return end
    self.patched, self.origDraw = true, gid.draw

    gid.draw = function(hudSelf)
        self.origDraw(hudSelf)
        -- Base geometry from vanilla
        local spacing   = hudSelf.spacing or 0.004
        local leftSafe  = 0.015
        local baseH     = (hudSelf.infoBgLeft and hudSelf.infoBgLeft.height) or 0.03
        local capWLeft  = (hudSelf.infoBgLeft  and hudSelf.infoBgLeft.width)  or 0.02
        local capWRight = (hudSelf.infoBgRight and hudSelf.infoBgRight.width) or capWLeft
        local bgTopY    = (hudSelf.infoBgRight and hudSelf.infoBgRight.y) or (hudSelf.infoBgLeft and hudSelf.infoBgLeft.y) or 0.90
        local bgY       = bgTopY

        local vanillaLeft = (hudSelf.infoBgLeft and hudSelf.infoBgLeft.x) or 0.5
        local attachGap   = (self.cfg.attachGap ~= nil) and self.cfg.attachGap or spacing
        local edgeGap     = (self.cfg.edgeGap   ~= nil) and self.cfg.edgeGap   or (spacing * 0.75)
        local rightEdge   = vanillaLeft - (attachGap + edgeGap)
        if rightEdge <= leftSafe then return end

        -- Values
        local hungerVal = math.floor(getCurrentHunger(self) or 0)
        local vigorVal  = math.floor(getCurrentVigor(self)  or 0)
        local valueHunger = string.format("%d%%", hungerVal)
        local valueVigor  = string.format("%d%%", vigorVal)

        -- Sizes
        local sizeMain = (hudSelf.calendarTextSize or 0.017)
        local sizeSub  = sizeMain * 0.92
        local padX, padY = UI.padX, UI.padY
        local iconSize   = sizeMain * UI.iconSizeMul
        local colGap     = spacing * UI.colGapMul

        -- Measure value column width (right-aligned)
        local wVal1 = getTextWidth(sizeMain, valueHunger)
        local wVal2 = getTextWidth(sizeSub,  valueVigor)
        local valColW = math.max(wVal1, wVal2) + 0.002

        -- Panel width = padding + icon + gap + value + padding
        local panelW  = padX + iconSize + colGap + valColW + padX
        panelW = math.max(UI.minPanelW, math.min(UI.maxPanelW, panelW))

        -- Panel height fits icon/value snugly
        local contentH = math.max(iconSize, sizeMain + (sizeSub * 0.9))
        local bgH = math.max(baseH * UI.scaleHFactor, contentH + padY * 2)

        -- Compute bg blocks (right anchored)
        local bgRightX = rightEdge - capWRight
        local bgScaleX = bgRightX - panelW
        local bgLeftX  = bgScaleX - capWLeft
        if bgLeftX < leftSafe then
            local d = leftSafe - bgLeftX
            bgLeftX  = leftSafe; bgScaleX = bgScaleX + d; bgRightX = bgRightX + d
        end

        -- Render backgrounds
        self.bgScale:setDimension(panelW, bgH);   self.bgScale:setPosition(bgScaleX, bgY);   self.bgScale:render()
        self.bgLeft:setDimension(capWLeft, bgH);  self.bgLeft:setPosition(bgLeftX, bgY);     self.bgLeft:render()
        self.bgRight:setDimension(capWRight, bgH);self.bgRight:setPosition(bgRightX, bgY);   self.bgRight:render()

        -- Content positions
        local contentX = bgScaleX + padX
        local topY     = bgY + bgH - padY - sizeMain
        local row2Y    = topY - (sizeMain * 1.5)

        -- Colors for values
        local function cH(v) if v>50 then return 0.55,1,0.55 elseif v>20 then return 1,0.93,0.55 else return 1,0.55,0.55 end end
        local function cV(v) if v>60 then return 0.55,1,0.55 elseif v>20 then return 1,0.93,0.55 else return 1,0.55,0.55 end end
        local hr,hg,hb = cH(hungerVal); local vr,vg,vb = cV(vigorVal)

        -- Draw icons (if available)
        if self.icHunger then
            self.icHunger:setPosition(contentX, topY - iconSize * 0.15)
            self.icHunger:setDimension(iconSize, iconSize)
            self.icHunger:render()
        end
        if self.icVigor then
            self.icVigor:setPosition(contentX, row2Y - iconSize * 0.15)
            self.icVigor:setDimension(iconSize, iconSize)
            self.icVigor:render()
        end

        -- Values column (right-aligned)
        local valX = contentX + iconSize + colGap
        setTextAlignment(RenderText.ALIGN_RIGHT)
        if UI.valueBold then setTextBold(true) end
        setTextColor(hr,hg,hb,1); renderText(valX + valColW, topY,  sizeMain, valueHunger)
        setTextColor(vr,vg,vb,1); renderText(valX + valColW, row2Y, sizeSub,  valueVigor)
        if UI.valueBold then setTextBold(false) end

        setTextAlignment(RenderText.ALIGN_LEFT); setTextColor(1,1,1,1)
    end
end

function HungerHUD:update(dt) end

function HungerHUD:draw()
    if ItemEffects and ItemEffects.getActiveEffectInfo then
        local act = ItemEffects:getActiveEffectInfo()
        if act and (act.timeLeftSec or 0) > 0 then
            local m = math.floor((act.timeLeftSec or 0)/60)
            local s = (act.timeLeftSec or 0)%60
            local stacks = (act.maxStacks and act.maxStacks>1) and string.format("  x%d/%d", act.stacks or 1, act.maxStacks) or ""
            local txt = string.format("%s%s: %d:%02d", tostring(act.title or "Эффект"), stacks, m, s)
            setTextBold(true); setTextAlignment(RenderText.ALIGN_LEFT); setTextColor(0.95,1,0.95,1)
            renderText(0.02, 0.012, 0.012, txt)
            setTextBold(false); setTextColor(1,1,1,1)
        end
    end
end

function HungerHUD:delete()
    if g_currentMission and g_currentMission.hud and g_currentMission.hud.gameInfoDisplay and self.patched and self.origDraw then
        g_currentMission.hud.gameInfoDisplay.draw = self.origDraw
    end
    self.patched=false; self.origDraw=nil

    local function _safeDel(ov)
        if not ov then return end
        pcall(function()
            if ov.delete then ov:delete()
            elseif type(ov)=="number" then delete(ov) end
        end)
    end

    _safeDel(self.bgLeft);  self.bgLeft=nil
    _safeDel(self.bgScale); self.bgScale=nil
    _safeDel(self.bgRight); self.bgRight=nil

    _safeDel(self.icHunger); self.icHunger=nil
    _safeDel(self.icVigor);  self.icVigor=nil
end

function HungerHUD:mouseEvent() end
function HungerHUD:keyEvent() end
addModEventListener(HungerHUD)
