-- Author: U_BMP
-- Group: vk.com/https://vk.com/biomodprod_utilit_fs
-- Date: 11.11.2025

HungerDrowsyDriver = {}
HungerDrowsyDriver.modName      = g_currentModName or "FS25_HungerSystem"
HungerDrowsyDriver.modDirectory = g_currentModDirectory or ""

-- ====== КОНФИГ: усталость ======
local CFG = {
    vigorStart         = 0.16,  -- ниже этого порога бодрости включаем усталость
    chanceMinPerSec    = 0.01,  -- шанс/сек при 15%
    chanceMaxPerSec    = 0.15,  -- шанс/сек при 0%
    cooldownMaxMs      = 13400, -- пауза между эффектами при слабой тяжести
    cooldownMinMs      = 8250,  -- пауза при сильной тяжести

    -- амплитуда/длительность рывка руля при усталости
    steerAmpMin        = 0.35,
    steerAmpMax        = 0.99,
    steerHoldMinMsMin  = 400,
    steerHoldMinMsMax  = 700,
    steerHoldMaxMsMin  = 800,
    steerHoldMaxMsMax  = 2600,

    minSpeedKmh        = 0.5,   -- автоподруливание только если движемся
    debug              = false
}

-- ====== КОНФИГ: пьянка (руль) ======
local DRUNK = {
    defaultSteerAmplitude = 0.20,   -- 0..1 (ось)
    defaultSteerPeriod    = 2.6,    -- сек (для подбора длительности)
    defaultDeadzoneHold   = 0.15,

    chancePerSec          = 0.22,   -- чаще, чем усталость
    cooldownMinMs         = 1850,
    cooldownMaxMs         = 3800
}

-- ====== КОНФИГ: пьянка (КАМЕРА ТЕХНИКИ) ======
local CAM_DEF = {
    ampX=0.006, ampY=0.004, ampZ=0.0, rollAmp=0.006,
    period=2.8, phaseX=0.0, phaseY=1.2, phaseZ=0.0, phaseRoll=0.7,
    noiseAmp=0.0, noisePeriod=2.4,
    fpScale=1.0, tpScale=0.6
}
local DRUNK_SWAY_MAX = {
    yaw   = 0.020,   -- ~1.15°
    pitch = 0.016,   -- ~0.92°
    roll  = 0.012    -- ~0.69°
}

local function nowMs() return g_time or 0 end
local function clamp01(v) if v < 0 then return 0 elseif v > 1 then return 1 else return v end end
local function lerp(a,b,t) return a + (b-a) * t end
local function logI(...) if CFG.debug then print(string.format("[HungerDrowsyDriver] "..(select(1, ...) or ""), select(2, ...))) end end

local function getSpeedKmh(self)
    local s = 0
    if self.getLastSpeed then
        local ok, v = pcall(self.getLastSpeed, self)
        if ok and v then s = v * 3.6 end
    elseif self.lastSpeedRealKmh then
        s = self.lastSpeedRealKmh
    end
    return math.abs(s or 0)
end

local function severityFromVigor(v01)
    if v01 >= CFG.vigorStart then return 0 end
    return (CFG.vigorStart - v01) / CFG.vigorStart
end

local function getVigor01()
    local player = (g_playerSystem and g_playerSystem.getLocalPlayer and g_playerSystem:getLocalPlayer())
                    or (g_currentMission and g_currentMission.player)
    if player and player.vigor ~= nil then
        return clamp01((tonumber(player.vigor) or 100) / 100.0)
    end
    if HungerSystem ~= nil then
        if HungerSystem.vigor ~= nil then
            return clamp01((tonumber(HungerSystem.vigor) or 100) / 100.0)
        end
        if type(HungerSystem.getVigor01) == "function" then
            local ok, v = pcall(HungerSystem.getVigor01, HungerSystem)
            if ok and v ~= nil then return clamp01(tonumber(v) or 1) end
        end
    end
    return 1.0
end

local function isDrunkActive()
    return (g_DrunkVehicleEffects ~= nil and g_DrunkVehicleEffects.enabled == true)
end

local function getDrunkSteerParams()
    local amp   = DRUNK.defaultSteerAmplitude
    local per   = DRUNK.defaultSteerPeriod
    local dhold = DRUNK.defaultDeadzoneHold
    if g_DrunkVehicleEffects and g_DrunkVehicleEffects.steer then
        amp   = g_DrunkVehicleEffects.steer.amplitude    or amp
        per   = g_DrunkVehicleEffects.steer.period       or per
        dhold = g_DrunkVehicleEffects.steer.deadzoneHold or dhold
    end
    local holdMin = math.max(300, math.floor(per * 300))   
    local holdMax = math.max(600, math.floor(per * 1000))  
    return amp, per, dhold, holdMin, holdMax
end

local function getDrunkCamParams(isInside)
    local cam = (g_DrunkVehicleEffects and g_DrunkVehicleEffects.cam) or CAM_DEF
    local S = {
        ampX      = math.min(cam.ampX    or CAM_DEF.ampX,    DRUNK_SWAY_MAX.pitch),
        ampY      = math.min(cam.ampY    or CAM_DEF.ampY,    DRUNK_SWAY_MAX.yaw),
        rollAmp   = math.min(cam.rollAmp or CAM_DEF.rollAmp, DRUNK_SWAY_MAX.roll),
        period    = math.max(0.2, cam.period or CAM_DEF.period),
        phaseX    = cam.phaseX    or CAM_DEF.phaseX,
        phaseY    = cam.phaseY    or CAM_DEF.phaseY,
        phaseZ    = cam.phaseZ    or CAM_DEF.phaseZ,
        phaseRoll = cam.phaseRoll or CAM_DEF.phaseRoll,
        noiseAmp  = cam.noiseAmp  or CAM_DEF.noiseAmp,
        noisePeriod = math.max(0.2, cam.noisePeriod or CAM_DEF.noisePeriod),
        fpScale   = cam.fpScale   or CAM_DEF.fpScale,
        tpScale   = cam.tpScale   or CAM_DEF.tpScale
    }
    S.scale = isInside and S.fpScale or S.tpScale
    return S
end

-- ====== SPEC API ======
function HungerDrowsyDriver.prerequisitesPresent(specs)
    return SpecializationUtil.hasSpecialization(Enterable, specs)
end

function HungerDrowsyDriver.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad",          HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete",        HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate",        HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateEnd",     HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onPostUpdate",    HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onEnterVehicle",  HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onLeaveVehicle",  HungerDrowsyDriver)
    SpecializationUtil.registerEventListener(vehicleType, "onCameraChanged", HungerDrowsyDriver)
end

function HungerDrowsyDriver:onLoad(_)
    self.spec_hungerDrowsyDriver = {
        isDriverInside = false,
        lastEventTime  = 0,
        activeUntil    = 0,
        jerkAxisSide   = 0,
        _canSteer      = false,
        _sevDrowsy     = 0,
        _drunkActive   = false,

        camPhase       = 0.0,
        camNoisePhase  = 0.0
    }
end

function HungerDrowsyDriver:onDelete()
    self.spec_hungerDrowsyDriver = nil
end

function HungerDrowsyDriver:onEnterVehicle(_, _)
    local s = self.spec_hungerDrowsyDriver
    if s then s.isDriverInside = true end
end

function HungerDrowsyDriver:onLeaveVehicle(_, _)
    local s = self.spec_hungerDrowsyDriver
    if not s then return end
    s.isDriverInside = false
    s.activeUntil    = 0
    s.jerkAxisSide   = 0
    s.camPhase, s.camNoisePhase = 0.0, 0.0
end

function HungerDrowsyDriver:onCameraChanged()
    local s = self.spec_hungerDrowsyDriver
    if s then
        s.camPhase, s.camNoisePhase = 0.0, 0.0
    end
end

-- ====== ТРИГГЕРЫ ПОВЕДЕНИЯ (руль) ======
local function triggerMicroSteer(self, axisAmplitude, holdMs, doVignette, sevDbgTag)
    local side = (math.random() < 0.5 and -1 or 1) * math.max(0.01, math.min(1.0, axisAmplitude))
    local spec = self.spec_hungerDrowsyDriver
    spec.jerkAxisSide = side
    spec.activeUntil  = nowMs() + holdMs

    if self.setSteeringInput then pcall(self.setSteeringInput, self, side) end
    HungerDrowsyActionEvent.send(self, 0.0, side, holdMs)

    if doVignette and HungerSystem and HungerSystem.triggerVehicleEffect then
        pcall(HungerSystem.triggerVehicleEffect, HungerSystem, math.floor(holdMs * 0.8), 0.0)
    end

    logI("MicroSteer[%s] axis=%.2f hold=%dms", tostring(sevDbgTag or "?"), side, holdMs)
end

local function tryTriggerDrowsy(self, dt, sev01)
    local spec = self.spec_hungerDrowsyDriver
    local t    = nowMs()
    local cooldown = math.floor(lerp(CFG.cooldownMaxMs, CFG.cooldownMinMs, sev01))
    if (t - spec.lastEventTime) < cooldown then return end

    local chancePerSec = lerp(CFG.chanceMinPerSec, CFG.chanceMaxPerSec, sev01)
    local frameChance  = chancePerSec * (dt / 1000.0)
    if frameChance <= 0 or math.random() >= frameChance then return end

    spec.lastEventTime = t
    local amp     = lerp(CFG.steerAmpMin, CFG.steerAmpMax, sev01)
    local holdMin = math.floor(lerp(CFG.steerHoldMinMsMin, CFG.steerHoldMinMsMax, sev01))
    local holdMax = math.floor(lerp(CFG.steerHoldMaxMsMin, CFG.steerHoldMaxMsMax, sev01))
    local hold    = holdMin + math.floor(math.random() * math.max(0, (holdMax - holdMin + 1)))

    triggerMicroSteer(self, amp, hold, true, "drowsy")
end

local function tryTriggerDrunk(self, dt)
    local spec = self.spec_hungerDrowsyDriver
    local t    = nowMs()

    local cooldown = math.random(DRUNK.cooldownMinMs, DRUNK.cooldownMaxMs)
    if (t - spec.lastEventTime) < cooldown then return end

    local frameChance = DRUNK.chancePerSec * (dt / 1000.0)
    if frameChance <= 0 or math.random() >= frameChance then return end

    spec.lastEventTime = t

    local amp, period, _, holdMin, holdMax = getDrunkSteerParams()
    local hold  = holdMin + math.floor(math.random() * math.max(0, (holdMax - holdMin + 1)))

    triggerMicroSteer(self, amp, hold, false, "drunk")
end

-- ====== ПОКАЧИВАНИЕ КАМЕРЫ ПРИ ПЬЯНКЕ ======
local function getActiveCameraNode(self, enterable)
    if g_cameraManager and g_cameraManager.getActiveCamera then
        local ok, cam = pcall(g_cameraManager.getActiveCamera, g_cameraManager)
        if ok and cam then
            if type(cam) == "number" then
                return cam
            elseif type(cam) == "table" then
                if cam.getNode then
                    local ok2, n = pcall(cam.getNode, cam)
                    if ok2 and n then return n end
                end
                if cam.cameraNode then return cam.cameraNode end
                if cam.node then return cam.node end
            end
        end
    end
    if enterable and enterable.cameras then
        local idx = enterable.cameraIndex or enterable.activeCameraIndex or 1
        local vc  = enterable.cameras[idx]
        if vc then return vc.cameraNode or vc.node end
    end
    return nil
end

local function applyDrunkCameraSway(self, dt)
    if not self.isClient then return end
    local enterable = self.spec_enterable
    if not (enterable and enterable.isEntered) then return end
    if g_cameraFlightManager and g_cameraFlightManager.cameraFlightIsActive then return end
    if not isDrunkActive() then return end

    local camNode = getActiveCameraNode(self, enterable)
    if not camNode or not entityExists(camNode) then return end

    local isInside = false
    if enterable.cameras then
        local idx = enterable.cameraIndex or enterable.activeCameraIndex or 1
        local vc  = enterable.cameras[idx]
        if vc and vc.isInside ~= nil then isInside = (vc.isInside == true) end
    end
    local P = getDrunkCamParams(isInside)

    local s = self.spec_hungerDrowsyDriver
    local dtSec = (dt or 0) * 0.001
    local w  = (2*math.pi) / (P.period > 0.2 and P.period or 2.8)
    local wn = (2*math.pi) / (P.noisePeriod > 0.2 and P.noisePeriod or 2.4)
    s.camPhase      = (s.camPhase or 0) + w  * dtSec
    s.camNoisePhase = (s.camNoisePhase or 0) + wn * dtSec

    local sx = math.sin(s.camPhase + (P.phaseX or 0))
    local sy = math.sin(s.camPhase + (P.phaseY or 0))
    local sr = math.sin(s.camPhase + (P.phaseRoll or 0))
    local n  = (P.noiseAmp or 0) ~= 0 and math.sin(s.camNoisePhase) * (P.noiseAmp or 0) or 0.0
    local k  = P.scale or 1.0

    local baseRx, baseRy, baseRz = getRotation(camNode)
    local dRx = (P.ampX or 0)    * sx * k + n * 0.2   -- pitch
    local dRy = (P.ampY or 0)    * sy * k + n * 0.2   -- yaw
    local dRz = (P.rollAmp or 0) * sr * k + n         -- roll
    setRotation(camNode, baseRx + dRx, baseRy + dRy, baseRz + dRz)
end

-- ====== UPDATE ======
function HungerDrowsyDriver:onUpdate(dt)
    local spec = self.spec_hungerDrowsyDriver; if not spec then return end
    if not spec.isDriverInside then return end
    if self.getIsAIActive and self:getIsAIActive() then return end
    if g_gui and g_gui.getIsGuiVisible and g_gui:getIsGuiVisible() then return end

    spec._drunkActive = isDrunkActive()

    local vigor01 = getVigor01()
    spec._sevDrowsy = severityFromVigor(vigor01)

    if spec._sevDrowsy <= 0 and not spec._drunkActive then
        spec.activeUntil, spec.jerkAxisSide = 0, 0
        return
    end

    spec._canSteer = (getSpeedKmh(self) >= CFG.minSpeedKmh)

    if spec._canSteer and nowMs() < (spec.activeUntil or 0) and self.setSteeringInput then
        pcall(self.setSteeringInput, self, spec.jerkAxisSide or 0)
    end

    if spec._sevDrowsy > 0 then
        tryTriggerDrowsy(self, dt, spec._sevDrowsy)
    end
    if spec._drunkActive then
        tryTriggerDrunk(self, dt)
    end
end

function HungerDrowsyDriver:onUpdateEnd(_)
    local spec = self.spec_hungerDrowsyDriver
    if not spec or not spec._canSteer then return end
    if nowMs() >= (spec.activeUntil or 0) then return end

    local drv = self.spec_drivable; if not drv then return end
    local current = drv.axisSide or 0
    local target  = spec.jerkAxisSide or 0

    local k = (spec._drunkActive and 0.78 or lerp(0.55, 0.75, spec._sevDrowsy or 0))
    local blended = current*(1-k) + target*k
    if blended < -1 then blended = -1 elseif blended > 1 then blended = 1 end
    drv.axisSide = blended

    if self.isServer then self:raiseActive() end
end

function HungerDrowsyDriver:onPostUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    applyDrunkCameraSway(self, dt)
end
