--[[
    AutoAttachTrailers.lua

    Changes the attacher joints behavior to automatically attach attachables with jointType "trailer" when the vehicle is close enough.

	@author: 		BayernGamers
	@date: 			08.06.2025
	@version:		1.0

	History:		v1.0 @08.06.2025 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------

	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage:
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/AttachImplementsManuallySettingsManager.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/AttachImplementEvent.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "AutoAttachTrailers.lua")

AutoAttachTrailers = {}
AutoAttachTrailers.MOD_DIRECTORY = g_currentModDirectory
AutoAttachTrailers.MOD_NAME = g_currentModName
AutoAttachTrailers.AUTO_ATTACH_DISTANCE = 0.1
AutoAttachTrailers.AUTO_ATTACH_RESET_DISTANCE = 0.75

function AutoAttachTrailers.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(AttachImplementsManually, specializations)
end

function AutoAttachTrailers.initSpecialization()
    local schemaSavegame = Vehicle.xmlSchemaSavegame

    schemaSavegame:register(XMLValueType.STRING, "vehicles.vehicle(?).FS25_attachImplementsManually.autoAttachTrailers.attachable(?)#uniqueId", "Unique ID of the vehicle to auto attach trailers to", nil)
    schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).FS25_attachImplementsManually.autoAttachTrailers.attachable(?)#attacherJointIndex", "Attacher joint index to auto attach trailers to", nil)
    schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).FS25_attachImplementsManually.autoAttachTrailers.attachable(?)#inputJointDescIndex", "Input attacher joint index of the trailer to auto attach", nil)
end

function AutoAttachTrailers.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", AutoAttachTrailers)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", AutoAttachTrailers)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", AutoAttachTrailers)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", AutoAttachTrailers)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", AutoAttachTrailers)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", AutoAttachTrailers)

    SpecializationUtil.registerEventListener(vehicleType, "onPreDetachImplement", AutoAttachTrailers)
end

function AutoAttachTrailers.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadDisAllowAutoAttachFromXMLFile", AutoAttachTrailers.loadDisAllowAutoAttachFromXMLFile)
end

function AutoAttachTrailers:onPreLoad(savegame)
    self.spec_autoAttachTrailers = {}
end

function AutoAttachTrailers:onLoad(savegame)
    local spec = self.spec_autoAttachTrailers
    
    spec.modSettingsManager = AttachImplementsManuallySettingsManager.getInstance()
    spec.disallowAutoAttachUntilReset = {}
    spec.allowsAutoAttach = false
    self:loadDisAllowAutoAttachFromXMLFile(savegame)
end

function AutoAttachTrailers:onPostLoad(savegame)
    local spec = self.spec_autoAttachTrailers

    for _, attacherJoint in pairs(self.spec_attacherJoints.attacherJoints) do
        if attacherJoint.jointType == AttacherJoints.JOINTTYPE_TRAILER or attacherJoint.jointType == AttacherJoints.JOINTTYPE_HOOKLIFT then
            spec.allowsAutoAttach = true
        end
    end
end

function AutoAttachTrailers:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self.spec_autoAttachTrailers

    if spec.allowsAutoAttach then
        local index = 0
        for uniqueId, attacherJointIndices in pairs(spec.disallowAutoAttachUntilReset) do
            local subKey = string.format("%s.attachable(%d)", key, index)

            xmlFile:setValue(subKey .. "#uniqueId", uniqueId)
            xmlFile:setValue(subKey .. "#attacherJointIndex", attacherJointIndices.jointDescIndex)
            xmlFile:setValue(subKey .. "#inputJointDescIndex", attacherJointIndices.inputJointDescIndex)

            index = index + 1
        end
    end
end

function AutoAttachTrailers:loadDisAllowAutoAttachFromXMLFile(savegame)
    local spec = self.spec_autoAttachTrailers
    local disallowAutoAttachUntilReset = {}

    if savegame ~= nil and savegame.xmlFile ~= nil then
        local xmlFile = savegame.xmlFile
        local key = savegame.key .. ".FS25_attachImplementsManually.autoAttachTrailers.attachable"

        xmlFile:iterate(key, function(index, subKey)
            local uniqueId = xmlFile:getValue(subKey .. "#uniqueId")
            local attacherJointIndex = xmlFile:getValue(subKey .. "#attacherJointIndex")
            local inputJointDescIndex = xmlFile:getValue(subKey .. "#inputJointDescIndex")

            if uniqueId ~= nil and attacherJointIndex ~= nil and inputJointDescIndex ~= nil then
                disallowAutoAttachUntilReset[uniqueId] = {
                    jointDescIndex = attacherJointIndex,
                    inputJointDescIndex = inputJointDescIndex
                }
            end
        end)
    end

    spec.disallowAutoAttachUntilReset = disallowAutoAttachUntilReset
end

function AutoAttachTrailers:onUpdateTick(dt)
    local spec = self.spec_autoAttachTrailers

    if spec.allowsAutoAttach then
        if self:getCanToggleAttach() then
            local info = self.spec_attacherJoints.attachableInfo

            if info ~= nil and info.attachable ~= nil then
                local uniqueId = info.attachable:getUniqueId()
                local attachable = info.attachable
                local attacherVehicle = info.attacherVehicle
                local attachableJointDescIndex = info.attachableJointDescIndex
                local attacherVehicleJointDescIndex = info.attacherVehicleJointDescIndex

                if attacherVehicle == self then
                    local attacherJoint = attacherVehicle:getAttacherJointByJointDescIndex(attacherVehicleJointDescIndex)
                    local attachableJoint = attachable:getInputAttacherJointByJointDescIndex(attachableJointDescIndex)

                    local isTrailerAttacher = attacherJoint.jointType == AttacherJoints.JOINTTYPE_TRAILER and attachableJoint.jointType == AttacherJoints.JOINTTYPE_TRAILER
                    local isHookliftAttacher = attacherJoint.jointType == AttacherJoints.JOINTTYPE_HOOKLIFT and attachableJoint.jointType == AttacherJoints.JOINTTYPE_HOOKLIFT

                    if isTrailerAttacher or isHookliftAttacher then
                        local jointNode = attacherJoint.jointTransform
                        local attachableJointNode = attachableJoint.node

                        local jointPosX, _, jointPosZ = getWorldTranslation(jointNode)
                        local attachableJointPosX, _, attachableJointPosZ = getWorldTranslation(attachableJointNode)
                        local distance = MathUtil.vector2Length(jointPosX - attachableJointPosX, jointPosZ - attachableJointPosZ)
                        local minDistance = AutoAttachTrailers.AUTO_ATTACH_DISTANCE

                        if not spec.disallowAutoAttachUntilReset[uniqueId] then
                            if distance <= minDistance then

                                if self.isServer then
                                    self:attachImplementFromInfo(info)
                                else
                                    g_client:getServerConnection():sendEvent(VehicleAttachRequestEvent.new(info))
                                    if not spec.modSettingsManager:getEnableAttachImplementsManually() then
                                        AttachImplementEvent.sendEvent(self, attachable, attacherVehicleJointDescIndex, attachableJointDescIndex, true, false)
                                    end
                                end
                            end
                        end
                    end
                end
            end
        end

        for uniqueId, indices in pairs(spec.disallowAutoAttachUntilReset) do
            local vehicle = g_currentMission.vehicleSystem:getVehicleByUniqueId(uniqueId)

            if vehicle == nil then
                spec.disallowAutoAttachUntilReset[uniqueId] = nil
            else
                local jointDescIndex = indices.jointDescIndex
                local inputJointDescIndex = indices.inputJointDescIndex

                local attacherJoint = self:getAttacherJointByJointDescIndex(jointDescIndex)
                local attachableJoint = vehicle:getInputAttacherJointByJointDescIndex(inputJointDescIndex)

                local minResetDistance = AutoAttachTrailers.AUTO_ATTACH_RESET_DISTANCE

                if attacherJoint ~= nil and attachableJoint ~= nil then
                    local jointNode = attacherJoint.jointTransform
                    local attachableJointNode = attachableJoint.node

                    local jointPosX, _, jointPosZ = getWorldTranslation(jointNode)
                    local attachableJointPosX, _, attachableJointPosZ = getWorldTranslation(attachableJointNode)
                    local distance = MathUtil.vector2Length(jointPosX - attachableJointPosX, jointPosZ - attachableJointPosZ)

                    if distance > minResetDistance then
                        spec.disallowAutoAttachUntilReset[uniqueId] = nil
                    end
                else
                    spec.disallowAutoAttachUntilReset[uniqueId] = nil
                end
            end
        end
    end
end

function AutoAttachTrailers:onWriteStream(streamId, connection)
    local spec = self.spec_autoAttachTrailers

    streamWriteBool(streamId, spec.allowsAutoAttach)

    if spec.allowsAutoAttach then
        --streamWriteInt32(streamId, #spec.disallowAutoAttachUntilReset)
        local count = 0
        for _, _ in pairs(spec.disallowAutoAttachUntilReset) do
            count = count + 1
        end
        streamWriteInt32(streamId, count)

        for uniqueId, indices in pairs(spec.disallowAutoAttachUntilReset) do
            streamWriteString(streamId, uniqueId)
            streamWriteInt8(streamId, indices.jointDescIndex)
            streamWriteInt8(streamId, indices.inputJointDescIndex)
        end
    end
end

function AutoAttachTrailers:onReadStream(streamId, connection)
    local spec = self.spec_autoAttachTrailers

    spec.allowsAutoAttach = streamReadBool(streamId)

    if spec.allowsAutoAttach then
        local count = streamReadInt32(streamId)

        for i = 1, count do
            local uniqueId = streamReadString(streamId)
            local jointDescIndex = streamReadInt8(streamId)
            local inputJointDescIndex = streamReadInt8(streamId)

            spec.disallowAutoAttachUntilReset[uniqueId] = {
                jointDescIndex = jointDescIndex,
                inputJointDescIndex = inputJointDescIndex
            }
        end
    end
end

function AutoAttachTrailers:onPreDetachImplement(implement)
    local spec = self.spec_autoAttachTrailers

    if spec.allowsAutoAttach then
        if implement ~= nil then
            local detachedVehicle = implement.object

            if detachedVehicle ~= nil then
                local uniqueId = detachedVehicle:getUniqueId()
                if spec.disallowAutoAttachUntilReset[uniqueId] == nil then
                    local implementIndex = self:getImplementIndexByObject(detachedVehicle)
                    local jointDescIndex = self:getAttacherJointIndexFromImplementIndex(implementIndex)
                    local inputJointDescIndex = detachedVehicle.spec_attachable.inputAttacherJointDescIndex

                    spec.disallowAutoAttachUntilReset[uniqueId] = {
                        jointDescIndex = jointDescIndex,
                        inputJointDescIndex = inputJointDescIndex
                    }
                end
            end
        end
    end
end