--[[
Copyright (C) GtX (Andy), 2020

Author: GtX | Andy
Date: 24.08.2020
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
]]

VehicleSpeedSyncManager = {}

local VehicleSpeedSyncManager_mt = Class(VehicleSpeedSyncManager)

local vehicleSpeedSync
local validationFail

local buildId = 1
local versionString = "0.0.0.0"

local modName = g_currentModName
local modDirectory = g_currentModDirectory
local modSettingsDirectory = g_currentModSettingsDirectory

function VehicleSpeedSyncManager.new()
    local self = setmetatable({}, VehicleSpeedSyncManager_mt)

    self.modName = modName
    self.modDirectory = modDirectory

    self.buildId = buildId
    self.versionString = versionString

    self.numInserted = 0
    self.elementsCreated = false

    if string.isNilOrWhitespace(modSettingsDirectory) then
        modSettingsDirectory = getUserProfileAppPath() .. "modSettings/" .. modName
    end

    self.settingsXMLFilename = modSettingsDirectory .. "settings.xml"

    return self
end

function VehicleSpeedSyncManager:delete()
end

function VehicleSpeedSyncManager:saveSettings()
    if g_dedicatedServerInfo == nil then
        if not fileExists(modSettingsDirectory) then
            createFolder(modSettingsDirectory)
        end

        local xmlFile = createXMLFile("VehicleSpeedSyncXML", self.settingsXMLFilename, "vehicleSpeedSync")

        if xmlFile ~= 0 then
            setXMLString(xmlFile, "vehicleSpeedSync#version", versionString)
            setXMLFloat(xmlFile, "vehicleSpeedSync#buildId", buildId)

            setXMLBool(xmlFile, "vehicleSpeedSync.activeSpeedSync#enabled", VehicleSpeedSync.SPEED_SYNC_ACTIVE)
            setXMLBool(xmlFile, "vehicleSpeedSync.activeSpeedSync#showInputBinding", VehicleSpeedSync.SPEED_SYNC_VISIBLE)

            setXMLBool(xmlFile, "vehicleSpeedSync.cruiseControlSync#enabled", VehicleSpeedSync.CRUISE_SYNC_ACTIVE)
            setXMLBool(xmlFile, "vehicleSpeedSync.cruiseControlSync#showInputBinding", VehicleSpeedSync.CRUISE_SYNC_VISIBLE)

            setXMLBool(xmlFile, "vehicleSpeedSync.hudIcon#enabled", VehicleSpeedSync.HUD_ICON_VISIBLE)

            saveXMLFile(xmlFile)
            delete(xmlFile)
        end
    end
end

function VehicleSpeedSyncManager:loadSettings()
    if g_dedicatedServerInfo == nil then
        if fileExists(self.settingsXMLFilename) then
            local xmlFile = loadXMLFile("VehicleSpeedSyncXML", self.settingsXMLFilename)

            if xmlFile ~= 0 then
                VehicleSpeedSync.SPEED_SYNC_ACTIVE = Utils.getNoNil(getXMLBool(xmlFile, "vehicleSpeedSync.activeSpeedSync#enabled"), true)
                VehicleSpeedSync.SPEED_SYNC_VISIBLE = Utils.getNoNil(getXMLBool(xmlFile, "vehicleSpeedSync.activeSpeedSync#showInputBinding"), true)

                VehicleSpeedSync.CRUISE_SYNC_ACTIVE = Utils.getNoNil(getXMLBool(xmlFile, "vehicleSpeedSync.cruiseControlSync#enabled"), true)
                VehicleSpeedSync.CRUISE_SYNC_VISIBLE = Utils.getNoNil(getXMLBool(xmlFile, "vehicleSpeedSync.cruiseControlSync#showInputBinding"), true)

                VehicleSpeedSync.HUD_ICON_VISIBLE = Utils.getNoNil(getXMLBool(xmlFile, "vehicleSpeedSync.hudIcon#enabled"), true)

                delete(xmlFile)
            end
        end
    end
end

function VehicleSpeedSyncManager:updateSetting(name, isEnabled, titleText)
    if VehicleSpeedSync == nil or VehicleSpeedSync[name] == nil then
        Logging.devInfo("[%s] Attempt to update a setting that does not exist '%s'!", modName, name)

        return
    end

    VehicleSpeedSync[name] = isEnabled

    if g_localPlayer ~= nil then
        local vehicle = g_localPlayer:getCurrentVehicle()

        if vehicle ~= nil and vehicle.requestActionEventUpdate ~= nil then
            vehicle:requestActionEventUpdate()
        end
    end

    self:saveSettings()

    print(string.format("Setting '%s': %s", titleText or name, isEnabled))
end

local function validateMod()
    local mod = g_modManager:getModByName(modName)

    if mod == nil or g_iconGenerator ~= nil then
        return true
    end

    versionString = mod.version or versionString

    if mod.modName == "FS25_VehicleSpeedSync" or mod.modName == "FS25_VehicleSpeedSync_update" then
        if mod.author ~= nil and #mod.author == 3 then
            return true
        end
    end

    validationFail = {
        startUpdateTime = 2000,

        update = function(self, dt)
            self.startUpdateTime = self.startUpdateTime - dt

            if self.startUpdateTime < 0 then
                local text = string.format(g_i18n:getText("vehicleSpeedSync_loadError", mod.modName), mod.modName, mod.author or "Unknown")

                if g_dedicatedServerInfo == nil then
                    if not g_gui:getIsGuiVisible() then
                        local title = string.format("%s - Version %s", mod.title, versionString)
                        local yesText = g_i18n:getText("button_modHubDownload")
                        local noText = g_i18n:getText("button_ok")

                        YesNoDialog.show(self.openModHubLink, nil, text, title, yesText, noText, DialogElement.TYPE_LOADING)
                    end
                else
                    print("\n" .. text .. "\n    - https://farming-simulator.com/mods.php?&title=fs2025&filter=org&org_id=129652&page=0" .. "\n")
                    self.openModHubLink(false)
                end
            end
        end,

        openModHubLink = function(yes)
            if yes then
                openWebFile("mods.php?title=fs2025&filter=org&org_id=129652&page=0", "")
            end

            removeModEventListener(validationFail)
            validationFail = nil
        end
    }

    addModEventListener(validationFail)

    return false
end

local function overwriteGameFunctions()
    SpeedMeterDisplay.storeScaledValues = Utils.appendedFunction(SpeedMeterDisplay.storeScaledValues, function(self)
        if self.speedSyncIcon == nil then
            local activeColor = HUD.COLOR.ACTIVE

            self.speedSyncIcon = g_overlayManager:createOverlay("gui.icon_speedSync", 0, 0, 0, 0, modName)
            self.speedSyncIcon:setColor(activeColor[1], activeColor[2], activeColor[3], activeColor[4])
        end

        if self.speedSyncIcon ~= nil then
            local width, height = self:scalePixelValuesToScreenVector(30, 30)

            self.speedSyncIcon:setDimension(width, height)
            self.speedSyncIconOffsetX, self.speedSyncIconOffsetY = self:scalePixelValuesToScreenVector(-139, 2)
        end
    end)

    SpeedMeterDisplay.delete = Utils.prependedFunction(SpeedMeterDisplay.delete, function(self)
        if self.speedSyncIcon ~= nil then
            self.speedSyncIcon:delete()
            self.speedSyncIcon = nil
        end
    end)

    SpeedMeterDisplay.draw = Utils.appendedFunction(SpeedMeterDisplay.draw, function(self)
        local vehicle = self.vehicle

        if vehicle == nil or not self.isVehicleDrawSafe then
            return
        end

        local speedSyncIcon = self.speedSyncIcon

        if speedSyncIcon ~= nil then
            local spec = vehicle.spec_vehicleSpeedSync

            if spec ~= nil and VehicleSpeedSync.getSyncedSpeedIconVisible() then
                local posX, posY = self:getPosition()
                local r, g, b, a = 1, 1, 1, 1

                if vehicle:vehicleSpeedSyncGetSyncedSpeedActive() then
                    if self.vehicle:vehicleSpeedSyncGetSyncedSpeedWaiting() then
                        if math.cos(g_time / 200) > 0 then
                            r, g, b, a = 1, 0.4287, 0.0006, 1
                        end
                    else
                        r, g, b, a = unpack(HUD.COLOR.ACTIVE)
                    end
                elseif spec.connectionLost ~= nil then
                    if spec.connectionLost > g_time then
                        r, g, b, a = 1, 0.1233, 0, 1
                    else
                        spec.connectionLost = nil
                    end
                end

                speedSyncIcon:setColor(r, g, b, a)
                speedSyncIcon:setPosition(posX + self.speedSyncIconOffsetX, posY + self.speedSyncIconOffsetY)
                speedSyncIcon:render()
            end
        end
    end)

    InGameMenuSettingsFrame.initializeSubCategoryPages = Utils.prependedFunction(InGameMenuSettingsFrame.initializeSubCategoryPages, function(self)
        if vehicleSpeedSync ~= nil then
            if not vehicleSpeedSync.elementsCreated then
                if self.generalSettingsLayout ~= nil and self.checkAutoHelp ~= nil then
                    local numElements = 0
                    local requiredElements = 6

                    local function onClickCallback(target, state, element, isLeftButtonEvent)
                        if vehicleSpeedSync ~= nil then
                            vehicleSpeedSync:updateSetting(element.name or "NO_NAME", state == BinaryOptionElement.STATE_RIGHT, element.titleText)
                        end
                    end

                    local function createBinaryOptionElement(id, titleL10n, toolTipL10n, name)
                        local containerElement = self.checkAutoHelp.parent:clone(self.generalSettingsLayout)
                        local checkElement = containerElement.elements[1]
                        local titleElement = containerElement.elements[2]

                        if checkElement ~= nil and titleElement ~= nil then
                            local toolTipElement = checkElement.elements[1]

                            if toolTipElement ~= nil then
                                toolTipElement:setText(g_i18n:getText(toolTipL10n, modName))
                            end

                            local titleText = g_i18n:getText(titleL10n, modName)

                            titleElement:setText(titleText)

                            checkElement.titleText = titleText
                            checkElement.id = `vehicleSpeedSync_{id}`
                            checkElement.name = name
                            checkElement.onClickCallback = onClickCallback

                            containerElement:reloadFocusHandling(true)

                            self[checkElement.id] = checkElement
                            numElements += 1

                            checkElement:setIsChecked(VehicleSpeedSync[name])
                            vehicleSpeedSync[id] = checkElement
                        end
                    end

                    for i, element in ipairs (self.generalSettingsLayout.elements) do
                        if element.name == "sectionHeader" and element.typeName == "Text" then
                            local headerElement = element:clone(self.generalSettingsLayout)

                            headerElement:setText(g_i18n:getText("ui_sectionHeader", modName))
                            headerElement:reloadFocusHandling(true)

                            self.vehicleSpeedSync_sectionHeader = headerElement
                            numElements += 1

                            break
                        end
                    end

                    createBinaryOptionElement("checkSpeedSync", "ui_speedSync", "toolTip_syncAvailable", "SPEED_SYNC_ACTIVE")
                    createBinaryOptionElement("checkSpeedSyncInputDisplay", "ui_speedSyncInputDisplay", "toolTip_inputDisplay", "SPEED_SYNC_VISIBLE")
                    createBinaryOptionElement("checkCruiseSync", "ui_cruiseSync", "toolTip_syncAvailable", "CRUISE_SYNC_ACTIVE")
                    createBinaryOptionElement("cruiseSyncInputDisplay", "ui_cruiseSyncInputDisplay", "toolTip_inputDisplay", "CRUISE_SYNC_VISIBLE")
                    createBinaryOptionElement("hudIcon", "ui_hudIcon", "toolTip_hudIcon", "HUD_ICON_VISIBLE")

                    --
                    --
                    -- Future: Maybe an option for range however this may cause issues if other vehicles are close to a long range.
                    -- Future: Add icon to vehicle HUD and allow the user to disable if they do not want it.
                    --
                    --

                    if numElements ~= requiredElements then
                        Logging.warning("[%s] Failed to correctly add settings elements to 'General Settings'. This could be due to base game changes / update or a mod conflict.", modName)
                    end

                    self.generalSettingsLayout:invalidateLayout()
                end

                vehicleSpeedSync.elementsCreated = true
            else
                if vehicleSpeedSync.checkSpeedSync ~= nil then
                    vehicleSpeedSync.checkSpeedSync:setIsChecked(VehicleSpeedSync.SPEED_SYNC_ACTIVE)
                end

                if vehicleSpeedSync.checkSpeedSyncInputDisplay ~= nil then
                    vehicleSpeedSync.checkSpeedSyncInputDisplay:setIsChecked(VehicleSpeedSync.SPEED_SYNC_VISIBLE)
                end

                if vehicleSpeedSync.checkCruiseSync ~= nil then
                    vehicleSpeedSync.checkCruiseSync:setIsChecked(VehicleSpeedSync.CRUISE_SYNC_ACTIVE)
                end

                if vehicleSpeedSync.checkCruiseSyncInputDisplay ~= nil then
                    vehicleSpeedSync.checkCruiseSyncInputDisplay:setIsChecked(VehicleSpeedSync.CRUISE_SYNC_VISIBLE)
                end

                if vehicleSpeedSync.hudIcon ~= nil then
                    vehicleSpeedSync.hudIcon:setIsChecked(VehicleSpeedSync.HUD_ICON_VISIBLE)
                end
            end
        end
    end)
end

local function typeManager_validateTypes(typeManager)
    if typeManager.typeName == "vehicle" then
        local specializationName = string.format("%s.vehicleSpeedSync", modName)
        local specializationObject = g_specializationManager:getSpecializationObjectByName(specializationName)

        if specializationObject ~= nil then
            for typeName, typeEntry in pairs (typeManager:getTypes()) do
                if specializationObject.prerequisitesPresent(typeEntry.specializations) then
                    typeManager:addSpecialization(typeName, specializationName)

                    vehicleSpeedSync.numInserted = vehicleSpeedSync.numInserted + 1
                end
            end
        end

        if vehicleSpeedSync.numInserted > 0 then
            overwriteGameFunctions()
        end
    end
end

local function unload(mission)
    if vehicleSpeedSync ~= nil then
        vehicleSpeedSync:delete()
    end

    if g_globalMods ~= nil then
        g_globalMods.vehicleSpeedSync = nil
    end

    vehicleSpeedSync = nil
end

local function init()
    if g_globalMods == nil then
        g_globalMods = {}
    end

    if g_globalMods.vehicleSpeedSync then
        Logging.error("Failed to load mod '%s', 'VehicleSpeedSync.lua' was already loaded by '%s'!", modName, g_globalMods.vehicleSpeedSync.modName or "N/A")

        return
    end

    if validateMod() then
        source(modDirectory .. "scripts/specializations/events/VehicleSpeedSyncEvent.lua")
        g_specializationManager:addSpecialization("vehicleSpeedSync", "VehicleSpeedSync", modDirectory .. "scripts/specializations/VehicleSpeedSync.lua", nil)

        TypeManager.validateTypes = Utils.prependedFunction(TypeManager.validateTypes, typeManager_validateTypes)
        FSBaseMission.delete = Utils.appendedFunction(FSBaseMission.delete, unload)

        vehicleSpeedSync = VehicleSpeedSyncManager.new()
        vehicleSpeedSync:loadSettings()

        g_overlayManager:addTextureConfigFile(modDirectory .. "menu/textures.xml", "gui", modName)

        g_globalMods.vehicleSpeedSync = vehicleSpeedSync
    else
        Logging.devWarning("[%s] Failed to validate mod!", modName)
    end
end

init()
